/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the Qt Linguist of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef MSGEDIT_H
#define MSGEDIT_H

#include "trwindow.h"
#include "phrase.h"
#include "messagemodel.h"

#include <QString>
#include <QPixmap>
#include <QBitmap>
#include <QToolTip>
#include <QtEvents>
#include <QShortcut>
#include <QFrame>
#include <QTextEdit>
#include <QScrollArea>
#include <QTextCharFormat>

QT_BEGIN_NAMESPACE

template <typename T> class QList;
class QSplitter;
class QDockWidget;
class QLabel;
class QTreeView;
class QBoxLayout;
class QMenu;

class EditorPage;
class MessageEditor;
class LanguagesManager;

class TransEditor : public QWidget
{
    Q_OBJECT
public:
    TransEditor(const QString &title, QWidget *parent = 0);

    void setLabel(const QString &text);
    void setEditingEnabled(bool enabled);
    QTextEdit *editor() const { return m_editor; }
    QLabel *label() const { return m_label; }

    QString translation() const;

    void gotFocusInEvent( QFocusEvent * e);

signals:
    void heightUpdated(int height);
    void gotFocusIn();

private:
    QTextEdit *m_editor;
    QLabel *m_label;
};

class ExpandingTextEdit : public QTextEdit
{
    Q_OBJECT

public:
    ExpandingTextEdit(QWidget *parent = 0);

public slots:
    void updateHeight(const QSizeF &documentSize);
};

class SourceTextEdit : public ExpandingTextEdit
{
    Q_OBJECT
public:
    SourceTextEdit(QWidget *parent = 0);

public slots:
    void copySelection();

protected:
    void contextMenuEvent(QContextMenuEvent *e);

private:
    QAction *actCopy;
    QAction *actSelect;
    QMenu *srcmenu;
};

class GuessShortcut : public QShortcut
{
    Q_OBJECT
public:
    GuessShortcut(int nkey, QWidget *parent, const char *member)
        : QShortcut(parent), nrkey(nkey)
    {
        setKey(Qt::CTRL + (Qt::Key_1 + nrkey));
        connect(this, SIGNAL(activated()), this, SLOT(keyActivated()));
        connect(this, SIGNAL(activated(int)), parent, member);
    }

private slots:
    void keyActivated()
    {
        emit activated(nrkey);
    }

signals:
    void activated(int nkey);

private:
    int nrkey;
};

class AuxTranslationWidget : public QWidget
{
    Q_OBJECT
public:
    AuxTranslationWidget(const QString &lang, QWidget *parent = 0);
    void clearText() { m_textEdit->clear(); }
    SourceTextEdit *getTextEdit() { return m_textEdit; }

signals:
    void textChanged();
    void selectionChanged();

private:
    QLabel *m_label;
    SourceTextEdit *m_textEdit;

private slots:
    void textEditChanged();
};

class EditorPage : public QFrame
{
    Q_OBJECT

public:
    EditorPage(QLocale::Language targetLanguage, MessageEditor *parent = 0, const char *name = 0);
    QTextEdit *activeTransText() const;
    int activeTranslationNumerus() const;
    QStringList translations() const;
    void setNumerusForms(const QStringList &numerusForms);
    int currentTranslationEditor();

public slots:
    void setTargetLanguage(QLocale::Language lang);

protected:
    void resizeEvent(QResizeEvent *);
    void updateCommentField();

private:
    void addPluralForm(const QString &label);
    void adjustTranslationFieldHeights();
    void handleChanges();
    void showNothing();
    void setAuxLanguages(const QStringList &languages);

    QBoxLayout *m_layout;
    QLabel *m_srcTextLabel;
    SourceTextEdit *srcText;
    QList<AuxTranslationWidget *> auxTranslations;
    QTextEdit *cmtText;
    QStringList m_numerusForms;
    QString     m_invariantForm;
    bool        m_pluralEditMode;
    QList<TransEditor*> m_transTexts;
    friend class MessageEditor;

private slots:
    void sourceSelectionChanged();
    void translationSelectionChanged();

signals:
    void selectionChanged();
    void currentTranslationEditorChanged();
};

class MessageEditor : public QScrollArea
{
    Q_OBJECT
public:
    MessageEditor(LanguagesManager *languages, QMainWindow *parent = 0);
    QTreeView *phraseView() const;
    inline QDockWidget *phraseDockWnd() const {return bottomDockWidget;}

    void showNothing();
    void showMessage(const QString &context, const QString &text, const QString &comment,
        const QString &fullContext, const QString &fileName, const int lineNumber,
        const QStringList &translation, TranslatorMessage::Type type,
        const QList<Phrase> &phrases);
    void setNumerusForms(const QStringList &numerusForms);
    bool eventFilter(QObject *, QEvent *);
    void setTranslation(const QString &translation, int numerus, bool emitt);
signals:
    void translationChanged(const QStringList &translations);
    void finished(bool finished);
    void prevUnfinished();
    void nextUnfinished();
    void updateActions(bool enable);

    void undoAvailable(bool avail);
    void redoAvailable(bool avail);
    void cutAvailable(bool avail);
    void copyAvailable(bool avail);
    void pasteAvailable(bool avail);

    void focusPhraseList();

public slots:
    void undo();
    void redo();
    void cut();
    void copy();
    void paste();
    void selectAll();
    void beginFromSource();
    void toggleGuessing();
    void setEditorFocus();
    void checkUndoRedo();
    void messageModelListChanged();

private slots:
    void emitTranslationChanged();
    void guessActivated(int key);
    void insertPhraseInTranslation(const QModelIndex &index);
    void insertPhraseInTranslationAndLeave(const QModelIndex &index);
    void updateButtons();
    void updateCanPaste();
    void clipboardChanged();

    void updateCutAndCopy();

public:
    static const char backTab[];
    static const char * const friendlyBackTab[];

private:
    void visualizeBackTabs(const QString &text, QTextEdit *te);
    void setEditingEnabled(bool enabled);

    QDockWidget *bottomDockWidget;
    EditorPage *editorPage;

    QTreeView *phraseTv;
    PhraseModel *phrMdl;
    QTextCharFormat defFormat;

    LanguagesManager *m_languages;
    QString sourceText;

    bool cutAvail;
    bool copyAvail;

    bool mayOverwriteTranslation;
    bool clipboardEmpty;
    bool doGuesses;
};

QT_END_NAMESPACE

#endif
