/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QtCore/QStringList>

#include "qacceltreeresourceloader_p.h"
#include "qcommonvalues_p.h"
#include "qxmlresultitems.h"
#include "qxmlresultitems_p.h"
#include "qxmlserializer.h"
#include "qxpathhelper_p.h"

#include "qxmlquery.h"
#include "qxmlquery_p.h"

Q_DECLARE_METATYPE(QIODevice *)

QT_BEGIN_NAMESPACE

// TODO Mention XML chars and QString.

// TODO document how we send to message handler.

/*!
  \class QXmlQuery
  \brief The QXmlQuery class is used to evaluate XQuery queries.
  \reentrant
  \since 4.4
  \ingroup xml-tools

  QXmlQuery is the central point for running XQuery queries. First setQuery() is called
  specifying the source code of the query, and one of the following members
  are called for the actual evaluation.

  \section1 Evaluating Queries

  serialize() evaluates the query and writes the result as XML to a QIODevice. For instance,
  the result is indented and then written to a file using QFile. With QSerializationSettings,
  options governing the serialization process can be specified.

  evaluateToReceiver() takes a pointer to a QAbstractXmlReceiver, whose members
  will be called to appropriately mirror the result of the query. This is
  conceptually like the SAX API.

  evaluateToResult() tranfers the evaluation to a QXmlResultItems instance, a Java-like iterator, which allows items
  to be accessed on an indivual basis using QXmlNodeModelIndex and QVariant.

  evaluateToStringList() is similar to evaluateToResult(), but converts the result directly to a QStringList.

  \section1 Variable Bindings

  Apart using the \a fn:doc() function to retrieve data into the query for processing,
  one can bind variables using bindVariable(). When having called bindVariable(), a variable binding will
  be available to the query by the specified name. It is not necessary to declare the variable as external
  inside the query.

  \section1 Thread Support

  Multiple queries can run in multiple threads, which is achieved by simply copying QXmlQuery and in
  the new instance modify it accordingly, such as changing the variable bindings or evaluate it
  with a different method. Behind the scenes, QXmlQuery will reuse resources such as opened files and
  compiled queries between threads, to the extent possible.

  \section1 Error Handling

  During the evaluation of a query, an error can occur. A dynamic type error, or failing to load
  a file are examples of such runtime errors.

  When an evaluation error occur, the following happens:

  \list
    \o Error messages are sent to the messageHandler()
    \o QXmlResultItems::hasError() will return \c true or evaluateToReceiver() will return \c false
    \o The query result, such as the bytes written out using serialize(), or
       the events sent to QAbstractXmlReceiver, and items returned from
       evaluateToResult(), is undefined
  \endlist

  \section1 Resource Management

  A query potentially creates nodes, opens documents, and in other ways allocate resources. These
  are automatically managed, and will be deallocated as soon they aren't needed anymore.
  If it is of interest to deallocate the resources a query has allocated, make sure the relevant
  QXmlQuery and QAbstractXmlReceiver & QAbstractXmlForwardIterator instances have been destructed.

  In order to use QXmlQuery, QCoreApplication must first be instantiated.
 */

/*!
  Constructs an invalid query that cannot be used. setQuery() must be called.
 */
QXmlQuery::QXmlQuery() : d(new QXmlQueryPrivate())
{
}

/*!
  Constructs a QXmlQuery instance that is a copy of \a other.
 */
QXmlQuery::QXmlQuery(const QXmlQuery &other) : d(new QXmlQueryPrivate(*other.d))
{
}

/*!
  Constructs a QXmlQuery instance that use \a np as name pool.
 */
QXmlQuery::QXmlQuery(const QXmlNamePool &np) : d(new QXmlQueryPrivate())
{
    d->namePool = np;
}

/*!
  Destructs this QXmlQuery instance.
 */
QXmlQuery::~QXmlQuery()
{
    delete d;
}

/*!
  Assigns \a other to this QXmlQuery instance.
 */
QXmlQuery &QXmlQuery::operator=(const QXmlQuery &other)
{
    if(this != &other)
    {
        // TODO
        Q_ASSERT(false);
    }

    return *this;
}

/*!
  Makes QXmlQuery send compile and runtime messages to \a aMessageHandler.

  QXmlQuery does not claim ownership of \a aMessageHandler, unless the default
  message handler is used.

  If a new message handler is set after the query is compiled(which is triggered by a call
  to isValid(), for instance), runtime will use the message handler that was used during compilation.

  The default message handler will write messages to \c stderr using color codes,
  if \c stderr will be able to render the color codes.
 */
void QXmlQuery::setMessageHandler(QAbstractMessageHandler *aMessageHandler)
{
    d->messageHandler = aMessageHandler;
}

/*!
  Returns the message handler that is being used.
 */
QAbstractMessageHandler *QXmlQuery::messageHandler() const
{
    return d->messageHandler;
}

/*!
  Sets this QXmlQuery instance to use the query contained in \a sourceCode. The
  encoding of \a sourceCode will be detected as according to XQuery's rules.

  \a sourceCode must be opened with at least the QIODevice::ReadOnly flag.

  \a documentURI should be set to the location of the \a sourceCode. It is used for message reporting,
  and is used for resolving some of relative URIs in the query(it is the default value of the
  static base URI, to be specific). \a documentURI must be empty or valid. If it's empty, QCoreApplication::applicationFilePath()
  is used. If it is relative, it is resolved against QCoreApplication::applicationFilePath(). If it's invalid, result is undefined.

  If the query contains a static error such as a syntax error, descriptive messages are sent to messageHandler()
  and isValid() will return \c false.

  \a documentURI should be the location of the query. It is used for resolving relative URIs appearing in the query,
  and for message reporting. To be specific, it is the static base URI. If \a documentURI is not empty or valid,
  behavior is undefined. If \a documentURI is empty, the application's executable path is used.

  QXmlQuery only own the QAbstractMessageHandler if the user hasn't supplied it.

  \sa isValid()
 */
void QXmlQuery::setQuery(QIODevice *sourceCode, const QUrl &documentURI)
{
    Q_ASSERT_X(sourceCode, Q_FUNC_INFO, "A null QIODevice pointer cannot be passed.");
    Q_ASSERT_X(sourceCode->isReadable(), Q_FUNC_INFO, "The device must be readable.");

    // FIXME do encoding sniffing
    setQuery(QString::fromUtf8(sourceCode->readAll().constData()), documentURI);
}

/*!
  \overload
  Equivalent to setQuery(QIODevice *, bool &), with the difference that it takes a QString
  for convenience.

  The same behaviors and obligations apply as for the prepareQuery() version that takes a QIODevice.

  No encoding detection will be done, since \a sourceCode is already a Unicode string.
 */
void QXmlQuery::setQuery(const QString &sourceCode, const QUrl &documentURI)
{
    Q_ASSERT_X(documentURI.isEmpty() || documentURI.isValid(), Q_FUNC_INFO,
               "The document URI must be valid.");

    d->componentsForUpdate = QXmlQueryPrivate::QuerySource;
    d->querySource = sourceCode;
    d->queryURI = QXmlQueryPrivate::normalizeQueryURI(documentURI);
}

/*!
  Reads the query found at \a queryURI and sets it.

  If any error occurs, such as that the query doesn't exist, can't be read or is invalid, isValid()
  returns \c false.

  The supported URI schemes are the same as for the XQuery function \c fn:doc(except that
  no query can be loaded from a variable binding).
 
  \a queryURI is used as the static base URI and is resolved against QCoreApplication::applicationFilePath() if it
  is relative. If \a queryURI is empty or invalid, effects and behavior is undefined.

 */
void QXmlQuery::setQuery(const QUrl &queryURI)
{
    Q_ASSERT_X(queryURI.isValid(), Q_FUNC_INFO, "The passed URI must be valid.");

    const QUrl canonicalURI(QXmlQueryPrivate::normalizeQueryURI(queryURI));
    Q_ASSERT(canonicalURI.isValid());
    Q_ASSERT(!canonicalURI.isRelative());

    QNetworkAccessManager networkManager;
    QIODevice *const result = QPatternist::AccelTreeResourceLoader::load(canonicalURI, &networkManager);

    if(result)
        setQuery(QString::fromUtf8(result->readAll().constData()), queryURI);
    else
    {
        d->querySource = QString();
        d->componentsForUpdate = QXmlQueryPrivate::QuerySource;
    }
}

/*!
 Makes a variable called \a name that has value \a value, avalable to the query.

 If \a value is \c null, any existing binding by name \a name is erased.

 If a variable by name \a name has previously been bound, the previous binding
 is replaced.

 \sa QVariant::isValid(),
     {QtXDM}{How QVariant maps to XQuery's Data Model},
     QXmlItem::isNull()
 */
void QXmlQuery::bindVariable(const QXmlName &name, const QXmlItem &value)
{
    Q_ASSERT_X(!name.isNull(), Q_FUNC_INFO, "The name cannot be null.");

    if(d->variableBindings.contains(name))
    {
        /* If the type of the variable changed(as opposed to only the value),
         * we will have to recompile. */
        if(QXmlQueryPrivate::isSameType(d->variableBindings.value(name), value) || value.isNull())
            d->componentsForUpdate = QXmlQueryPrivate::VariableBindings;
    }
    else if(value.isNull())
        d->componentsForUpdate = QXmlQueryPrivate::VariableBindings;

    d->deviceBindings.take(name);
    d->variableBindings.insert(name, value);
}

/*!
 \overload

 Same as above, but constructs a QXmlName that has an empty namespace and local name \a localName.
 This is convenience for calling:

 \code
    query.bindVariable(QXmlName(query.namePool(), localName), value);
 \endcode

 */
void QXmlQuery::bindVariable(const QString &localName, const QXmlItem &value)
{
    bindVariable(QXmlName(d->namePool, localName), value);
}

/*!
 Binds the QIODevice \a device to the variable \a name.

 A QIODevice is exposed as a URI of type \c xs:anyURI to the query, which
 can be passed to the \c fn:doc() function for reading.

 \code
    QByteArray myDocument;
    QBuffer buffer(&myDocument); // This is a QIODevice.
    buffer.open(QIODevice::ReadOnly);
    QXmlQuery query;
    query.bindVariable("myDocument", &buffer);
    query.setQuery("declare variable $myDocument external; $myDocument");
 \endcode

 The user is reponsible for ensuring the device is opened, and is readable, otherwise
 effects and behaviors are undefined.

 If a variable binding by name \a name already exists, the existing is overriden.

 If \a device is \c null, result is undefined.

 The URI that the variable represents is arbitrary and may change.
 */
void QXmlQuery::bindVariable(const QXmlName &name, QIODevice *device)
{
    Q_ASSERT_X(device, Q_FUNC_INFO, "A valid QIODevice pointer must be passed.");
    Q_ASSERT_X(device->isReadable(), Q_FUNC_INFO, "A readable QIODevice must be passed.");
    d->deviceBindings.insert(name, device);
    d->variableBindings.take(name);

    d->componentsForUpdate = QXmlQueryPrivate::VariableBindings;
}

/*!
 \overload
 Same as above, but constructs variable binding in form of o a URI, whose name is an empty
 namespace and local name \a localName and binds \a device.

 This is convenience for calling:

 \code
    query.bindVariable(QXmlName(query.namePool(), localName), device);
 \endcode
 */
void QXmlQuery::bindVariable(const QString &localName, QIODevice *device)
{
    bindVariable(QXmlName(d->namePool, localName), device);
}

/*!
  Evaluates this query and sends the result as a stream of events to \a callback.

  If \a callback is null or if this query is invalid, behavior is undefined.

  QXmlQuery does not claim ownership of \a callback.

  If an error occur during evaluation, messages are sent to messageHandler() and \c false
  is returned.

  If this query is invalid, \c false is returned and it is undefined what events that
  has been sent to \a callback.

  \sa QAbstractXmlReceiver, evaluateToResult(), isValid()
 */
bool QXmlQuery::evaluateToReceiver(const QAbstractXmlReceiverPointer &callback) const
{
    Q_ASSERT_X(callback, Q_FUNC_INFO,
               "A valid callback must be passed. Otherwise the result cannot be sent anywhere.");

    if(isValid())
    {
        try
        {
            /* This order is significant. expression() might cause query recompilation, and as
             * part of that it recreates the static context. However, if we create the
             * dynamic context before the query recompilation has been triggered, it
             * will use the old static context, and hence old source locations. */
            const QPatternist::Expression::Ptr expr(d->expression());
            QPatternist::DynamicContext::Ptr dynContext(d->dynamicContext(callback));
            expr->evaluateToSequenceReceiver(dynContext);
            return true;
        }
        catch(const QPatternist::Exception)
        {
            return false;
        }
    }
    else
        return false;
}

/*!
  Attempts to evaluate the query, and returns the result in the QStringList \a target.

  If evaluation succeeds, \c true is returned, otherwise \c false. If the query is invalid or
  evaluation fails, \c false is returned and the content of \a target is undefined.

  The query must evaluate to a sequence of \c xs:string values, and this is checked by compiling
  the query as if it was passed to a function accepting \c xs:string*. This means nodes
  will be converted to strings and values that cannot be promoted to \c xs:string trigger
  type errors, for instance.

  If target is \c null, behavior is undefined.
 */
bool QXmlQuery::evaluateToStringList(QStringList *target) const
{
    Q_ASSERT_X(target, Q_FUNC_INFO, "The pointer to the QStringList cannot be null.");

    if(isValid())
    {
        try
        {
            d->setRequiredType(QPatternist::CommonSequenceTypes::ZeroOrMoreStrings);

            /* This order is significant. expression() might cause query recompilation, and as
             * part of that it recreates the static context. However, if we create the
             * dynamic context before the query recompilation has been triggered, it
             * will use the old static context, and hence old source locations. */
            const QPatternist::Expression::Ptr expr(d->expression());
            if(!expr)
                return false;

            QPatternist::DynamicContext::Ptr dynContext(d->dynamicContext(QAbstractXmlReceiverPointer()));
            const QPatternist::Item::Iterator::Ptr it(expr->evaluateSequence(dynContext));
            QPatternist::Item next(it->next());

            while(!next.isNull())
            {
                target->append(next.stringValue());
                next = it->next();
            }

            return true;
        }
        catch(const QPatternist::Exception)
        {
            return false;
        }
    }
    else
        return false;
}

/*!
    Starts the evaluation and makes it available in \a result.

    If \a result is \c null, result & behavior is undefined.

    The actual evaluation takes place incrementally,
    when QXmlResultItems::next() is called.

    \sa evaluateToReceiver(), QXmlResultItems::next()
*/
void QXmlQuery::evaluateToResult(QXmlResultItems *result) const
{
    Q_ASSERT_X(result, Q_FUNC_INFO, "A null pointer cannot be passed to QXmlQuery::evaluateToResult.");

    if(isValid())
    {
        try
        {
            /* We don't have the d->expression() calls and d->dynamicContext() calls in the same order
             * as seen in QXmlQuery::evaluateToReceiver(), and the reason to why that isn't a problem,
             * is that we call isValid(). */
            const QPatternist::DynamicContext::Ptr dynContext(d->dynamicContext(QAbstractXmlReceiverPointer()));
            result->d_ptr->iterator= d->expression()->evaluateSequence(dynContext);
        }
        catch(bool)
        {
            result->d_ptr->iterator = QPatternist::CommonValues::emptyIterator, QPatternist::DynamicContext::Ptr();
            result->d_ptr->hasError = true;
        }
    }
    else
    {
        result->d_ptr->iterator= QPatternist::CommonValues::emptyIterator, QPatternist::DynamicContext::Ptr();
    }
}

/*!
  Returns \c true if this query is valid, otherwise \c false.

  An invalid query is for instance a query that contains a syntax error, or a QXmlQuery instance
  for which setQuery() hasn't been called.
 */
bool QXmlQuery::isValid() const
{
    return d->isValid();
}

/*!
  Sets the URI resolver to \a resolver.

  \sa uriResolver()
 */
void QXmlQuery::setUriResolver(const QAbstractUriResolver *resolver)
{
    d->uriResolver = resolver;
}

/*!
  Returns the URI resolver in use. If no URI resolver has been set, Patternist
  will use the URI in queries as is.

  The URI resolver provides a level of abstraction or "polymorphic URIs." For instance,
  a query can operate on "logical" URIs which a URI resolver rewrites to physical ones, or
  to translate obsolete or invalid URIs, to new ones.

  Patternist calls the URI resolver for all URIs it encounters, except for
  namespaces. This means more specically:

  \list
    \o All builtin functions that deals with URIs. This is \c fn:doc(),
       and \c fn:doc-available()
    \o The URI for module imports
  \endlist

  For instance, in the case of \c fn:doc(), the
  absolute URI is the base URI in the static context(which most likely is the location
  of the query). Instead of using the URI the user specified, the return value
  of QAbstractUriResolver::resolve() will be used.

  When Patternist calls QAbstractUriResolver::resolve() the absolute URI is
  the URI which the XQuery language mandates should be used, and the relative URI
  is the URI which the user specified.

  QXmlQuery does not own the QAbstractUriResolver instance.

  \sa setUriResolver()
 */
const QAbstractUriResolver *QXmlQuery::uriResolver() const
{
    return d->uriResolver;
}

/*!
 Returns the name pool which this QXmlQuery, and all instances which somehow relates to it,
 is using.

 There is no setter for the name pool, since mixing name pools is dangerous.
 */
QXmlNamePool QXmlQuery::namePool() const
{
    return d->namePool;
}

/*!
 Sets the context item to \a item.

 The context item is the item which the focus currently contains. For instance, in the expression \c p/span, the
 elements that the expression \c p evaluations(a sequence), acts as a focus for the following expression, \c span.

 The context item is the global focus for the query. It can only be a single item, and can be accessed using
 the context item expression: \c .(a dot).

 By default the return value is a default constructed QXmlItem, which means that the focus
 is undefined.

 \sa contextItem()
 */
void QXmlQuery::setContextItem(const QXmlItem &item)
{
    d->contextItem = item;
}

/*!
 Returns the context item. See setContextItem() for an explanation of what it is.

 \sa setContextItem()
 */
QXmlItem QXmlQuery::contextItem() const
{
    return d->contextItem;
}

QT_END_NAMESPACE

