/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

/*
 * QXmlName is conceptually identical to QXmlName. The difference is that the latter
 * is elegant, powerful and fast.
 *
 * However, it is too powerful and too open and not at all designed for being public. QXmlName,
 * in contrast, is only a public marker, that for instance uses a qint64 instead of qint32, such
 * that we in the future can use that, if needed.
 */

#include "qnamepool_p.h"
#include "qxmlname.h"
#include "qxmlnamepool.h"
#include "qxpathhelper_p.h"

QT_BEGIN_NAMESPACE

/*!
  \class QXmlName
  \brief The QXmlName class represents a name for an XML node in a namespace-aware and efficient way.
  \reentrant
  \since 4.4
  \ingroup xml-tools

  QXmlName represents a name in XML, such as \c body in the namespace http://www.w3.org/1999/xhtml/, in
  an efficient and safe way. Instances are created using the constructor and
  localName(), namespaceUri() and prefix() can be used to access the components.
  QXmlName's default constructor constructs null names.

  QXmlName represents names by using a pooling mechanism behind the curtains. One of the side
  effects of this mechanism, is that QXmlName instances are tied to the QXmlNamePool it
  was created with. For instance, one cannot call namespaceUri() with a name
  that was created with another QXmlNamePool. If one do, behavior is undefined. This makes
  QXmlName efficient to pass around, and its comparison operators are swift too.

  \target What is a QXmlNameName?
  \section1 What is a QXmlNameName?

  The name of an element or attribute has three components: its namespace URI, a local name,
  and a prefix. For the purpose of what a name refers to the prefix is insignficant, and is only
  a mapping to the namespace. For instance, this element node:

  \code
  <svg xmlns="http://www.w3.org/2000/svg"/>
  \endcode

  and:

  \code
  <x:svg xmlns:x="http://www.w3.org/2000/svg"/>
  \endcode

  both represents the same name and compares equal.
  
  A local name can never be an empty string, although the prefix and namespace
  URI can. If the prefix is not empty, the namespace URI cannot be empty. Both
  the local name and the prefix(when not being empty), must be valid so called
  NCNames, such as "abc.def" or "abc123", as according to Namespaces in XML 1.0.

  \sa {http://www.w3.org/TR/REC-xml-names/#NT-NCName} {Namespaces in XML 1.0 (Second Edition), [4] NCName}
 */

/*!
   \enum QXmlName::Constant
   \internal
   Various constants used in the QPatternist::NamePool and QXmlName.

   Setting of the mask enums use essentially this:

   \code
   // Fills the bits from beg to end with 1s and leaves the rest as 0.

   template<typename IntegralT>
    inline IntegralT bitmask(IntegralT begin, IntegralT end)
    {
        IntegralT filled_bits = (1 << (end - begin + 1)) - 1;
        return filled_bits << begin;
    }
   \endcode

   The masks, such as LocalNameMask, are positive. That is, for the
   area which the name resides, the bits are set.
 */

/*!
 Constructs a QXmlName instance that inserts \a localName,
 \a namespaceURI and \a prefix
 into \a namePool.

 The user guarantees that the string components are valid for a QName. See
 the description of this class for a summary.
 */
QXmlName::QXmlName(QXmlNamePool &namePool,
                   const QString &localName,
                   const QString &namespaceURI,
                   const QString &prefix)
{
    Q_ASSERT_X(prefix.isEmpty() || QPatternist::XPathHelper::isNCName(prefix), Q_FUNC_INFO,
               "The prefix is invalid, maybe the arguments were mixed up?");
    Q_ASSERT_X(QPatternist::XPathHelper::isNCName(localName), Q_FUNC_INFO,
               "The local name is invalid, maybe the arguments were mixed up?");
 
    m_qNameCode = namePool.d->allocateQName(namespaceURI, localName, prefix).code();
}

/*!
  \typedef QXmlName::Code

   Stores a namespace URI, local name and prefix.
 */

/*!
  Returns true if this QXmlName is uninitialized and does not contain a name at all.
 */
bool QXmlName::isNull() const
{
    return m_qNameCode == InvalidCode;
}

/*!
  Constructs a QXmlName instance that is null.

  \sa isNull()
 */
QXmlName::QXmlName() : m_qNameCode(InvalidCode)
{
}

/*!
  \fn QXmlName::QXmlName(const NamespaceCode uri,
                         const LocalNameCode ln,
                         const PrefixCode p = 0)
  \internal
 */

/*!
 \fn QXmlName::hasPrefix() const
 \internal

 Returns \c true if this QXmlName has a prefix that isn't
 the empty string, otherwise false.
 */

/*!
 \fn bool QXmlName::hasNamespace() const
 \internal

 Returns \c true if this QXmlName has a non-empty namespace.
 */

/*!
  \fn Code QXmlName::code() const
  \internal

   Returns the internal code that contains the local name, prefix and
   namespace components. It is opaque when used outside QXmlName, but can
   be useful when one wants to put a QXmlName in a hash, and the prefix is
   significant.
 */

/*!
  Returns \c true if this QXmlName is equal to \a other, otherwise \c false.

  Two QXmlNames are equal if the namespace URIs and local names they represents
  are equal. The prefix is ignored.
 */
bool QXmlName::operator==(const QXmlName &other) const
{
    return (m_qNameCode & ExpandedNameMask) == (other.m_qNameCode & ExpandedNameMask);
}

/*!
  Returns the opposite of applying operator==() on this QXmlName and \a other.
 */
bool QXmlName::operator!=(const QXmlName &other) const
{
    return !operator==(other);
}

/*!
  \fn bool QXmlName::isLexicallyEqual(const QXmlName &other) const

  Returns \c true if this and \a other are lexically equal. That is, if their
  prefix and local names are equal.
 */

/*!
 \fn uint qHash(const QXmlName &name)
 \relates QXmlName

 Computes a hash key based on the local name combined with the namespace URI in \a name.

 That is, the prefix, which only is a syntactical distriction, is ignored.
 */
uint qHash(const QXmlName &name)
{
    return name.m_qNameCode & QXmlName::ExpandedNameMask;
}

/*!
 Returns the namespace URI.

 \a namePool is used internally for looking up the string.
 */
QString QXmlName::namespaceUri(const QXmlNamePool &namePool) const
{
    if(isNull())
        return QString();
    else
        return namePool.d->stringForNamespace(namespaceURI());
}

/*!
 Returns the prefix.

 \a namePool is used internally for looking up the string.
 */
QString QXmlName::prefix(const QXmlNamePool &namePool) const
{
    if(isNull())
        return QString();
    else
        return namePool.d->stringForPrefix(prefix());
}

/*!
 Returns the local name.

 \a namePool is used internally for looking up the string.
 */
QString QXmlName::localName(const QXmlNamePool &namePool) const
{
    if(isNull())
        return QString();
    else
        return namePool.d->stringForLocalName(localName());
}

/*!
 Returns a string that is this QXmlName formatted as a so called Clark Name. For instance,
 the local name "html" with prefix "x" while residing in the XHTML namespace, would be
 returned as "{http://www.w3.org/1999/xhtml/}x:html". The local name "QWidget" with an empty namespace,
 would be returned as "QWidget".

 This function can be useful for debugging, or when a name needs to be represented as a string in an
 informal way. As implemented, a Clark Name cannot be deterministically parsed.

 \a namePool is used internally for looking up the strings.

 \sa {http://www.jclark.com/xml/xmlns.htm} {XML Namespaces, James Clark}
 */
QString QXmlName::toClarkName(const QXmlNamePool &namePool) const
{
    if(isNull())
        return QLatin1String("QXmlName(null)");
    else
    {
        const QString ns(namespaceUri(namePool));

        if(ns.isEmpty())
            return localName(namePool);
        else
        {
            const QString p(prefix(namePool));
            const QString l(localName(namePool));

            return   QChar::fromLatin1('{')
                   + ns
                   + QChar::fromLatin1('}')
                   + (p.isEmpty() ? l : p + QChar::fromLatin1(':') + l);
        }
    }
}

/*!
  Assigns \a other to this and return this.
 */
QXmlName &QXmlName::operator=(const QXmlName &other)
{
    m_qNameCode = other.m_qNameCode;
    return *this;
}

/*!
  \typedef QXmlName::LocalNameCode
  \internal
 */

/*!
  \typedef QXmlName::PrefixCode
  \internal
 */

/*!
  \typedef QXmlName::NamespaceCode
  \internal
 */

/*!
  \fn void QXmlName::setLocalName(const LocalNameCode c)
  \internal
*/

/*!
  \fn LocalNameCode QXmlName::localName() const
  \internal
*/

/*!
  \fn PrefixCode QXmlName::prefix() const
  \internal
*/

/*!
  \fn NamespaceCode QXmlName::namespaceURI() const
  \internal
*/

/*!
  \fn void QXmlName::setNamespaceURI(const NamespaceCode c)
  \internal
*/

/*!
  \fn void QXmlName::setPrefix(const PrefixCode c)
  \internal
*/
QT_END_NAMESPACE

