/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include <QUrl>
#include <QVector>
#include <QXmlNamePool>

#include "qabstractxmlnodemodel_p.h"
#include "qemptyiterator_p.h"
#include "qitemmappingiterator_p.h"
#include "qsequencemappingiterator_p.h"
#include "qsimplexmlnodemodel.h"
#include "qsingletoniterator_p.h"

QT_BEGIN_NAMESPACE

using namespace QPatternist;

class QSimpleXmlNodeModelPrivate : public QAbstractXmlNodeModelPrivate
{
public:
    QSimpleXmlNodeModelPrivate(const QXmlNamePool &np) : namePool(np)
    {
    }

    const QXmlNamePool namePool;
};

/*!
  \enum QSimpleXmlNodeModel::SimpleAxis

  Four axes that only contain one node each.

  \value Parent The parent of the context node
  \value FirstChild The first child of the context node
  \value PreviousSibling The previous sibling of the context node
  \value NextSibling The next sibling of the context node
*/

/*!
  \class QSimpleXmlNodeModel
  \brief The QSimpleXmlNodeModel class provides a simple yet powerful way of implementing QAbstractXmlNodeModel.
  \reentrant
  \since 4.4
  \ingroup xml-tools

  Since QAbstractXmlNodeModel::iterate() requires a sub-class of QAbstractXmlForwardIterator
  to be returned for each axis, it can be a bit time consuming to implement. For that
  reason QSimpleXmlNodeModel exists which has nextFromSimpleAxis(), that that is a quick
  but often sufficient way of providing the node navigation. Behind the scenes
  QSimpleXmlNodeModel::iterate() "emulates" the full XPath axes using nextFromSimpleAxis().

  Apart from nextFromSimpleAxis(), QSimpleXmlNodeModel provides default implementations
  for many of the members in QAbstractXmlNodeModel that are suitable for a wide range of scenarios.
 */

/*!
  Constructs a QSimpleXmlNodeModel instance that can be used with the name pool \a namePool. It can subsequently be accessed
  with namePool().
 */
QSimpleXmlNodeModel::QSimpleXmlNodeModel(const QXmlNamePool &namePool) : QAbstractXmlNodeModel(new QSimpleXmlNodeModelPrivate(namePool))
{
}

/*!
  Destructor.
 */
QSimpleXmlNodeModel::~QSimpleXmlNodeModel()
{
}

/*!
  \fn QXmlNodeModelIndex QSimpleXmlNodeModel::nextFromSimpleAxis(SimpleAxis axis, const QXmlNodeModelIndex &origin) const = 0

  When Patternist calls iterate(), QSimpleXmlNodeModel create iterators that calls nextFromSimpleAxis() and "emulates" real
  XPath axes using QSimpleXmlNodeModel::SimpleAxis. Therefore, the implementation of this function should return
  the node, if any, that appear on axis \a axis, from \a origin.

  If no such node is available, a default constructed QXmlNodeModelIndex is returned.

  QSimpleXmlNodeModel removes the need to handle redundant corner cases by guaranteeing that it will never ask for:

  \list
    \o Children or siblings for attributes.
    \o Children for comments, processing instructions, and text nodes.
    \o Siblings or parents for document nodes.
  \endlist

  A typical implementation does a switch over \a axis:

  \code
  QXmlItemIteratorPointer MyTreeModel::nextFromSimpleAxis(SimpleAxis axis, const QXmlNodeModelIndex &origin) const
  {
    // Convert the QXmlNodeModelIndex to a value that is specific to what we represent.
    const MyValue value = toMyValue(ni);

    switch(axis)
    {
        case Parent:
            return toNodeIndex(value.parent());
        case FirstChild:
        case PreviousSibling:
        case NextSibling:
            // and so on
    }
  }
  \endcode

 */



QXmlNodeModelIndex QSimpleXmlNodeModel::mapToItem(const QXmlNodeModelIndex &ni,
                                                  const DynamicContext::Ptr &) const
{
    return ni;
}

inline QAbstractXmlForwardIterator<QXmlNodeModelIndex>::Ptr QSimpleXmlNodeModel::mapToSequence(const QXmlNodeModelIndex &ni,
                                                                                            const DynamicContext::Ptr &) const
{
    return ni.iterate(QXmlNodeModelIndex::AxisChild);
}

namespace QPatternist
{
    class MergeIterator
    {
    public:
        inline MergeIterator()
        {
        }

        inline
        QXmlNodeModelIndexIteratorPointer
        mapToSequence(const QXmlNodeModelIndexIteratorPointer &it,
                      const DynamicContext::Ptr &) const
        {
            return it;
        }

    private:
        Q_DISABLE_COPY(MergeIterator)
    };

    static MergeIterator mergeIterator;
};

/*!
  \reimp

  Returns iterators that track state and calls nextFromSimpleAxis() iteratively. Typically, when sub-classing QSimpleXmlNodeModel,
  you don't reimplement this function, but instead implement nextFromSimpleAxis().
 */
QXmlNodeModelIndexIteratorPointer QSimpleXmlNodeModel::iterate(const QXmlNodeModelIndex &ni,
                                                               QXmlNodeModelIndex::Axis axis) const
{
    switch(axis)
    {
        case QXmlNodeModelIndex::AxisSelf:
            return makeSingletonIterator(ni);
        case QXmlNodeModelIndex::AxisParent:
        {
            if(kind(ni) == QXmlNodeModelIndex::Document)
                return makeEmptyIterator<QXmlNodeModelIndex>();
            else
                return makeSingletonIterator(nextFromSimpleAxis(Parent, ni));
        }
        case QXmlNodeModelIndex::AxisNamespace:
            return makeEmptyIterator<QXmlNodeModelIndex>();
        case QXmlNodeModelIndex::AxisAncestor:
        {
            QList<QXmlNodeModelIndex> ancestors;
            QXmlNodeModelIndex ancestor = nextFromSimpleAxis(Parent, ni);

            while(!ancestor.isNull())
            {
                ancestors.append(ancestor);
                ancestor = nextFromSimpleAxis(Parent, ni);
            }

            return makeListIterator(ancestors);
        }
        case QXmlNodeModelIndex::AxisAncestorOrSelf:
        {
            QList<QXmlNodeModelIndex> ancestors;
            ancestors.append(ni);
            QXmlNodeModelIndex ancestor = nextFromSimpleAxis(Parent, ni);

            while(!ancestor.isNull())
            {
                ancestors.append(ancestor);
                ancestor = nextFromSimpleAxis(Parent, ni);
            }

            return makeListIterator(ancestors);
        }
        case QXmlNodeModelIndex::AxisPrecedingSibling:
        {
            QList<QXmlNodeModelIndex> preceding;
            QXmlNodeModelIndex sibling = nextFromSimpleAxis(PreviousSibling, ni);

            while(!sibling.isNull())
            {
                preceding.append(sibling);
                sibling = nextFromSimpleAxis(PreviousSibling, sibling);
            }

            return makeListIterator(preceding);
        }
        case QXmlNodeModelIndex::AxisFollowingSibling:
        {
            QList<QXmlNodeModelIndex> preceding;
            QXmlNodeModelIndex sibling = nextFromSimpleAxis(NextSibling, ni);

            while(!sibling.isNull())
            {
                preceding.append(sibling);
                sibling = nextFromSimpleAxis(NextSibling, sibling);
            }

            return makeListIterator(preceding);
        }
        case QXmlNodeModelIndex::AxisChild:
        {
            QList<QXmlNodeModelIndex> children;
            QXmlNodeModelIndex child = nextFromSimpleAxis(FirstChild, ni);

            while(!child.isNull())
            {
                children.append(child);
                child = nextFromSimpleAxis(NextSibling, child);
            }

            return makeListIterator(children);
        }
        case QXmlNodeModelIndex::AxisDescendant:
        {
            return makeItemMappingIterator<QXmlNodeModelIndex>(this, makeSequenceMappingIterator<QXmlNodeModelIndex>(this,
                                                                                                                     mapToSequence(ni,
                                                                                                                                 DynamicContext::Ptr()),
                                                                                                                     DynamicContext::Ptr()),
                                                               DynamicContext::Ptr());
        }
        case QXmlNodeModelIndex::AxisAttribute:
            return makeVectorIterator(attributes(ni));
        case QXmlNodeModelIndex::AxisDescendantOrSelf:
        {
            const QXmlNodeModelIndexIteratorPointer single(makeSingletonIterator(ni));
            const QXmlNodeModelIndexIteratorPointer descendant(iterate(ni, QXmlNodeModelIndex::AxisDescendant));
            QVector<QXmlNodeModelIndexIteratorPointer> iterators;
            iterators.append(single);
            iterators.append(descendant);

            return makeSequenceMappingIterator<QXmlNodeModelIndex>(&mergeIterator,
                                                                   makeVectorIterator(iterators),
                                                                   DynamicContext::Ptr());

        }
        case QXmlNodeModelIndex::AxisFollowing:
        {
        }
        case QXmlNodeModelIndex::AxisPreceding:
        {
            Q_ASSERT(false);
            // TODO
            return makeEmptyIterator<QXmlNodeModelIndex>();
            /*
            QList result;

            QXmlNodeModelIndex current = ni;
            QXmlNodeModelIndex prev(nextFromSimpleAxis(PreviousSibling, current));

            do
            {
                if(prev)
                {
                    result.append(prev);
                    current = prev;
                    continue;
                }

                current = nextFromSimpleAxis(Parent, current);
                if(current)
                    result.append(current);
                else
                    return nextFromSimpleAxis(PreviousSibling, parent);
            }
            */

        }
    }

    Q_ASSERT_X(false, Q_FUNC_INFO, "Unknown axis, internal error.");
    return makeEmptyIterator<QXmlNodeModelIndex>();
}

/*!
 If \a node is an element or attribute, typedValue() is called, and the return value
 converted to a string, as per XQuery's rules.

 If \a node is another type of node, the empty string is returned.

 If this function is overriden for for instance comments or processing instructions, it is important
 to remember to call this function for elements and attribute, if they have atomic values that are not
 of type \c xs:string, such that they get formatted according to XQuery.

 */
QString QSimpleXmlNodeModel::stringValue(const QXmlNodeModelIndex &node) const
{
    const QXmlNodeModelIndex::NodeKind k= kind(node);
    if(k == QXmlNodeModelIndex::Element || k == QXmlNodeModelIndex::Attribute)
        return AtomicValue::toXDM(typedValue(node)).stringValue();
    else
        return QString();
}

/*!
  Returns the base URI for \a node. This is always the document URI. This is safe, works and is often ok.

  \sa documentUri()
 */
QUrl QSimpleXmlNodeModel::baseUri(const QXmlNodeModelIndex &node) const
{
    return documentUri(node);
}

/*!
  Returns the name pool that is associated with this model. The implementation of name()
  would use this to create names.
 */
QXmlNamePool QSimpleXmlNodeModel::namePool() const
{
    Q_D(const QSimpleXmlNodeModel);

    return d->namePool;
}

/*!
  Returns always an empty QVector. This signals that no namespace bindings are in scope for \a node.
 */
QVector<QXmlName> QSimpleXmlNodeModel::namespaceBindings(const QXmlNodeModelIndex &node) const
{
    Q_UNUSED(node);
    return QVector<QXmlName>();
}

/*!
  Returns the attributes that element \a element has.

  The caller guarantees that \a element is always an element.
 */
QVector<QXmlNodeModelIndex> QSimpleXmlNodeModel::attributes(const QXmlNodeModelIndex &element) const
{
    Q_UNUSED(element);
    return QVector<QXmlNodeModelIndex>();
}

/*!
  Always returns a default constructed QXmlNodeModelIndex instance, regardless of \a id.

  This effectively means the model has no elements that has an id.
 */
QXmlNodeModelIndex QSimpleXmlNodeModel::id(const QString &id) const
{
    Q_UNUSED(id);
    return QXmlNodeModelIndex();
}

/*!
  Always returns a default constructed QXmlNodeModelIndex instance, regardless of \a idref.

  This effectively means the model has no elements or attributes are of type \c IDREF.
 */
QXmlNodeModelIndex QSimpleXmlNodeModel::idref(const QString &idref) const
{
    Q_UNUSED(idref);
    return QXmlNodeModelIndex();
}

QT_END_NAMESPACE


