/****************************************************************************
**
** Copyright (C) 2006-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtXMLPatterns module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "qabstractxmlreceiver.h"
#include "qacceliterators_p.h"
#include "qacceltree_p.h"
#include "qatomicstring_p.h"
#include "qcommonvalues_p.h"
#include "qcompressedwhitespace_p.h"
#include "qdebug_p.h"
#include "quntypedatomic_p.h"
#include "qxpathhelper_p.h"

QT_BEGIN_NAMESPACE

using namespace QPatternist;

void AccelTree::printStats(const NamePool::Ptr &np) const
{
    Q_ASSERT(np);
#ifdef QT_NO_DEBUG
    Q_UNUSED(np); /* Needed when compiling in release mode. */
#else
    const int len = basicData.count();

    pDebug() << "AccelTree stats for" << (m_documentURI.isEmpty() ? QString::fromLatin1("<empty URI>") : m_documentURI.toString());
    pDebug() << "Maximum pre number:" << maximumPreNumber();
    pDebug() << "+---------------+-------+-------+---------------+-------+--------------+-------+";
    pDebug() << "| Pre number    | Depth | Size  | Post Number   | Kind  | Name         | Value |";
    pDebug() << "+---------------+-------+-------+---------------+-------+--------------+-------+";
    for(int i = 0; i < len; ++i)
    {
        const BasicNodeData &v = basicData.at(i);
        pDebug() << "|" << i
                 << "\t\t|" << v.depth()
                 << "\t|" << v.size()
                 << "\t|" << postNumber(i)
                 << "\t|" << v.kind()
                 << "\t\t|" << (v.name().isNull() ? QString::fromLatin1("(none)") : np->displayName(v.name()))
                 << "\t\t|" << data.value(i)
                 << "\t|";
        /*
        pDebug() << "|" << QString().arg(i, 14)
                 << "|" << QString().arg(v.depth(), 6)
                 << "|" << QString().arg(v.size(), 6)
                 << "|" << QString().arg(postNumber(i), 14)
                 << "|" << QString().arg(v.kind(), 6)
                 << "|";
                 */
    }
    pDebug() << "+---------------+-------+-------+---------------+-------+--------------+";
#endif
}

QUrl AccelTree::baseUri(const QXmlNodeModelIndex &ni) const
{
    switch(kind(toPreNumber(ni)))
    {
        case QXmlNodeModelIndex::Document:
            return baseUri();
        case QXmlNodeModelIndex::Element:
        {
            const QXmlNodeModelIndex::Iterator::Ptr it(iterate(ni, QXmlNodeModelIndex::AxisAttribute));
            QXmlNodeModelIndex next(it->next());

            while(!next.isNull())
            {
                if(next.name() == QXmlName(StandardNamespaces::xml, StandardLocalNames::base))
                {
                    const QUrl candidate(next.stringValue());
                    //  TODO. The xml:base spec says to do URI escaping here.

                    if(!candidate.isValid())
                        return QUrl();
                    else if(candidate.isRelative())
                    {
                        const QXmlNodeModelIndex par(parent(ni));

                        if(par.isNull())
                            return baseUri().resolved(candidate);
                        else
                            return par.baseUri().resolved(candidate);
                    }
                    else
                        return candidate;
                }

                next = it->next();
            }

            /* We have no xml:base-attribute. Can any parent supply us a base URI? */
            const QXmlNodeModelIndex par(parent(ni));

            if(par.isNull())
                return baseUri();
            else
                return par.baseUri();
        }
        case QXmlNodeModelIndex::ProcessingInstruction:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Comment:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Attribute:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Text:
        {
            const QXmlNodeModelIndex par(ni.iterate(QXmlNodeModelIndex::AxisParent)->next());
            if(par.isNull())
                return QUrl();
            else
                return par.baseUri();
        }
        case QXmlNodeModelIndex::Namespace:
            return QUrl();
    }

    Q_ASSERT_X(false, Q_FUNC_INFO, "This line is never supposed to be reached.");
    return QUrl();
}

QUrl AccelTree::documentUri(const QXmlNodeModelIndex &ni) const
{
    if(kind(toPreNumber(ni)) == QXmlNodeModelIndex::Document)
        return documentUri();
    else
        return QUrl();
}

QXmlNodeModelIndex::NodeKind AccelTree::kind(const QXmlNodeModelIndex &ni) const
{
    return kind(toPreNumber(ni));
}

QXmlNodeModelIndex::DocumentOrder AccelTree::compareOrder(const QXmlNodeModelIndex &ni1, const QXmlNodeModelIndex &ni2) const
{
    Q_ASSERT_X(ni1.model() == ni2.model(), Q_FUNC_INFO, "The API docs guarantees the two nodes are from the same model");

    const PreNumber p1 = ni1.data();
    const PreNumber p2 = ni2.data();

    if(p1 == p2)
        return QXmlNodeModelIndex::Is;
    else if(p1 < p2)
        return QXmlNodeModelIndex::Precedes;
    else
        return QXmlNodeModelIndex::Follows;
}

QXmlNodeModelIndex AccelTree::root(const QXmlNodeModelIndex &) const
{
    return createIndex(0);
}

QXmlNodeModelIndex AccelTree::parent(const QXmlNodeModelIndex &ni) const
{
    const AccelTree::PreNumber p = basicData.at(toPreNumber(ni)).parent();

    if(p == -1)
        return QXmlNodeModelIndex();
    else
        return createIndex(p);
}

QXmlNodeModelIndex::Iterator::Ptr AccelTree::iterate(const QXmlNodeModelIndex &ni, QXmlNodeModelIndex::Axis axis) const
{
    const PreNumber preNumber = toPreNumber(ni);

    switch(axis)
    {
        case QXmlNodeModelIndex::AxisChild:
        {
            if(hasChildren(preNumber))
                return QXmlNodeModelIndex::Iterator::Ptr(new ChildIterator(this, preNumber));
            else
                return makeEmptyIterator<QXmlNodeModelIndex>();
        }
        case QXmlNodeModelIndex::AxisAncestor:
        {
            if(hasParent(preNumber))
                return QXmlNodeModelIndex::Iterator::Ptr(new AncestorIterator<false>(this, preNumber));
            else
                return makeEmptyIterator<QXmlNodeModelIndex>();
        }
        case QXmlNodeModelIndex::AxisAncestorOrSelf:
            return QXmlNodeModelIndex::Iterator::Ptr(new AncestorIterator<true>(this, preNumber));
        case QXmlNodeModelIndex::AxisParent:
        {
            if(hasParent(preNumber))
                return makeSingletonIterator(createIndex(parent(preNumber)));
            else
                return makeEmptyIterator<QXmlNodeModelIndex>();
        }
        case QXmlNodeModelIndex::AxisDescendant:
        {
            if(hasChildren(preNumber))
                return QXmlNodeModelIndex::Iterator::Ptr(new DescendantIterator<false>(this, preNumber));
            else
                return makeEmptyIterator<QXmlNodeModelIndex>();
        }
        case QXmlNodeModelIndex::AxisDescendantOrSelf:
            return QXmlNodeModelIndex::Iterator::Ptr(new DescendantIterator<true>(this, preNumber));
        case QXmlNodeModelIndex::AxisFollowing:
        {
            if(preNumber == maximumPreNumber())
                return makeEmptyIterator<QXmlNodeModelIndex>();
            else
                return QXmlNodeModelIndex::Iterator::Ptr(new FollowingIterator(this, preNumber));
        }
        case QXmlNodeModelIndex::AxisAttribute:
        {
            if(hasChildren(preNumber) && kind(preNumber + 1) == QXmlNodeModelIndex::Attribute)
                return QXmlNodeModelIndex::Iterator::Ptr(new AttributeIterator(this, preNumber));
            else
                return makeEmptyIterator<QXmlNodeModelIndex>();
        }
        case QXmlNodeModelIndex::AxisPreceding:
        {
            if(preNumber == 0)
                return makeEmptyIterator<QXmlNodeModelIndex>();
            else
                return QXmlNodeModelIndex::Iterator::Ptr(new PrecedingIterator(this, preNumber));
        }
        case QXmlNodeModelIndex::AxisSelf:
            return makeSingletonIterator(createIndex(toPreNumber(ni)));
        case QXmlNodeModelIndex::AxisFollowingSibling:
        {
            if(preNumber == maximumPreNumber())
                return makeEmptyIterator<QXmlNodeModelIndex>();
            else
                return QXmlNodeModelIndex::Iterator::Ptr(new SiblingIterator<true>(this, preNumber));
        }
        case QXmlNodeModelIndex::AxisPrecedingSibling:
        {
            if(preNumber == 0)
                return makeEmptyIterator<QXmlNodeModelIndex>();
            else
                return QXmlNodeModelIndex::Iterator::Ptr(new SiblingIterator<false>(this, preNumber));
        }
        case QXmlNodeModelIndex::AxisNamespace:
            return makeEmptyIterator<QXmlNodeModelIndex>();
    }

    Q_ASSERT(false);
    return QXmlNodeModelIndex::Iterator::Ptr();
}

QXmlName AccelTree::name(const QXmlNodeModelIndex &ni) const
{
    /* If this node type does not have a name(for instance, it's a comment)
     * we will return the default constructed value, which is conformant with
     * this function's contract. */
    return name(toPreNumber(ni));
}

QVector<QXmlName> AccelTree::namespaceBindings(const QXmlNodeModelIndex &ni) const
{
    /* We get a hold of the ancestor, and loop them in reverse document
     * order(first the parent, then the parent's parent, etc). As soon
     * we find a binding that hasn't already been added, we add it to the
     * result list. In that way, declarations appearing further down override
     * those further up. */

    const PreNumber preNumber = toPreNumber(ni);

    const QXmlNodeModelIndex::Iterator::Ptr it(new AncestorIterator<true>(this, preNumber));
    QVector<QXmlName> result;
    Item n(it->next());

    /* Whether xmlns="" has been encountered. */
    bool hasUndeclaration = false;

    while(n)
    {
        const QVector<QXmlName> forNode(namespaces.value(toPreNumber(n.asNode())));
        const int len = forNode.size();

        for(int i = 0; i < len; ++i)
        {
            const QXmlName &nsb = forNode.at(i);

            if(nsb.prefix() == StandardPrefixes::empty &&
               nsb.namespaceURI() == StandardNamespaces::empty)
            {
                hasUndeclaration = true;
                continue;
            }

            if(!hasPrefix(result, nsb.prefix()))
            {
                /* We've already encountered an undeclaration, so we're supposed to skip
                 * them. */
                if(hasUndeclaration && nsb.prefix() == StandardPrefixes::empty)
                    continue;
                else
                    result.append(forNode.at(i));
            }
        }

        n = it->next();
    }

    result.append(QXmlName(StandardNamespaces::xml, StandardLocalNames::empty, StandardPrefixes::xml));

    return result;
}

void AccelTree::sendNamespaces(const QXmlNodeModelIndex &n,
                               const QExplicitlySharedDataPointer<QAbstractXmlReceiver> &receiver) const
{
    Q_ASSERT(n.kind() == QXmlNodeModelIndex::Element);

    const QXmlNodeModelIndex::Iterator::Ptr it(iterate(n, QXmlNodeModelIndex::AxisAncestorOrSelf));
    QXmlNodeModelIndex next(it->next());
    QVector<QXmlName::PrefixCode> alreadySent;

    while(!next.isNull())
    {
        const PreNumber preNumber = toPreNumber(next);

        const QVector<QXmlName> &nss = namespaces.value(preNumber);

        /* This is by far the most common case. */
        if(nss.isEmpty())
        {
            next = it->next();
            continue;
        }

        const int len = nss.count();

        for(int i = 0; i < len; ++i)
        {
            const QXmlName &name = nss.at(i);

            if(!alreadySent.contains(name.prefix()))
            {
                alreadySent.append(name.prefix());
                receiver->namespaceBinding(nss.at(i));
            }
        }

        next = it->next();
    }
}

QString AccelTree::stringValue(const QXmlNodeModelIndex &ni) const
{
    const PreNumber preNumber = toPreNumber(ni);

    switch(kind(preNumber))
    {
        case QXmlNodeModelIndex::Element:
        {
            /* Concatenate all text nodes that are descendants of this node. */
            if(!hasChildren(preNumber))
                return QString();

            const AccelTree::PreNumber stop = preNumber + size(preNumber);
            AccelTree::PreNumber pn = preNumber + 1; /* Jump over ourselves. */
            QString result;

            for(; pn <= stop; ++pn)
            {
                if(kind(pn) == QXmlNodeModelIndex::Text)
                    result.append(data.value(pn));
            }

            return result;
        }
        case QXmlNodeModelIndex::Text:
        {
            if(isCompressed(preNumber))
                return CompressedWhitespace::decompress(data.value(preNumber));
            /* Else, fallthrough. It's not compressed so use it as it is. */
        }
        case QXmlNodeModelIndex::Attribute:
        /* Fallthrough */
        case QXmlNodeModelIndex::ProcessingInstruction:
        /* Fallthrough */
        case QXmlNodeModelIndex::Comment:
            return data.value(preNumber);
        case QXmlNodeModelIndex::Document:
        {
            /* Concatenate all text nodes in the whole document. */

            QString result;
            // Perhaps we can QString::reserve() the result based on the size?
            const AccelTree::PreNumber max = maximumPreNumber();

            for(AccelTree::PreNumber i = 0; i <= max; ++i)
            {
                if(kind(i) == QXmlNodeModelIndex::Text)
                    result += data.value(i);
            }

            return result;
        }
        default:
        {
            Q_ASSERT_X(false, Q_FUNC_INFO,
                       "A node type that doesn't exist in the XPath Data Model was encountered.");
            return QString(); /* Dummy, silence compiler warning. */
        }
    }
}

QVariant AccelTree::typedValue(const QXmlNodeModelIndex &n) const
{
    return stringValue(n);
}

bool AccelTree::hasPrefix(const QVector<QXmlName> &nbs, const QXmlName::PrefixCode prefix)
{
    const int size = nbs.size();

    for(int i = 0; i < size; ++i)
    {
        if(nbs.at(i).prefix() == prefix)
            return true;
    }

    return false;
}

ItemType::Ptr AccelTree::type(const QXmlNodeModelIndex &ni) const
{
    /* kind() is manually inlined here to avoid a virtual call. */
    return XPathHelper::typeFromKind(basicData.at(toPreNumber(ni)).kind());
}

Item::Iterator::Ptr AccelTree::sequencedTypedValue(const QXmlNodeModelIndex &n) const
{
    const PreNumber preNumber = toPreNumber(n);

    switch(kind(preNumber))
    {
        case QXmlNodeModelIndex::Element:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Document:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Attribute:
            return makeSingletonIterator(Item(UntypedAtomic::fromValue(stringValue(n))));

        case QXmlNodeModelIndex::Text:
        /* Fallthrough. */
        case QXmlNodeModelIndex::ProcessingInstruction:
        /* Fallthrough. */
        case QXmlNodeModelIndex::Comment:
            return makeSingletonIterator(Item(AtomicString::fromValue(stringValue(n))));
        default:
        {
            Q_ASSERT_X(false, Q_FUNC_INFO,
                       qPrintable(QString::fromLatin1("A node type that doesn't exist "
                                                      "in the XPath Data Model was encountered.").arg(kind(preNumber))));
            return Item::Iterator::Ptr(); /* Dummy, silence compiler warning. */
        }
    }
}

QXmlNodeModelIndex AccelTree::id(const QString &) const
{
    return QXmlNodeModelIndex(); // TODO Look up xml:id at least.
}

QXmlNodeModelIndex AccelTree::idref(const QString &) const
{
    return QXmlNodeModelIndex();
}

QT_END_NAMESPACE

