/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "qpixmap.h"
#include "qpixmap_raster_p.h"

#include "qbitmap.h"
#include "qimage.h"
#include <private/qwidget_p.h>
#include <private/qdrawhelper_p.h>

#if !defined(QT_NO_DIRECT3D) && defined(Q_WS_WIN)
#include <private/qpaintengine_d3d_p.h>
#include <d3d9.h>
extern QDirect3DPaintEngine *qt_d3dEngine();
#endif

QT_BEGIN_NAMESPACE

const uchar qt_pixmap_bit_mask[] = { 0x01, 0x02, 0x04, 0x08,
                                     0x10, 0x20, 0x40, 0x80 };

QRasterPixmapData::QRasterPixmapData(PixelType type)
    : QPixmapData(type, RasterClass)
#if defined(Q_WS_WIN) && !defined(QT_NO_DIRECT3D)
    , texture(0)
#endif
{
}

QRasterPixmapData::~QRasterPixmapData()
{
}

void QRasterPixmapData::resize(int width, int height)
{
    QImage::Format format;
#ifdef Q_WS_QWS
    if (pixelType() == BitmapType)
        format = QImage::Format_Mono;
    else
        format = QScreen::instance()->pixelFormat();

    if (format == QImage::Format_Invalid)
        format = QImage::Format_ARGB32_Premultiplied;
#else
    if (pixelType() == BitmapType)
        format = QImage::Format_MonoLSB;
    else
        format = QImage::Format_RGB32;
#endif

    image = QImage(width, height, format);

    if (pixelType() == BitmapType && !image.isNull()) {
        image.setNumColors(2);
        image.setColor(0, QColor(Qt::color0).rgba());
        image.setColor(1, QColor(Qt::color1).rgba());
    }

    setSerialNumber(image.serialNumber());
}

void QRasterPixmapData::fromImage(const QImage &img,
                                  Qt::ImageConversionFlags flags)
{
    Q_UNUSED(flags);

#ifdef Q_WS_QWS
    QImage::Format format;
    if (pixelType() == BitmapType)
        format = QImage::Format_Mono;
    else
        format = QScreen::instance()->pixelFormat();

    if (format == QImage::Format_Invalid || img.hasAlphaChannel())
        image = img.convertToFormat(QImage::Format_ARGB32_Premultiplied);
    else
        image = img.convertToFormat(format);
#else
    if (pixelType() == BitmapType) {
        image = img.convertToFormat(QImage::Format_MonoLSB);
    } else {
        switch (image.format()) {
        case QImage::Format_Mono:
        case QImage::Format_MonoLSB:
            image = img.hasAlphaChannel()
	        ? img.convertToFormat(QImage::Format_ARGB32_Premultiplied)
	        : img.convertToFormat(QImage::Format_RGB32);
        break;
        case QImage::Format_RGB32:
        case QImage::Format_ARGB32_Premultiplied:
        case QImage::Format_RGB16:
            image = img;
            break;
        default:
            image = img.convertToFormat(QImage::Format_ARGB32_Premultiplied);
            break;
        }
    }
#endif

    setSerialNumber(image.serialNumber());
}

void QRasterPixmapData::fill(const QColor &color)
{
    uint pixel;

    if (image.depth() == 1) {
        int gray = qGray(color.rgba());
        // Pick the best approximate color in the image's colortable.
        if (qAbs(qGray(image.color(0)) - gray) < qAbs(qGray(image.color(1)) - gray))
            pixel = 0;
        else
            pixel = 1;
    } else if (image.depth() == 32 || image.depth() == 16) {
        int alpha = color.alpha();
        if (alpha != 255) {
            if (image.format() == QImage::Format_RGB32
                || image.format() == QImage::Format_RGB16)
            {
                image = image.convertToFormat(QImage::Format_ARGB32_Premultiplied);
            }
            pixel = PREMUL(color.rgba());
        } else if (image.depth() == 16) {
            pixel = qt_colorConvert<quint16, quint32>(color.rgba(), 0);
        } else {
            pixel = color.rgba();
        }
    } else {
        pixel = 0;
        // ### what about 8 bits
    }

    image.fill(pixel);
}

QBitmap QRasterPixmapData::mask() const
{
    if (!image.hasAlphaChannel() || image.depth() != 32) {
        return QBitmap();
    }

    // Create image and setup color table
    int w = image.width();
    int h = image.height();
    QImage mask(w, h, QImage::Format_MonoLSB);
    mask.setNumColors(2);
    mask.setColor(0, QColor(Qt::color0).rgba());
    mask.setColor(1, QColor(Qt::color1).rgba());

    int bpl = mask.bytesPerLine();

    // copy over the data
    for (int y = 0; y < h; ++y) {
        QRgb *src = (QRgb *)image.scanLine(y);
        uchar *dest = mask.scanLine(y);
        memset(dest, 0, bpl);
        for (int x=0; x<w; ++x) {
            if (qAlpha(*src) > 0)
                dest[x>>3] |= qt_pixmap_bit_mask[x&7];
            ++src;
        }
    }

    return QBitmap::fromImage(mask); // hw: check this!
}

void QRasterPixmapData::setMask(const QBitmap &mask)
{
    if (mask.size().isEmpty()) {
        if (image.depth() != 1) { // hw: ????
            image = image.convertToFormat(QImage::Format_RGB32);
        }
    } else {
#ifdef Q_WS_QWS
        const QImage imageMask = mask.toImage().convertToFormat(QImage::Format_Mono);
#else
        const QImage imageMask = mask.toImage().convertToFormat(QImage::Format_MonoLSB);
#endif

        const int w = image.width();
        const int h = image.height();

        switch (image.depth()) {
        case 1:
            for (int y = 0; y < h; ++y) {
                const uchar *mscan = imageMask.scanLine(y);
                uchar *tscan = image.scanLine(y);
                int bytesPerLine = image.bytesPerLine();
                for (int i = 0; i < bytesPerLine; ++i)
                    tscan[i] &= mscan[i];
            }
            break;

        default:
            image = image.convertToFormat(QImage::Format_ARGB32_Premultiplied);
            for (int y = 0; y < h; ++y) {
                const uchar *mscan = imageMask.scanLine(y);
                QRgb *tscan = (QRgb *)image.scanLine(y);
                for (int x = 0; x < w; ++x) {
                    if (!(mscan[x>>3] & qt_pixmap_bit_mask[x&7]))
                        tscan[x] = 0;
                }
            }
            break;
        }
    }
}

bool QRasterPixmapData::hasAlphaChannel() const
{
    return image.hasAlphaChannel();
}

QImage QRasterPixmapData::toImage() const
{
    return image;
}

void QRasterPixmapData::setAlphaChannel(const QPixmap &alphaChannel)
{
    image.setAlphaChannel(alphaChannel.toImage());
}

QPixmap QRasterPixmapData::alphaChannel() const
{
    return QPixmap::fromImage(image.alphaChannel());
}

QPaintEngine* QRasterPixmapData::paintEngine() const
{
    return image.paintEngine();
}

extern int qt_defaultDpi();
extern int qt_defaultDpiX();
extern int qt_defaultDpiY();

int QRasterPixmapData::metric(QPaintDevice::PaintDeviceMetric metric) const
{
    // override the image dpi with the screen dpi when rendering to a pixmap
    // ### Qt 4.4: remove #ifdef
#ifdef Q_WS_QWS
    int dpmX = qRound(qt_defaultDpiX()*100./2.54);
    int dpmY = qRound(qt_defaultDpiY()*100./2.54);
#else
    int dpmX = qRound(qt_defaultDpi()*100./2.54);
    int dpmY = dpmX;
#endif
    switch (metric) {
    case QPaintDevice::PdmWidthMM:
        return qRound(image.width() * 1000 / dpmX);
        break;

    case QPaintDevice::PdmHeightMM:
        return qRound(image.height() * 1000 / dpmY);
        break;

    case QPaintDevice::PdmDpiX:
        return qRound(dpmX * 0.0254);
        break;

    case QPaintDevice::PdmDpiY:
        return qRound(dpmY * 0.0254);
        break;

    case QPaintDevice::PdmPhysicalDpiX:
        return qRound(dpmX * 0.0254);
        break;

    case QPaintDevice::PdmPhysicalDpiY:
        return qRound(dpmY * 0.0254);
        break;
    default:
        return image.metric(metric);
    }
}

QT_END_NAMESPACE
