/****************************************************************************
**
** Copyright (C) 1992-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QT_NO_GRAPHICSVIEW

#include "qgraphicslayout.h"
#include "qgraphicsscene.h"
#include "qgraphicslayoutitem.h"
#include "qgraphicslayoutitem_p.h"
#include "qwidget.h"

#include <QtDebug>

QT_BEGIN_NAMESPACE

/*
    COMBINE_SIZE() is identical to combineSize(), except that it
    doesn't evaluate 'size' unless necessary.
*/
#define COMBINE_SIZE(result, size) \
    do { \
        if ((result).width() < 0 || (result).height() < 0) \
            combineSize((result), (size)); \
    } while (false)

static void combineSize(QSizeF &result, const QSizeF &size)
{
    if (result.width() < 0)
        result.setWidth(size.width());
    if (result.height() < 0)
        result.setHeight(size.height());
}

static void boundSize(QSizeF &result, const QSizeF &size)
{
    if (size.width() >= 0 && size.width() < result.width())
        result.setWidth(size.width());
    if (size.height() >= 0 && size.height() < result.height())
        result.setHeight(size.height());
}

static void expandSize(QSizeF &result, const QSizeF &size)
{
    if (size.width() >= 0 && size.width() > result.width())
        result.setWidth(size.width());
    if (size.height() >= 0 && size.height() > result.height())
        result.setHeight(size.height());
}

static void normalizeHints(qreal &minimum, qreal &preferred, qreal &maximum, qreal &descent)
{
    if (minimum >= 0 && maximum >= 0 && minimum > maximum)
        minimum = maximum;

    if (preferred >= 0) {
        if (minimum >= 0 && preferred < minimum) {
            preferred = minimum;
        } else if (maximum >= 0 && preferred > maximum) {
            preferred = maximum;
        }
    }

    if (minimum >= 0 && descent > minimum)
        descent = minimum;
}

/*!
    \internal
*/
QGraphicsLayoutItemPrivate::QGraphicsLayoutItemPrivate(QGraphicsLayoutItem *par, bool layout)
    : parent(par), isLayout(layout)
{
}

/*!
    \internal
*/
void QGraphicsLayoutItemPrivate::init()
{
    sizeHintCacheDirty = true; 
    sizePolicy = QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
}

/*!
    \internal
*/
QSizeF *QGraphicsLayoutItemPrivate::effectiveSizeHints(const QSizeF &constraint) const
{
    Q_Q(const QGraphicsLayoutItem);
    if (!sizeHintCacheDirty && cachedConstraint == constraint)
        return cachedSizeHints;

    for (int i = 0; i < Qt::NSizeHints; ++i) {
        cachedSizeHints[i] = constraint;
        combineSize(cachedSizeHints[i], userSizeHints[i]);
    }

    QSizeF &minS = cachedSizeHints[Qt::MinimumSize];
    QSizeF &prefS = cachedSizeHints[Qt::PreferredSize];
    QSizeF &maxS = cachedSizeHints[Qt::MaximumSize];
    QSizeF &descentS = cachedSizeHints[Qt::MinimumDescent];

    normalizeHints(minS.rwidth(), prefS.rwidth(), maxS.rwidth(), descentS.rwidth());
    normalizeHints(minS.rheight(), prefS.rheight(), maxS.rheight(), descentS.rheight());

    COMBINE_SIZE(maxS, q->sizeHint(Qt::MaximumSize, maxS));
    combineSize(maxS, QSizeF(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX));
    expandSize(maxS, prefS);
    expandSize(maxS, minS);
    boundSize(maxS, QSizeF(QWIDGETSIZE_MAX, QWIDGETSIZE_MAX));

    COMBINE_SIZE(minS, q->sizeHint(Qt::MinimumSize, minS));
    expandSize(minS, QSizeF(0, 0));
    boundSize(minS, prefS);
    boundSize(minS, maxS);

    COMBINE_SIZE(prefS, q->sizeHint(Qt::PreferredSize, prefS));
    expandSize(prefS, minS);
    boundSize(prefS, maxS);

    COMBINE_SIZE(descentS, q->sizeHint(Qt::MinimumDescent, constraint));

    cachedConstraint = constraint;
    sizeHintCacheDirty = false;
    return cachedSizeHints;
}

/*!
    \class QGraphicsLayoutItem
    \brief The QGraphicsLayoutItem class can be inherited to make your custom items
    layout-aware.
    \since 4.4
    \ingroup multimedia

    QGraphicsLayoutItem is an abstract class that defines a set of virtual functions
    describing sizes, size policies, and size hints for any object arranged by
    QGraphicsLayout. The API contains functions that are relevant for both the
    item itself, and for the user of the item (i.e., most of QGraphicsLayoutItem's
    functions are also part of the subclass's public API).

    By creating a subclass of QGraphicsLayoutItem (e.g., directly or with multiple
    inheritance), and reimplementing its virtual functions, you will enable
    the layout to resize and position your item together with other
    QGraphicsLayoutItem items (including QGraphicsWidget and QGraphicsLayout
    itself). You can start by reimplementing the most important functions: the
    protected sizeHint() function, and the public geometry() and setGeometry()
    functions. If you want your items to become aware of immediate geometry
    changes, you can also reimplement updateGeometry().

    The geometry, size hint, and size policy affect the item's size and
    position. Calling setGeometry() will always resize and reposition the item
    immediately; this function is normally called by QGraphicsLayout after the
    layout has been activated, but can also be called by the item's user at
    any time. sizeHint() returns the item' minimum, preferred and maximum size
4    hints. You can override these properties by calling setMinimumSize(),
    setPreferredSize() or setMaximumSize(). The public effectiveSizeHint()
    function returns a size hint for any given Qt::SizeHint, and guarantees
    that the returned size is bound to the minimum and maximum sizes and size
    hints. You can set the item's vertical and horizontal size policy by
    calling setSizePolicy(); this property is used by the layout system and
    describes how this item prefers to grow or shrink.

    You can stack QGraphicsLayoutItem items inside others by either passing a
    QGraphicsLayoutItem pointer to QGraphicsLayoutItem's protected constructor, or by calling
    setParentLayoutItem(). This is how you "nest" layoutItem items, just like
    layouts that can contain other sublayouts. The parentLayoutItem() function
    returns a pointer to the item's layoutItem parent, or 0 if the item's
    parent is either 0, or if the item's parent does not inherit from
    QGraphicsLayoutItem. isLayout() returns true if the QGraphicsLayoutItem subclass is itself
    a layout, or false otherwise.

    Most of the time, you will not need to worry about this class; existing
    layout-aware classes such as QGraphicsWidget and QGraphicsLayout already
    provide the functionality you need, and by subclassing these classes
    instead, you can easily create both layout-aware graphical elements
    (QGraphicsWidget), or custom layouts (QGraphicsLayout).

    Qt uses QGraphicsLayoutItem for providing layout functionality in the
    \l{The Graphics View Framework}, but in the future its use may spread throughout
    Qt itself.

    \sa QGraphicsWidget, QGraphicsLayout, QGraphicsLinearLayout,
    QGraphicsGridLayout
*/

/*!
    Constructs the QGraphicsLayoutItem object. \a parent becomes the object's parent,
    and if \a isLayout is true the item is a layout, otherwise \a isLayout is
    false.
*/
QGraphicsLayoutItem::QGraphicsLayoutItem(QGraphicsLayoutItem *parent, bool isLayout)
    : d_ptr(new QGraphicsLayoutItemPrivate(parent, isLayout))
{
    Q_D(QGraphicsLayoutItem);
    d->init();
    d->q_ptr = this;
}

/*!
    \internal
*/
QGraphicsLayoutItem::QGraphicsLayoutItem(QGraphicsLayoutItemPrivate &dd)
    : d_ptr(&dd)
{
    Q_D(QGraphicsLayoutItem);
    d->q_ptr = this;
}

/*!
    Destroys the QGraphicsLayoutItem object.
*/
QGraphicsLayoutItem::~QGraphicsLayoutItem()
{
    delete d_ptr;
}

/*!
    \fn virtual QSizeF QGraphicsLayoutItem::sizeHint(Qt::SizeHint which, const QSizeF &constraint) const = 0;

    This pure virtual function returns the size hint for \a which of the
    QGraphicsLayoutItem, using the width or height of \a constraint to constrain the
    output.

    You must reimplement this function in a subclass of QGraphicsLayoutItem to provide
    the necessary size hints for your items.

    \sa effectiveSizeHint()
*/

/*!
    Sets the size policy to \a policy. The size policy describes how the item
    should grow horizontally and vertically when arranged in a layout.

    QGraphicsLayoutItem's default size policy is (QSizePolicy::Fixed,
    QSizePolicy::Fixed, QSizePolicy::DefaultType), but it's common for
    subclasses to change the default. For example, QGraphicsWidget defaults to
    (QSizePolicy::Preferred, QSizePolicy::Preferred,
    QSizePolicy::DefaultType).

    \sa sizePolicy(), QWidget::sizePolicy()
*/
void QGraphicsLayoutItem::setSizePolicy(const QSizePolicy &policy)
{
    Q_D(QGraphicsLayoutItem);
    if (d->sizePolicy == policy)
        return;
    d->sizePolicy = policy;
    updateGeometry();
}

/*!
    \overload

    Equivalent to calling setSizePolicy(QSizePolicy(\a hPolicy, \a vPolicy, \a controlType)).

    \sa sizePolicy(), QWidget::sizePolicy()
*/
void QGraphicsLayoutItem::setSizePolicy(QSizePolicy::Policy hPolicy,
                                        QSizePolicy::Policy vPolicy,
                                        QSizePolicy::ControlType controlType)
{
    setSizePolicy(QSizePolicy(hPolicy, vPolicy, controlType));
}

/*!
    Returns the current size policy. 

    \sa setSizePolicy(), QWidget::sizePolicy()
*/
QSizePolicy QGraphicsLayoutItem::sizePolicy() const
{
    Q_D(const QGraphicsLayoutItem);
    return d->sizePolicy;
}

/*!
    Sets the minimum size to \a size. This property overrides sizeHint() for
    Qt::MinimumSize and ensures that effectiveSizeHint() will never return a
    size smaller than \a size.

    \sa minimumSize(), maximumSize(), preferredSize(), Qt::MinimumSize,
    sizeHint()
*/
void QGraphicsLayoutItem::setMinimumSize(const QSizeF &size)
{
    Q_D(QGraphicsLayoutItem);
    if (!size.isValid()) {
        qWarning("QGraphicsLayoutItem::setMinimumSize: invalid size (%g, %g)",
                 size.width(), size.height());
        return;
    }
    if (size == d->userSizeHints[Qt::MinimumSize])
        return;

    d->userSizeHints[Qt::MinimumSize] = size;
    QRectF geom = geometry();
    if (d->userSizeHints[Qt::MinimumSize].width() > geom.width()
        || d->userSizeHints[Qt::MinimumSize].height() > geom.height()) {
        geom.setSize(d->userSizeHints[Qt::MinimumSize].expandedTo(geom.size()));
        setGeometry(geom);
    }
    updateGeometry();
}

/*!
    \fn QGraphicsLayoutItem::setMinimumSize(qreal w, qreal h)

    This convenience function is equivalent to calling setMinimumSize(QSizeF(\a w, \a h)).

    \sa minimumSize(), setMaximumSize(), setPreferredSize(), sizeHint()
*/

/*!
    Returns the minimum size.

    \sa setMinimumSize(), preferredSize(), maximumSize(), Qt::MinimumSize,
    sizeHint()
*/
QSizeF QGraphicsLayoutItem::minimumSize() const
{
    return effectiveSizeHint(Qt::MinimumSize);
}

/*!
    Sets the preferred size to \a size. This property overrides sizeHint() for
    Qt::PreferredSize and provides the default value for effectiveSizeHint().

    \sa preferredSize(), minimumSize(), maximumSize(), Qt::PreferredSize,
    sizeHint()
*/
void QGraphicsLayoutItem::setPreferredSize(const QSizeF &size)
{
    Q_D(QGraphicsLayoutItem);
    if (!size.isValid()) {
        qWarning("QGraphicsLayoutItem::setPreferredSize: invalid size (%g, %g)",
                 size.width(), size.height());
        return;
    }
    if (size == d->userSizeHints[Qt::PreferredSize])
        return;

    d->userSizeHints[Qt::PreferredSize] = size;
    updateGeometry();
}

/*!
    \fn QGraphicsLayoutItem::setPreferredSize(qreal w, qreal h)

    This convenience function is equivalent to calling setPreferredSize(QSizeF(\a w, \a h)).

    \sa preferredSize(), setMaximumSize(), setMinimumSize(), sizeHint()
*/

/*!
    Returns the preferred size.

    \sa setPreferredSize(), minimumSize(), maximumSize(), Qt::PreferredSize,
    sizeHint()
*/
QSizeF QGraphicsLayoutItem::preferredSize() const
{
    return effectiveSizeHint(Qt::PreferredSize);
}

/*!
    Sets the maximum size to \a size. This property overrides sizeHint() for
    Qt::MaximumSize and ensures that effectiveSizeHint() will never return a
    size larger than \a size.

    \sa maximumSize(), minimumSize(), preferredSize(), Qt::MaximumSize,
    sizeHint()
*/
void QGraphicsLayoutItem::setMaximumSize(const QSizeF &size)
{
    Q_D(QGraphicsLayoutItem);
    if (!size.isValid()) {
        qWarning("QGraphicsLayoutItem::setMaximumSize: invalid size (%g, %g)",
                 size.width(), size.height());
        return;
    }
    if (size == d->userSizeHints[Qt::MaximumSize])
        return;

    d->userSizeHints[Qt::MaximumSize] = size;
    QRectF geom = geometry();
    if (d->userSizeHints[Qt::MaximumSize].width() < geom.width()
        || d->userSizeHints[Qt::MaximumSize].height() < geom.width()) {
        geom.setSize(d->userSizeHints[Qt::MaximumSize].boundedTo(geom.size()));
        setGeometry(geom);
    }
    updateGeometry();
}

/*!
    \fn QGraphicsLayoutItem::setMaximumSize(qreal w, qreal h)

    This convenience function is equivalent to calling setMaximumSize(QSizeF(\a w, \a h)).

    \sa maximumSize(), setMinimumSize(), setPreferredSize(), sizeHint()
*/

/*!
    Returns the maximum size.

    \sa setMaximumSize(), minimumSize(), preferredSize(), Qt::MaximumSize,
    sizeHint()
*/
QSizeF QGraphicsLayoutItem::maximumSize() const
{
    return effectiveSizeHint(Qt::MaximumSize);
}

/*!
    \fn virtual void QGraphicsLayoutItem::setGeometry(const QRectF &rect) = 0

    This pure virtual function sets the geometry of the QGraphicsLayoutItem to \a
    rect, which is in parent coordinates (i.e., the top-left corner of \a rect
    is equivalent to the item's position in parent coordinates).

    You must reimplement this function in a subclass of QGraphicsLayoutItem for your
    item to receive geometry updates.
    
    \sa geometry()
*/

/*!
    \fn virtual QRectF QGraphicsLayoutItem::geometry() const = 0

    This pure virtual function returns the geometry of the QGraphicsLayoutItem as a
    QRectF in local coordinates.

    You must reimplement this function in a subclass of QGraphicsLayoutItem for layout
    managers to determine the geometry of your item.

    \sa setGeometry()
*/

/*!
    This virtual function provides the \a left, \a top, \a right and \a bottom
    contents margins for this QGraphicsLayoutItem. The default implementation assumes
    all contents margins are 0. The values are stored in the qreals pointed to
    by the input arguments. If any of the pointers is 0, that value will not
    be updated.

    \sa QGraphicsWidget::setContentsMargins()
*/
void QGraphicsLayoutItem::getContentsMargins(qreal *left, qreal *top, qreal *right, qreal *bottom) const
{
    if (left)
        *left = 0;
    if (top)
        *top = 0;
    if (right)
        *right = 0;
    if (bottom)
        *bottom = 0;
}

/*!
    Returns the contents rect in local coordinates.

    The contents rect defines the subrectangle used by an associated layout
    when arranging subitems. This function is a convenience function that
    adjusts the item's geometry() by its contents margins. Note that
    getContentsMargins() is a virtual function that you can reimplement to
    return the item's contents margins.

    \sa getContentsMargins(), geometry()
*/
QRectF QGraphicsLayoutItem::contentsRect() const
{
    qreal left, top, right, bottom;
    getContentsMargins(&left, &top, &right, &bottom);
    return QRectF(QPointF(), geometry().size()).adjusted(+left, +top, -right, -bottom);
}

/*!
    Returns the effective size hint for this QGraphicsLayoutItem. \a which is the size
    hint in question. \a constraint is an optional argument that defines a
    special constrain when calculating the effective size hint. By default, \a
    constraint is QSizeF(-1, -1), which means there is no constraint to the
    size hint.

    If you want the widget's size hint for a given width or height, you can
    provide the fixated dimension in \a constraint. This is useful for widgets
    that can grow only either vertically or horizontally, and need to fix
    either their width or their height to a special value (e.g., if a text
    paragraph item must fit into a column width of 200 may grow vertically,
    you can pass QSizeF(200, -1) as a constraint to get a suitable minimum,
    preferred and maximum height).

    You can affect the effective size hint by either reimplementing sizeHint()
    in a QGraphicsLayoutItem subclass, or by calling one of setMinimumSize(),
    setPreferredSize() or setMaximumSize() (or a combination of both).
    
    This function caches each of the size hints, and guarantees that unless
    updateGeometry() has been called, and as long as \a constraint is unused,
    sizeHint() will only be called once for each value of \a which.

    \sa sizeHint()
*/
QSizeF QGraphicsLayoutItem::effectiveSizeHint(Qt::SizeHint which, const QSizeF &constraint) const
{
    // ### should respect size policy???
    return d_ptr->effectiveSizeHints(constraint)[which];
}

/*!
    This virtual function discards any cached size hint information. You
    should always call this function if you change the return value of the
    sizeHint() function. Subclasses must always call the base implementation
    when reimplementing this function.

    \sa effectiveSizeHint()
*/
void QGraphicsLayoutItem::updateGeometry()
{ 
    Q_D(QGraphicsLayoutItem);
    d->sizeHintCacheDirty = true;
}

/*!
    Returns the parent of this QGraphicsLayoutItem, or 0 if either there is no parent,
    or if the parent does not inherit from QGraphicsLayoutItem (QGraphicsLayoutItem is often
    used by multiple inheritance with QObject-derived classes).

    \sa setParentLayoutItem()
*/
QGraphicsLayoutItem *QGraphicsLayoutItem::parentLayoutItem() const
{
    return d_func()->parent;
}

/*!
    Sets the parent of this QGraphicsLayoutItem to \a parent.

    \sa parentLayoutItem()
*/
void QGraphicsLayoutItem::setParentLayoutItem(QGraphicsLayoutItem *parent)
{ 
    d_func()->parent = parent;
}

/*!
    Returns true if this QGraphicsLayoutItem is a layout (i.e., is inherited by an
    object that arranges other QGraphicsLayoutItem objects); otherwise returns false.

    \sa QGraphicsLayout
*/
bool QGraphicsLayoutItem::isLayout() const
{
    return d_func()->isLayout;
}

QT_END_NAMESPACE
        
#endif //QT_NO_GRAPHICSVIEW
