/****************************************************************************
**
** Copyright (C) 2005-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

/*!
    \headerfile <QtConcurrentMap>
    \title Concurrent Map and Map-Reduce

    \brief The <QtConcurrentMap> header provides concurrent Map and MapReduce.

    The QtConcurrent::map(), QtConcurrent::mapped() and
    QtConcurrent::mappedReduced() functions run computations in parallel on
    the items in a sequence such as a QList or a QVector. QtConcurrent::map()
    modifies a sequence in-place, QtConcurrent::mapped() returns a new
    sequence containing the modified content, and QtConcurrent::mappedReduced()
    returns a single result.

    \section1 Concurrent Map

    QtConcurrent::mapped() takes an input sequence and a map function. This map
    function is then called for each item in the sequence, and a new sequence
    containing the return values from the map function is returned.

    The map function must be of the form:

    \code
    U function(const T &t);
    \endcode

    T and U can be any type (and they can even be the same type), but T must
    match the type stored in the sequence. The function returns the modified
    or \e mapped content.

    This example shows how to apply a scale function to all the items
    in a sequence:

    \code
    QImage scaled(const QImage &image)
    {
        return image.scaled(100, 100);
    }

    QList<QImage> images = ...;
    QFuture<QImage> thumbnails = QtConcurrent::mapped(images, scaled);
    \endcode

    The results of the map are made available through QFuture.  See the
    QFuture and QFutureWatcher documentation for more information on how to
    use QFuture in your applications.

    If you want to modify a sequence in-place, use QtConcurrent::map(). The
    map function must then be of the form:

    \code
    U function(T &t);
    \endcode

    Note that the return value and return type of the map function are not
    used.

    Using QtConcurrent::map() is similar to using QtConcurrent::mapped():

    \code
    void scale(QImage &image)
    {
        image = image.scaled(100, 100);
    }

    QList<QImage> images = ...;
    QFuture<void> future = QtConcurrent::map(images, scale);
    \endcode

    Since the sequence is modified in place, QtConcurrent::map() does not
    return any results via QFuture. However, you can still use QFuture and
    QFutureWatcher to monitor the status of the map.

    \section1 Concurrent Map-Reduce

    QtConcurrent::mappedReduced() is similar to QtConcurrent::mapped(), but
    instead of returning a sequence with the new results, the results are
    combined into a single value using a reduce function.

    The reduce function must be of the form:

    \code
    V function(T &result, const U &intermediate)
    \endcode

    T is the type of the final result, U is the return type of the map
    function. Note that the return value and return type of the reduce
    function are not used.

    Call QtConcurrent::mappedReduced() like this:

    \code
    void addToCollage(QImage &collage, const QImage &thumbnail)
    {
        QPainter p(&collage);
        static QPoint offset = QPoint(0, 0);
        p.drawImage(offset, thumbnail);
        offset += ...;
    }

    QList<QImage> images = ...;
    QFuture<QImage> collage = QtConcurrent::mappedReduced(images, scaled, addToCollage);
    \endcode

    The reduce function will be called once for each result returned by the map
    function, and should merge the \e{intermediate} into the \e{result}
    variable.  QtConcurrent::mappedReduced() guarantees that only one thread
    will call reduce at a time, so using a mutex to lock the result variable
    is not neccesary. The QtConcurrent::ReduceOptions enum provides a way to
    control the order in which the reduction is done. If
    QtConcurrent::UnorderedReduce is used (the default), the order is
    undefined, while QtConcurrent::OrderedReduce ensures that the reduction
    is done in the order of the original sequence.

    \section1 Additional API Features

    \section2 Using Iterators instead of Sequence

    Each of the above functions has a variant that takes an iterator range
    instead of a sequence. You use them in the same way as the sequence
    variants:

    \code
    QList<QImage> images = ...;

    QFuture<QImage> thumbnails = QtConcurrent::mapped(images.constBegin(), images.constEnd(), scaled);

    // map in-place only works on non-const iterators
    QFuture<void> future = QtConcurrent::map(images.begin(), images.end(), scale);

    QFuture<QImage> collage = QtConcurrent::mappedReduced(images.constBegin(), images.constEnd(), scaled, addToCollage);
    \endcode

    \section2 Blocking Variants

    Each of the above functions has a blocking variant that returns
    the final result instead of a QFuture. You use them in the same
    way as the asynchronous variants.

    \code
    QList<QImage> images = ...;

    // each call blocks until the entire operation is finished
    QList<QImage> future = QtConcurrent::blockingMapped(images, scaled);

    QtConcurrent::blockingMap(images, scale);

    QImage collage = QtConcurrent::blockingMappedReduced(images, scaled, addToCollage);
    \endcode

    Note that the result types above are not QFuture objects, but real result
    types (in this case, QList<QImage> and QImage).

    \section2 Using Member Functions

    QtConcurrent::map(), QtConcurrent::mapped(), and
    QtConcurrent::mappedReduced() also accept pointers to member functions.
    The member function class type must match the type stored in the sequence:

    \code
    // squeeze all strings in a QStringList
    QStringList strings = ...;
    QFuture<void> = QtConcurrent::map(strings, &QString::squeeze());

    // swap the rgb values of all pixels on a list of images
    QList<QImage> images = ...;
    QFuture<QImage> bgrImages = QtConcurrent::mapped(images, &QImage::rgbSwapped());

    // create a set of the lengths of all strings in a list
    QStringList strings = ...;
    QFuture<QSet<int> > wordLengths = QtConcurrent::mappedReduced(string, &QString::length(), &QSet<int>::insert);
    \endcode

    Note that when using QtConcurrent::mappedReduced(), you can mix the use of
    normal and member functions freely:

    \code
    // can mix normal functions and member functions with QtConcurrent::mappedReduced()

    // compute the average length of a list of strings
    extern void computeAverage(int &average, int length);
    QStringList strings = ...;
    QFuture<int> averageWordLength = QtConcurrent::mappedReduced(strings, &QString::length(), computeAverage);

    // create a set of the color distribution of all images in a list
    extern int colorDistribution(const QImage &string);
    QList<QImage> images = ...;
    QFuture<QSet<int> > totalColorDistribution = QtConcurrent::mappedReduced(images, colorDistribution, QSet<int>::insert);
    \endcode

    \section2 Using Bound Function Arguments

    Note that Qt does not provide support for bound functions. This is
    provided by 3rd party libraries like
    \l{http://www.boost.org/libs/bind/bind.html}{Boost} or
    \l{http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2005/n1836.pdf}{C++
    TR1 Library Extensions}.

    If you want to use a map function that takes more than one argument you can
    use boost::bind() or std::tr1::bind() to transform it onto a function that
    takes one argument.

    As an example, we'll use QImage::scaledToWidth():

    \code
    QImage QImage::scaledToWidth(int width, Qt::TransformationMode) const;
    \endcode

    scaledToWidth takes three arguments (including the "this" pointer) and
    can't be used with QtConcurrent::mapped() directly, because
    QtConcurrent::mapped() expects a function that takes one argument. To use
    QImage::scaledToWidth() with QtConcurrent::mapped() we have to provide a
    value for the \e{width} and the \e{transformation mode}:

    \code
    boost::bind(&QImage::scaledToWidth, 100 Qt::SmoothTransformation)
    \endcode

    The return value from boost::bind() is a function object (functor) with
    the following signature:

    \code
    QImage scaledToWith(const QImage &image)
    \endcode

    This matches what QtConcurrent::mapped() expects, and the complete example
    becomes:

    \code
    QList<QImage> images = ...;
    QFuture<QImage> thumbnails = QtConcurrent::mapped(images, boost::bind(&QImage::scaledToWidth, 100 Qt::SmoothTransformation));
    \endcode
*/

/*!
    \fn QFuture<void> QtConcurrent::map(Sequence &sequence, MapFunction function)
    \relates <QtConcurrentMap>

    Calls \a function once for each item in \a sequence. The \a function is
    passed a reference to the item, so that any modifications done to the item
    will appear in \a sequence.
*/

/*!
    \fn QFuture<void> QtConcurrent::map(Iterator begin, Iterator end, MapFunction function)
    \relates <QtConcurrentMap>

    Calls \a function once for each item from \a begin to \a end. The
    \a function is passed a reference to the item, so that any modifications
    done to the item will appear in the sequence which the iterators belong to.
*/

/*!
    \fn QFuture<T> QtConcurrent::mapped(const Sequence &sequence, MapFunction function)
    \relates <QtConcurrentMap>

    Calls \a function once for each item in \a sequence and returns a future
    with each mapped item as a result. You can use QFuture::const_iterator or
    QFutureIterator to iterate through the results.
*/

/*!
    \fn QFuture<T> QtConcurrent::mapped(ConstIterator begin, ConstIterator end, MapFunction function)
    \relates <QtConcurrentMap>

    Calls \a function once for each item from \a begin to \a end and returns a
    future with each mapped item as a result. You can use
    QFuture::const_iterator or QFutureIterator to iterate through the results.
*/

/*!
    \fn QFuture<T> QtConcurrent::mappedReduced(const Sequence &sequence,
    MapFunction mapFunction, ReduceFunction reduceFunction,
    QtConcurrent::ReduceOptions reduceOptions)

    \relates <QtConcurrentMap>

    Calls \a mapFunction once for each item in \a sequence. The return value of
    each \a mapFunction is passed to \a reduceFunction.

    Note that while \a mapFunction is called concurrently, only one thread at a
    time will call \a reduceFunction. The order in which \a reduceFunction is
    called is determined by \a reduceOptions.
*/

/*!
    \fn QFuture<T> QtConcurrent::mappedReduced(ConstIterator begin,
    ConstIterator end, MapFunction mapFunction, ReduceFunction reduceFunction,
    QtConcurrent::ReduceOptions reduceOptions)

    \relates <QtConcurrentMap>

    Calls \a mapFunction once for each item from \a begin to \a end. The return
    value of each \a mapFunction is passed to \a reduceFunction.

    Note that while \a mapFunction is called concurrently, only one thread at a
    time will call \a reduceFunction. The order in which \a reduceFunction is
    called is undefined.
*/
