#include <QtGui>

#define SLIDER_RANGE 5

#include <phonon/backendcapabilities.h>

#include <qdebug.h>
#include "mediaplayer.h"
#include "ui_settings.h"

MediaPlayer::MediaPlayer(const QString &filePath) :
        pauseButton(0), playButton(0), m_AudioOutput(Phonon::VideoCategory)
{
    setWindowTitle("Media Player");
    setAttribute(Qt::WA_MacBrushedMetal);

    setContextMenuPolicy(Qt::CustomContextMenu);
    
    QSize buttonSize(34, 28);

    QPushButton *openButton = new QPushButton(this);
    openButton->setMinimumSize(buttonSize);
    openButton->setIcon(style()->standardIcon(QStyle::SP_DialogOpenButton));

    rewindButton = new QPushButton(this);
    rewindButton->setMinimumSize(buttonSize);
    rewindButton->setIcon(QPixmap(":/images/rewind.png"));

    playButton = new QPushButton(this);
    playButton->setMinimumSize(buttonSize);
    playButton->setIcon(QPixmap(":/images/play.png"));

    pauseButton = new QPushButton(this);
    pauseButton->setMinimumSize(buttonSize);
    pauseButton->setIcon(QPixmap(":/images/pause.png"));

    slider = new Phonon::SeekSlider(this);
    slider->setMediaObject(&m_MediaObject);
    volume = new QSlider(Qt::Horizontal, this);

    QVBoxLayout *vLayout = new QVBoxLayout(this);
    QHBoxLayout *layout = new QHBoxLayout();

    info = new QTextEdit(this);
    info->setMaximumHeight(60);
    info->setReadOnly(true);
    info->setAcceptDrops(false);
    info->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    info->setTextInteractionFlags(Qt::NoTextInteraction);

#ifdef Q_WS_MAC
    QLinearGradient bgBrush(QPointF(0, 0), QPointF(0, 50));
    bgBrush.setColorAt(0, qRgb(40, 50, 60));
    bgBrush.setColorAt(1, qRgb(120, 130, 140));
    QPalette palette;
    palette.setBrush(QPalette::Base, bgBrush);
    info->setPalette(palette);
#else
    info->setStyleSheet("background-color:qlinearGradient(x1:0, y1:0, x2:0, y2:1, stop:0 #335577, "
                        "stop:1 #6688AA); color: #eeeeff");
#endif

    info->setMinimumWidth(300);
    volume->setRange(0, 100);
    volume->setValue(100);
    volume->setMinimumWidth(40);

    layout->addWidget(openButton);
    layout->addWidget(rewindButton);
    layout->addWidget(playButton);
    layout->addWidget(pauseButton);

    QLabel *volumeLabel = new QLabel(this);
    volumeLabel->setPixmap(QPixmap(":/images/volume.png"));
    layout->addWidget(volumeLabel);
    layout->addWidget(volume);

    vLayout->addWidget(info);
    vLayout->addLayout(layout);
    vLayout->addWidget(slider);

    QHBoxLayout *labelLayout = new QHBoxLayout();

    vLayout->addLayout(labelLayout);
    setLayout(vLayout);

    // Create menu bar:
    QMenuBar *menubar = new QMenuBar();
    fileMenu = menubar->addMenu(tr("&File"));
    QAction *settingsAction = fileMenu->addAction(tr("&Settings"));
 
    // Setup signal connections:
    connect(rewindButton, SIGNAL(clicked()), this, SLOT(rewind()));
    connect(openButton, SIGNAL(clicked()), this, SLOT(openFile()));
    connect(pauseButton, SIGNAL(clicked()), this, SLOT(pause()));
    connect(playButton, SIGNAL(clicked()), this, SLOT(play()));
    connect(volume, SIGNAL(valueChanged(int)), this, SLOT(setVolume(int)));
    connect(settingsAction, SIGNAL(triggered(bool)), this, SLOT(showSettingsDialog()));

    connect(this, SIGNAL(customContextMenuRequested(const QPoint &)), SLOT(showContextMenu(const QPoint &)));

    connect(&m_MediaObject, SIGNAL(metaDataChanged()), this, SLOT(updateInfo()));
    connect(&m_MediaObject, SIGNAL(totalTimeChanged(qint64)), this, SLOT(updateInfo()));
    connect(&m_MediaObject, SIGNAL(hasVideoChanged(bool)), this, SLOT(handleVideoChanged(bool)));
    connect(&m_MediaObject, SIGNAL(tick(qint64)), this, SLOT(updateInfo()));
    connect(&m_MediaObject, SIGNAL(finished()), this, SLOT(finished()));
    connect(&m_MediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)), this, SLOT(stateChanged(Phonon::State, Phonon::State)));

    rewindButton->setEnabled(false);
    pauseButton->setEnabled(false);
    playButton->setEnabled(false);
    setAcceptDrops(true);

    setFixedSize(sizeHint());
    initVideoWindow();

    m_MediaObject.setTickInterval(50);
    Phonon::createPath(&m_MediaObject, &m_AudioOutput);
    Phonon::createPath(&m_MediaObject, &m_videoWidget);
    if (!filePath.isEmpty())
        setFile(filePath);
}

void MediaPlayer::stateChanged(Phonon::State newstate, Phonon::State oldstate)
{
    Q_UNUSED(oldstate);

    switch (newstate) {
        case Phonon::ErrorState:
            QMessageBox::warning(this, "Phonon Mediaplayer", "Error : " + m_MediaObject.errorString(), QMessageBox::Close);
            pauseButton->setEnabled(false);
            playButton->setEnabled(false);
            rewindButton->setEnabled(false);
            break;
        case Phonon::PausedState:
        case Phonon::StoppedState:
            pauseButton->setEnabled(false);
            playButton->setEnabled(true);
            rewindButton->setEnabled(true);
            break;
        case Phonon::PlayingState:
        case Phonon::BufferingState:
            pauseButton->setEnabled(true);
            playButton->setEnabled(false);
            rewindButton->setEnabled(true);
            break;
        case Phonon::LoadingState:
            pauseButton->setEnabled(true);
            playButton->setEnabled(true);
            rewindButton->setEnabled(false);
            break;
    }

}

void MediaPlayer::showSettingsDialog()
{
    Ui_Dialog ui;
    QDialog dialog;
    ui.setupUi(&dialog);
    
    ui.crossFadeSlider->setValue(2 * m_MediaObject.transitionTime() / 1000.0f);
    
    QList<Phonon::AudioOutputDevice> devices = Phonon::BackendCapabilities::availableAudioOutputDevices();
    for (int i=0; i<devices.size(); i++){
        ui.deviceCombo->addItem(devices[i].name());
        if (devices[i] == m_AudioOutput.outputDevice())
            ui.deviceCombo->setCurrentIndex(i);
    }
    
    dialog.exec();
    
    if (dialog.result() == QDialog::Accepted){
        m_MediaObject.setTransitionTime(1000 * float(ui.crossFadeSlider->value()) / 2.0f);
        m_AudioOutput.setOutputDevice(devices[ui.deviceCombo->currentIndex()]);
    }
    
}

void MediaPlayer::initVideoWindow()
{
    QVBoxLayout *videoLayout = new QVBoxLayout();
    QHBoxLayout *sliderLayout = new QHBoxLayout();

    QSlider *brightnessSlider = new QSlider(Qt::Horizontal);
    brightnessSlider->setRange(-SLIDER_RANGE, SLIDER_RANGE);
    brightnessSlider->setValue(0);
    connect(brightnessSlider, SIGNAL(valueChanged(int)), this, SLOT(setBrightness(int)));

    QSlider *hueSlider = new QSlider(Qt::Horizontal);
    hueSlider->setRange(-SLIDER_RANGE, SLIDER_RANGE);
    hueSlider->setValue(0);
    connect(hueSlider, SIGNAL(valueChanged(int)), this, SLOT(setHue(int)));

    QSlider *saturationSlider = new QSlider(Qt::Horizontal);
    saturationSlider->setRange(-SLIDER_RANGE, SLIDER_RANGE);
    saturationSlider->setValue(0);
    connect(saturationSlider, SIGNAL(valueChanged(int)), this, SLOT(setSaturation(int)));

    QSlider *contrastSlider = new QSlider(Qt::Horizontal);
    contrastSlider->setRange(-SLIDER_RANGE, SLIDER_RANGE);
    contrastSlider->setValue(0);
    connect(contrastSlider , SIGNAL(valueChanged(int)), this, SLOT(setContrast(int)));

    sliderLayout->addWidget(new QLabel("bright"));
    sliderLayout->addWidget(brightnessSlider);
    sliderLayout->addWidget(new QLabel("col"));
    sliderLayout->addWidget(hueSlider);
    sliderLayout->addWidget(new QLabel("sat"));
    sliderLayout->addWidget(saturationSlider);
    sliderLayout->addWidget(new QLabel("cont"));
    sliderLayout->addWidget(contrastSlider);

    videoLayout->addWidget(&m_videoWidget);
    videoLayout->addLayout(sliderLayout);
    m_videoWindow.setLayout(videoLayout);
    m_videoWindow.setWindowTitle("Video");
    m_videoWindow.setAttribute(Qt::WA_QuitOnClose, false);
    m_videoWindow.setAttribute(Qt::WA_MacBrushedMetal);
    m_videoWindow.setMinimumSize(100, 100);

}

void MediaPlayer::handleVideoChanged(bool hasVideo)
{
    if (hasVideo){
        QDesktopWidget desktop;
        QRect videoHintRect = QRect(QPoint(0, 0), m_videoWindow.sizeHint());
        QRect newVideoRect = desktop.screenGeometry().intersected(videoHintRect);
        m_videoWindow.resize(newVideoRect.size());
    }
    m_videoWindow.setVisible(hasVideo);
}

void MediaPlayer::pause()
{
    m_MediaObject.pause();
}

void MediaPlayer::dropEvent(QDropEvent *e)
{
    if (e->mimeData()->hasUrls())
        e->acceptProposedAction();

    QList<QUrl> urls = e->mimeData()->urls();

    if (e->keyboardModifiers() & Qt::ShiftModifier){
        // Just add to the que:
        for (int i=0; i<urls.size(); i++)
            m_MediaObject.enqueue(Phonon::MediaSource(urls[i].toLocalFile()));
    } else {
        // Create new que:
        m_MediaObject.clearQueue();
        QString fileName = urls[0].toLocalFile();
        setFile(fileName);
        for (int i=1; i<urls.size(); i++)
            m_MediaObject.enqueue(Phonon::MediaSource(urls[i].toLocalFile()));
    }
    play();
}

void MediaPlayer::dragEnterEvent(QDragEnterEvent *e)
{
    if (e->mimeData()->hasUrls())
        e->acceptProposedAction();
}

void MediaPlayer::play()
{
    m_MediaObject.play();
}

void MediaPlayer::setVolume(int volume)
{
    Q_UNUSED(volume);
    m_AudioOutput.setVolume(volume/100.0f);
}

void MediaPlayer::setFile(const QString &fileName)
{
    if (fileName.contains("://"))
        m_MediaObject.setCurrentSource(Phonon::MediaSource(QUrl(fileName)));
    else
        m_MediaObject.setCurrentSource(Phonon::MediaSource(fileName));
}

void MediaPlayer::openFile()
{
    QString fileName = QFileDialog::getOpenFileName();
    if (!fileName.isEmpty())
        setFile(fileName);
}

void MediaPlayer::setSaturation(int val)
{
    m_videoWidget.setSaturation(val / float(SLIDER_RANGE));
}

void MediaPlayer::setHue(int val)
{
    m_videoWidget.setHue(val / float(SLIDER_RANGE));
}

void MediaPlayer::setBrightness(int val)
{
    m_videoWidget.setBrightness(val / float(SLIDER_RANGE));
}

void MediaPlayer::setContrast(int val)
{
    m_videoWidget.setContrast(val / float(SLIDER_RANGE));
}

void MediaPlayer::updateInfo()
{
    long len = m_MediaObject.totalTime();
    long pos = m_MediaObject.currentTime();

    QString font = "<font color=#ffffd0>";
    QString fontmono = "<font family=\"monospace,courier new\" color=#ffffd0>";

    QMap <QString, QString> metaData = m_MediaObject.metaData();
    QString trackArtist = metaData.value("ARTIST");
    QString trackTitle = metaData.value("TITLE");
    QString timeString, time;
    if (pos || len)
    {
        int sec = pos/1000;
        int min = sec/60;
        int hour = min/60;
        int msec = pos;

        QTime playTime(hour%60, min%60, sec%60, msec%1000);
        sec = len / 1000;
        min = sec / 60;
        hour = min / 60;
        msec = len;

        QTime stopTime(hour%60, min%60, sec%60, msec%1000);
        timeString = playTime.toString("hh:mm:ss:zzz") + "</font>";
        if (len)
            timeString += "&nbsp; Duration: " + fontmono + stopTime.toString("hh:mm:ss:zzz") + "</font>";
        time =   "Time: " + font + timeString + "</font>";
    }

    QString fileName = m_MediaObject.currentSource().fileName();
    fileName = fileName.right(fileName.length() - fileName.lastIndexOf('/') - 1);
    QString title = "File: " + font + fileName + "<br></font>";
    if (!trackTitle.isEmpty())
        title = "Title: " + font + trackTitle + "<br></font>";
    QString artist;
    if (!trackArtist.isEmpty())
        artist = "Artist:  " + font + trackArtist + "<br></font>";
    info->setHtml(title + artist + time);

}

void MediaPlayer::rewind()
{
    m_MediaObject.setTickInterval(50);
    m_MediaObject.seek(0);
    updateInfo();
}

void MediaPlayer::finished()
{
    updateInfo();
}

void MediaPlayer::showContextMenu(const QPoint &p)
{
    fileMenu->popup(mapToGlobal(p));
}

