/****************************************************************************
**
** Copyright (C) 2007-2007 Trolltech ASA. All rights reserved.
**
** This file is part of the demonstration applications of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.0, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** In addition, as a special exception, Trolltech, as the sole copyright
** holder for Qt Designer, grants users of the Qt/Eclipse Integration
** plug-in the right for the Qt/Eclipse Integration to link to
** functionality provided by Qt Designer and its related libraries.
**
** Trolltech reserves all rights not expressly granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "browsermainwindow.h"

#include "browsertab.h"
#include "browsertabproxy.h"
#include "searchwidget.h"
#include "sourcecodeview.h"

#include <QtGui/QApplication>
#include <QtGui/QFileDialog>
#include <QtGui/QLabel>
#include <QtGui/QLineEdit>
#include <QtGui/QMenu>
#include <QtGui/QMenuBar>
#include <QtGui/QMessageBox>
#include <QtGui/QMovie>
#include <QtGui/QStyle>
#include <QtGui/QStatusBar>
#include <QtGui/QShortcut>
#include <QtGui/QTabWidget>
#include <QtGui/QToolBar>
#include <QtGui/QDesktopWidget>
#include <QtGui/QShortcut>
#include <QtCore/QSettings>

#include <QtWebKit/QWebPage>
#include <QtWebKit/QWebFrame>

#define PERFORM_WEBPAGE_ACTION(ACTION) \
    if (currentTab())                  \
        currentTab()->page()->triggerAction(QWebPage::ACTION);

BrowserMainWindow::BrowserMainWindow()
    : m_mainTabWidget(new QTabWidget(this))
    , m_searchWidget(0)
    , m_historyBack(0)
    , m_historyForward(0)
    , m_stop(0)
    , m_reload(0)
    , m_cut(0)
    , m_copy(0)
    , m_paste(0)
    , m_undo(0)
    , m_redo(0)
    , m_fullScreen(0)
    , m_proxy(new BrowserTabProxy(this))
{
    setAttribute(Qt::WA_DeleteOnClose);
    statusBar()->setSizeGripEnabled(true);

    setupMenu();
    setupToolBar();
    setCentralWidget(m_mainTabWidget);

    slotNewTab();

    connect(m_mainTabWidget, SIGNAL(currentChanged(int)), SLOT(slotCurrentTabChanged(int)));
    connect(m_proxy, SIGNAL(linkHovered(const QString&)), statusBar(), SLOT(showMessage(const QString&)));
    m_mainTabWidget->setElideMode(Qt::ElideRight);

    QShortcut* locationBarShortcut = new QShortcut(QKeySequence(Qt::CTRL + Qt::Key_L), this);
    connect(locationBarShortcut, SIGNAL(activated()), m_urlEdit, SLOT(selectAll()));
    connect(locationBarShortcut, SIGNAL(activated()), m_urlEdit, SLOT(setFocus()));

    QStringList args = QCoreApplication::arguments();
    if (args.count() > 1)
        loadPage(args.last());
    else
        slotHome();
    slotUpdateWindowTitle(QString());

    QSettings settings;
    settings.beginGroup("MainWindow");

    QRect desktopRect = QApplication::desktop()->screenGeometry();
    QSize size = settings.value("Size", desktopRect.size() * 0.8).toSize();
    resize(size);

    settings.endGroup();
}

BrowserMainWindow::~BrowserMainWindow()
{
    QSettings settings;
    settings.beginGroup("MainWindow");
    settings.setValue("Size", size());
    settings.endGroup();
}

void BrowserMainWindow::setupMenu()
{
    QMenu* fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(tr("&New Window"), this, SLOT(slotNewWindow()), QKeySequence::New);
    fileMenu->addAction(tr("New &Tab"), this, SLOT(slotNewTab()), QKeySequence::AddTab);
    fileMenu->addAction(tr("&Open File"), this, SLOT(slotOpenFile()), QKeySequence::Open);
    fileMenu->addAction(tr("&Close Window"), this, SLOT(slotCloseTab()), QKeySequence::Close);
    fileMenu->addSeparator();
    fileMenu->addAction(tr("&Quit"), qApp, SLOT(quit()));

    QMenu* editMenu = menuBar()->addMenu(tr("&Edit"));
    m_undo = editMenu->addAction(tr("&Undo"), this, SLOT(slotUndo()), QKeySequence::Undo);
    m_redo = editMenu->addAction(tr("&Redo"), this, SLOT(slotRedo()), QKeySequence::Redo);
    editMenu->addSeparator();
    m_cut = editMenu->addAction(tr("Cu&t"), this, SLOT(slotCut()), QKeySequence::Cut);
    m_copy = editMenu->addAction(tr("&Copy"), this, SLOT(slotCopy()), QKeySequence::Copy);
    m_paste = editMenu->addAction(tr("&Paste"), this, SLOT(slotPaste()), QKeySequence::Paste);
#if 0
    editMenu->addSeparator();
    editMenu->addAction(tr("Select &All"), this, SLOT(slotSelectAll()), QKeySequence::SelectAll);
    editMenu->addSeparator();
    editMenu->addAction(tr("Find in This Page"), this, SLOT(slotFind()), QKeySequence::Find);
    editMenu->addAction(tr("Find A&gain"), this, SLOT(slotFindNext()), QKeySequence::FindNext);
    editMenu->addSeparator();
    editMenu->addAction(tr("S&witch Text Direction"), this, SLOT(slotSwitchTextDirection()), QKeySequence(tr("Ctrl+Shift+X", "Switch Text Direction")));
#endif

    QMenu* viewMenu = menuBar()->addMenu(tr("&View"));

    m_viewStatusbar = new QAction(tr("Hide Status Bar"), this);
    m_viewStatusbar->setShortcut(tr("Ctrl+/"));
    QObject::connect(m_viewStatusbar, SIGNAL(triggered()), this, SLOT(slotToggleStatusbar()));
    viewMenu->addAction(m_viewStatusbar);

    m_viewToolbar = new QAction(tr("Hide Toolbar"), this);
    m_viewToolbar->setShortcut(tr("Ctrl+|"));
    QObject::connect(m_viewToolbar, SIGNAL(triggered()), this, SLOT(slotToggleToolbar()));
    viewMenu->addAction(m_viewToolbar);
    viewMenu->addSeparator();

    m_stop = viewMenu->addAction(tr("&Stop"), this, SLOT(slotStop()));
    m_reload = viewMenu->addAction(tr("&Reload"), this, SLOT(slotReload()));
#if 0
    viewMenu->addSeparator();
    viewMenu->addAction(tr("&Zoom In"), this, SLOT(slotZoomIn()));
    viewMenu->addAction(tr("Zoom &Out"), this, SLOT(slotZoomOut()));
#endif
    viewMenu->addSeparator();
    m_fullScreen = viewMenu->addAction(tr("&Full Screen"), this, SLOT(slotToggleFullScreen()));
    viewMenu->addAction(tr("Page S&ource"), this, SLOT(slotPageSource()));

    QMenu* historyMenu = menuBar()->addMenu(tr("Hi&story"));
    m_historyBack = historyMenu->addAction(tr("Back"), this, SLOT(slotBack()));
    m_historyForward = historyMenu->addAction(tr("Forward"), this, SLOT(slotForward()));
    historyMenu->addAction(tr("Home"), this, SLOT(slotHome()));
    historyMenu->addSeparator();

    QMenu* toolsMenu = menuBar()->addMenu(tr("&Tools"));
#if 0
    toolsMenu->addAction(tr("Web &Search"), this, SLOT(slotWebSearch()), QKeySequence(tr("Ctrl+K", "Web Search")));
    toolsMenu->addSeparator();
    toolsMenu->addAction(tr("&Downloads"), this, SLOT(slotShowDownloads()), QKeySequence(tr("Ctrl+Y", "Downloads")));
    toolsMenu->addSeparator();
    toolsMenu->addAction(tr("Error &Console"), this, SLOT(slotShowErrorConsole()));
#endif
    QAction *a = toolsMenu->addAction(tr("Enable Web &Inspector"), this, SLOT(slotToggleInspector(bool)));
    a->setCheckable(true);

    QMenu* helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(tr("About &Qt"), qApp, SLOT(aboutQt()));
    helpMenu->addAction(tr("About &WebKit"), this, SLOT(slotAboutWebKit()));

    (void)new QShortcut(Qt::ControlModifier + Qt::Key_PageDown, this, SLOT(nextTab()));
    (void)new QShortcut(Qt::ControlModifier + Qt::Key_PageUp, this, SLOT(previousTab()));
}

void BrowserMainWindow::setupToolBar()
{
    navigationBar = addToolBar(tr("Navigation"));

    m_historyBack->setIcon(style()->standardIcon(QStyle::SP_ArrowBack, 0, this));
    navigationBar->addAction(m_historyBack);

    m_historyForward->setIcon(style()->standardIcon(QStyle::SP_ArrowForward, 0, this));
    navigationBar->addAction(m_historyForward);

    m_stop->setText(tr("Stop"));
    navigationBar->addAction(m_stop);

    m_reload->setText(tr("Reload"));
    navigationBar->addAction(m_reload);

    m_urlEdit = new QLineEdit(navigationBar);
    navigationBar->addWidget(m_urlEdit);
    connect(m_urlEdit, SIGNAL(returnPressed()), SLOT(slotUrlEntered()));

    m_searchWidget = new SearchWidget(navigationBar);
    navigationBar->addWidget(m_searchWidget);
    connect(m_searchWidget, SIGNAL(search(const QUrl&)), SLOT(slotSearch(const QUrl&)));

    m_browserLogo = new QMovie(":spinning-qt.gif");
    m_browserLogo->jumpToFrame(0);

    QLabel* browserLogo = new QLabel(navigationBar);
    browserLogo->setMargin(4);
    browserLogo->setMovie(m_browserLogo);
    navigationBar->addWidget(browserLogo);
}

void BrowserMainWindow::slotToggleToolbar()
{
    if (navigationBar->isVisible()) {
        m_viewToolbar->setText(tr("Show Toolbar"));
        navigationBar->close();
    } else {
        m_viewToolbar->setText(tr("Hide Toolbar"));
        navigationBar->show();
    }
}

void BrowserMainWindow::slotToggleStatusbar()
{
    if (statusBar()->isVisible()) {
        m_viewStatusbar->setText(tr("Show Status Bar"));
        statusBar()->close();
    } else {
        m_viewStatusbar->setText(tr("Hide Status Bar"));
        statusBar()->show();
    }
}

QUrl BrowserMainWindow::guessUrlFromString(const QString& string)
{
    QUrl url(string);
    if (url.scheme().isEmpty())
        url = QUrl("http://" + string);

    return url;
}

void BrowserMainWindow::slotSearch(const QUrl& url)
{
    replaceCurrentUrl(url);

    if (currentTab())
        currentTab()->loadUrl(url);
}

BrowserTab *BrowserMainWindow::createTab()
{
    BrowserTab* tab = new BrowserTab(this, m_mainTabWidget);
    connect(tab, SIGNAL(titleChanged(const QString&)), SLOT(slotBrowserTabTitleChanged(const QString&)));
    connect(tab, SIGNAL(iconLoaded()), SLOT(slotBrowserTabIconChanged()));
    m_mainTabWidget->addTab(tab, tr("(Untitled)"));
    return tab;
}

void BrowserMainWindow::slotNewTab()
{
    BrowserTab* tab = createTab();
    m_urlEdit->setFocus();
    m_mainTabWidget->setCurrentWidget(tab);
    setCurrentTab(tab);
}

void BrowserMainWindow::slotCloseTab(int index)
{
    QWidget* widget = m_mainTabWidget->widget(index);

    if (qobject_cast<BrowserTab*>(widget)) {
        BrowserTab* tab = static_cast<BrowserTab*>(widget);
        if (tab->isModified()) {
            QMessageBox closeConfirmation(tab);
            closeConfirmation.setWindowFlags(Qt::Sheet);
            closeConfirmation.setWindowTitle(tr("Do you really want to close this page?"));
            closeConfirmation.setInformativeText(tr("You have modified this page and when closing it you will lose the modification.\n"
                                                     "Do you really want to close this page?\n"));
            closeConfirmation.setIcon(QMessageBox::Question);
            closeConfirmation.addButton(QMessageBox::Yes);
            closeConfirmation.addButton(QMessageBox::No);
            closeConfirmation.setEscapeButton(QMessageBox::No);
            if (closeConfirmation.exec() == QMessageBox::No)
                return;
        }

        m_recentlyClosedUrls.prepend(tab->url());
        if (m_recentlyClosedUrls.size() >= BrowserMainWindow::LastClosedUrlsSize)
            m_recentlyClosedUrls.resize(BrowserMainWindow::LastClosedUrlsSize -1);
    }

    m_mainTabWidget->removeTab(index);
    setCurrentTab(currentTab());

    // close the window when the last tab is gone
    if (m_mainTabWidget->count() == 0) {
        close();
        slotNewTab();
        slotHome();
    }
}

void BrowserMainWindow::slotUpdateStatusbar(const QString& txt)
{
    statusBar()->showMessage(txt);
}

void BrowserMainWindow::slotUpdateWindowTitle(const QString& title)
{
    if (title.isEmpty())
        setWindowTitle(tr("Qt Example Browser"));
    else
        setWindowTitle(tr("%1 - Qt Example Browser", "Page title and Browser name").arg(title));
}

void BrowserMainWindow::slotBrowserTabTitleChanged(const QString& newTitle)
{
    Q_ASSERT(sender() && qobject_cast<BrowserTab*>(sender()));

    m_mainTabWidget->setTabText(m_mainTabWidget->indexOf(qobject_cast<BrowserTab*>(sender())), newTitle);
}

void BrowserMainWindow::slotBrowserTabIconChanged()
{
    BrowserTab *browser = qobject_cast<BrowserTab*>(sender());

    m_mainTabWidget->setTabIcon(m_mainTabWidget->indexOf(browser), browser->icon());
}

void BrowserMainWindow::nextTab()
{
    int next = m_mainTabWidget->currentIndex() + 1;
    if (next == m_mainTabWidget->count())
        next = 0;
    m_mainTabWidget->setCurrentIndex(next);
}

void BrowserMainWindow::previousTab()
{
    int next = m_mainTabWidget->currentIndex() - 1;
    if (next < 0)
        next = m_mainTabWidget->count() - 1;
    m_mainTabWidget->setCurrentIndex(next);
}

void BrowserMainWindow::slotAboutWebKit()
{
    QMessageBox::about(this, tr("About"), tr(
        "<p>This example demonstrates Qt's "
        "webkit facilities in action, providing an example "
        "browser for you to experiment with.<p>"
        "<p>QtWebKit is based on the Open Source WebKit Project developed at <a href=\"http://webkit.org/\">http://webkit.org/</a>."
        ));
}

void BrowserMainWindow::slotNewWindow()
{
    QWidget* w = new BrowserMainWindow;
    w->show();
}


void BrowserMainWindow::slotOpenFile()
{
    QString file = QFileDialog::getOpenFileName(this, tr("Open Web Resource"), QString(),
            tr("Web Resources (*.html *.htm *.svg *.svgz)"));

    if (file.isEmpty())
        return;

    loadPage("file://" + file);
}


void BrowserMainWindow::slotCloseTab()
{
    if (!m_mainTabWidget->currentWidget())
        return;

    slotCloseTab(m_mainTabWidget->currentIndex());
}


void BrowserMainWindow::slotUndo()
{
    PERFORM_WEBPAGE_ACTION(Undo)
}


void BrowserMainWindow::slotRedo()
{
    PERFORM_WEBPAGE_ACTION(Redo)
}


void BrowserMainWindow::slotCut()
{
    PERFORM_WEBPAGE_ACTION(Cut)
}


void BrowserMainWindow::slotCopy()
{
    PERFORM_WEBPAGE_ACTION(Copy)
}


void BrowserMainWindow::slotPaste()
{
    PERFORM_WEBPAGE_ACTION(Paste)
}


void BrowserMainWindow::slotSelectAll()
{
    // WARNING XXX API
}


void BrowserMainWindow::slotFind()
{
    // WARNING XXX API
}


void BrowserMainWindow::slotFindNext()
{
    // WARNING XXX API
}


void BrowserMainWindow::slotSwitchTextDirection()
{
    // WARNING XXX API
}


void BrowserMainWindow::slotStop()
{
    PERFORM_WEBPAGE_ACTION(Stop)
}


void BrowserMainWindow::slotReload()
{
    PERFORM_WEBPAGE_ACTION(Reload)
}


void BrowserMainWindow::slotZoomIn()
{
}


void BrowserMainWindow::slotZoomOut()
{
}


void BrowserMainWindow::slotToggleFullScreen()
{
    if (isFullScreen()) {
        showNormal();
        m_fullScreen->setText(tr("&Full Screen"));
    } else {
        showFullScreen();
        m_fullScreen->setText(tr("Normal S&creen"));
    }
}


void BrowserMainWindow::slotPageSource()
{
    if (!currentTab())
        return;

    QString markup = currentTab()->page()->mainFrame()->markup();
    SourceCodeView* view = new SourceCodeView(markup);
    view->setWindowTitle(tr("Page Source of %1").arg(currentTab()->title()));
    view->setMinimumWidth(640);
    view->setAttribute(Qt::WA_DeleteOnClose);
    view->show();
}


void BrowserMainWindow::slotBack()
{
    PERFORM_WEBPAGE_ACTION(GoBack)
}


void BrowserMainWindow::slotForward()
{
    PERFORM_WEBPAGE_ACTION(GoForward)
}


void BrowserMainWindow::slotHome()
{
    loadPage("http://www.trolltech.com");
}


void BrowserMainWindow::slotWebSearch()
{
    m_searchWidget->selectAll();
    m_searchWidget->setFocus();
}


void BrowserMainWindow::slotShowDownloads()
{
}


void BrowserMainWindow::slotShowErrorConsole()
{
}


void BrowserMainWindow::slotToggleInspector(bool enable)
{
    QWebSettings::defaultSettings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, enable);
    if (enable) {
        if (QMessageBox::question(this, tr("Web Inspector"),
            tr("The web inspector will only work correctly for pages that were loaded after enabling.\n"
                "Do you want to reload all pages?"),
                QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
        }
        for (int i = 0; i < m_mainTabWidget->count(); ++i)
            static_cast<BrowserTab *>(m_mainTabWidget->widget(i))->triggerAction(QWebPage::Reload);
    }
}

void BrowserMainWindow::slotUrlEntered()
{
    loadPage(m_urlEdit->text());
}

void BrowserMainWindow::loadPage(const QString& page)
{
    if (!currentTab())
        return;

    QUrl url = guessUrlFromString(page);
    m_urlEdit->setText(url.toString());

    currentTab()->loadUrl(url);
}

BrowserTab* BrowserMainWindow::currentTab() const
{
    // We might not have a tab, or it might be some kind of special widget
    return qobject_cast<BrowserTab*>(m_mainTabWidget->currentWidget());
}


/*
 * This method will set the content of the URL bar to @pararm url
 * if the text currently entered into it is unmodified as we want to preserve
 * user changes
 */
void BrowserMainWindow::replaceCurrentUrl(const QUrl& url)
{
    m_urlEdit->setText(url.toString());
}

void BrowserMainWindow::slotCurrentTabChanged(int index)
{
    // Can be 0, e.g. if we don't have a BrowserTab in our tabwidget
    setCurrentTab(qobject_cast<BrowserTab*>(m_mainTabWidget->widget(index)));
}

void BrowserMainWindow::setCurrentTab(BrowserTab* tab)
{
    // All the magic will happen there
    m_proxy->setTab(tab);
}

void BrowserMainWindow::slotLoadStarted()
{
    m_browserLogo->start();
}

void BrowserMainWindow::slotLoadFinished()
{
    m_browserLogo->stop();
    m_browserLogo->jumpToFrame(0);
}

void BrowserMainWindow::slotUrlChanged(const QUrl &url)
{
    replaceCurrentUrl(url);
}

