/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtOpenGL module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "qglpexshadermanager_p.h"

#include "glgc_shader_source.h"

QGLPEXShaderManager::QGLPEXShaderManager(const QGLContext* context)
{
    ctx = const_cast<QGLContext*>(context);

    vertexShader= new QGLShader(QGLShader::VertexShader, context);
    vertexShader->addSource(QLatin1String(qglVertexShaderSource));
    if (!vertexShader->compile())
        qWarning() << "Vertex shader failed to compile: " << vertexShader->log();

    noBrushShader = new QGLShader(QGLShader::FragmentShader, context);
    noBrushShader->addSource(QLatin1String(qglNoBrushSource));
    if (!noBrushShader->compile())
        qWarning() << "No brush shader failed to compile:" << noBrushShader->log();


    // Create a program for noBrush:
    QGLShaderProgram* noBrushProg = new QGLShaderProgram(ctx);
    noBrushProg->addShader(vertexShader);
    noBrushProg->addShader(noBrushShader);
    if (!noBrushProg->link())
        qWarning() << "NoBrush shader program failed to link:" << noBrushProg->log();

    // Add noBrush Program to cache:
    QGLCachedShaderProg cachedProg;
    cachedProg.hasMask = false;
    cachedProg.brushShader = noBrushShader;
    cachedProg.compositionShader = 0;
    cachedProg.shader = noBrushProg;
    cachedPrograms.append(cachedProg);


    // Set state
    opaquePainting = true;
    currentBrushStyle = Qt::NoBrush;
    shaderProgNeedsChanging = false;
    activeProgram = noBrushProg;

    solidBrushShader    = 0;
    patternBrushShader  = 0;
    linearBrushShader   = 0;
    conicalBrushShader  = 0;
    radialBrushShader   = 0;
    textureBrushShader  = 0;

    simpleFragmentShader = 0;
    simpleShaderProgram  = 0;

    imageVertexShader = 0;
    imageFragmentShader = 0;
    imageShaderProgram = 0;
}

QGLPEXShaderManager::~QGLPEXShaderManager()
{
    delete vertexShader;
    delete imageVertexShader;
    delete imageFragmentShader;
    delete imageShaderProgram;
    delete noBrushShader;
    delete solidBrushShader;
    delete patternBrushShader;
    delete linearBrushShader;
    delete conicalBrushShader;
    delete radialBrushShader;
    delete textureBrushShader;
    delete simpleFragmentShader;
    delete simpleShaderProgram;
}

void QGLPEXShaderManager::setIsOpaque(bool isOpaque)
{
    if (isOpaque != opaquePainting)
        shaderProgNeedsChanging = true;

    opaquePainting = isOpaque;
}

void QGLPEXShaderManager::setBrushStyle(Qt::BrushStyle style)
{
    if (currentBrushStyle != style)
        shaderProgNeedsChanging = true;

    currentBrushStyle = style;
}

bool QGLPEXShaderManager::useCorrectShaderProg()
{
    if (!shaderProgNeedsChanging) {
        activeProgram->use();
        return false;
    }

//     qDebug("QGLPEXShaderManager::useCorrectShader() - CHANGING THE SHADER PROGRAM!!!");

    QGLShader* newBrushShader = noBrushShader;

    // Make sure we compile up the correct brush shader
    switch (currentBrushStyle) {
    case Qt::NoBrush:
        break;
    case Qt::SolidPattern:
//         qDebug("Setting brush shader to SolidBrush");
        if (!solidBrushShader) {
            qDebug("Compiling solidBrushShader");
            solidBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            solidBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            solidBrushShader->addSource(QLatin1String(qglSolidBrushSource));
            if (!solidBrushShader->compile())
                qWarning() << "Solid brush shader failed to compile:" << solidBrushShader->log();
        }
        newBrushShader = solidBrushShader;
        break;
    case Qt::TexturePattern:
//         qDebug("Setting brush shader to TextureBrush");
        if (!textureBrushShader) {
            qDebug("Compiling textureBrushShader");
            textureBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            textureBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            textureBrushShader->addSource(QLatin1String(qglTextureBrushSource));
            if (!textureBrushShader->compile())
                qWarning() << "Texture brush shader failed to compile:" << textureBrushShader->log();
        }
        newBrushShader = textureBrushShader;
        break;
    case Qt::LinearGradientPattern:
//         qDebug("Setting brush shader to LinearGradientBrush");
        if (!linearBrushShader) {
            qDebug("Compiling linearBrushShader");
            linearBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            linearBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            linearBrushShader->addSource(QLatin1String(qglLinearGradientBrushSource));
            if (!linearBrushShader->compile())
                qWarning() << "Linear brush shader failed to compile:" << linearBrushShader->log();
        }
        newBrushShader = linearBrushShader;
        break;
    case Qt::RadialGradientPattern:
//         qDebug("Setting brush shader to RadialGradientBrush");
        if (!radialBrushShader) {
            qDebug("Compiling radialBrushShader");
            radialBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            radialBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            radialBrushShader->addSource(QLatin1String(qglRadialGradientBrushSource));
            if (!radialBrushShader->compile())
                qWarning() << "Radial brush shader failed to compile:" << radialBrushShader->log();
        }
        newBrushShader = radialBrushShader;
        break;
    case Qt::ConicalGradientPattern:
//         qDebug("Setting brush shader to ConicalGradientBrush");
        if (!conicalBrushShader) {
            qDebug("Compiling conicalBrushShader");
            conicalBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            conicalBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            conicalBrushShader->addSource(QLatin1String(qglConicalGradientBrushSource));
            if (!conicalBrushShader->compile())
                qWarning() << "Conical brush shader failed to compile:" << conicalBrushShader->log();
        }
        newBrushShader = conicalBrushShader;
        break;
    case Qt::Dense1Pattern:
    case Qt::Dense2Pattern:
    case Qt::Dense3Pattern:
    case Qt::Dense4Pattern:
    case Qt::Dense5Pattern:
    case Qt::Dense6Pattern:
    case Qt::Dense7Pattern:
    case Qt::HorPattern:
    case Qt::VerPattern:
    case Qt::CrossPattern:
    case Qt::BDiagPattern:
    case Qt::FDiagPattern:
    case Qt::DiagCrossPattern:
//         qDebug("Setting brush shader to PatternBrush");
        if (!patternBrushShader) {
            qDebug("Compiling patternBrushShader");
            patternBrushShader = new QGLShader(QGLShader::FragmentShader, ctx);
            patternBrushShader->addSource(QLatin1String(qglFragmentShaderMainSource));
            patternBrushShader->addSource(QLatin1String(qglPatternBrushSource));
            if (!patternBrushShader->compile())
                qWarning() << "Pattern brush shader failed to compile:" << patternBrushShader->log();
        }
        newBrushShader = patternBrushShader;
        break;
    default:
        qWarning("Unimplemented brush style (%d)", currentBrushStyle);
    }

    // Now newBrushShader is set correctly, check to see if we already have the program
    // already linked and ready to go in the cache:
    bool foundProgram = false;
    foreach (QGLCachedShaderProg cachedProg, cachedPrograms) {
        if ((cachedProg.hasMask == false) &&
            (cachedProg.brushShader == newBrushShader) &&
            (cachedProg.compositionShader == 0) ) {

            activeProgram = cachedProg.shader;
            foundProgram = true;
            break;
        }
    }

    if (!foundProgram) {
        qDebug() << "Linking shader program for " << currentBrushStyle;
        // Required program not found - create it.
        QGLShaderProgram* newProg = new QGLShaderProgram(ctx);

        newProg->addShader(vertexShader);
        newProg->addShader(newBrushShader);

        if (!newProg->link())
            qWarning() << "Shader program for " << currentBrushStyle << "failed to link:" << newProg->log();

        QGLCachedShaderProg cachedProg;
        cachedProg.hasMask = false;
        cachedProg.brushShader = newBrushShader;
        cachedProg.compositionShader = 0;
        cachedProg.shader = newProg;

        cachedPrograms.append(cachedProg);
        activeProgram = newProg;
    }

    activeProgram->use();
    shaderProgNeedsChanging = false;
    return true;
}

QGLShaderProgram* QGLPEXShaderManager::brushShader()
{
    return activeProgram;
}

// The only uniform the simple shader has is the PMV matrix
QGLShaderProgram* QGLPEXShaderManager::simpleShader()
{
    if (!simpleShaderProgram) {
        simpleShaderProgram = new QGLShaderProgram(ctx);

        if (!simpleFragmentShader) {
            simpleFragmentShader = new QGLShader(QGLShader::FragmentShader, ctx);
            simpleFragmentShader->addSource(QLatin1String(qglSimpleFragShaderSource));
            if (!simpleFragmentShader->compile())
                qWarning() << "Simple fragment shader failed to compile:" << simpleFragmentShader->log();
        }

        simpleShaderProgram->addShader(vertexShader);
        simpleShaderProgram->addShader(simpleFragmentShader);
        if (!simpleShaderProgram->link())
            qWarning() << "Simple shader program failed to link:" << simpleShaderProgram->log();
    }

    return simpleShaderProgram;
}

QGLShaderProgram* QGLPEXShaderManager::imageShader()
{
    if (!imageShaderProgram) {
        if (!imageVertexShader) {
            imageVertexShader = new QGLShader(QGLShader::VertexShader, ctx);
            imageVertexShader->addSource(QLatin1String(qglImageVertexShaderSource));
            if (!imageVertexShader->compile())
                qWarning() << "Image/Pixmap vertex shader failed to compile:" << imageVertexShader->log();
        }

        if (!imageFragmentShader) {
            imageFragmentShader = new QGLShader(QGLShader::FragmentShader, ctx);
            imageFragmentShader->addSource(QLatin1String(qglImageFragmentShaderSource));
            if (!imageFragmentShader->compile())
                qWarning() << "Image/Pixmap fragment shader failed to compile:" << imageFragmentShader->log();
        }

        imageShaderProgram = new QGLShaderProgram(ctx);
        imageShaderProgram->addShader(imageVertexShader);
        imageShaderProgram->addShader(imageFragmentShader);
        if (!imageShaderProgram->link())
            qWarning() << "Image/Pixmap shader program failed to link:" << imageShaderProgram->log();
    }

    return imageShaderProgram;
}


