/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

/****************************************************************************
**
** Copyright (c) 2007-2008, Apple, Inc.
**
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are met:
**
**   * Redistributions of source code must retain the above copyright notice,
**     this list of conditions and the following disclaimer.
**
**   * Redistributions in binary form must reproduce the above copyright notice,
**     this list of conditions and the following disclaimer in the documentation
**     and/or other materials provided with the distribution.
**
**   * Neither the name of Apple, Inc. nor the names of its contributors
**     may be used to endorse or promote products derived from this software
**     without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
** CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
** EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
** PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
** PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
** LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
****************************************************************************/

#include <private/qcore_mac_p.h>
#include <qwidget.h>
#include <qdesktopwidget.h>
#include <qevent.h>
#include <private/qt_cocoa_helpers_mac_p.h>
#include <private/qt_mac_p.h>
#include <private/qapplication_p.h>
#include <private/qcocoawindow_mac_p.h>
#include <private/qkeymapper_p.h>


extern bool qt_sendSpontaneousEvent(QObject *receiver, QEvent *event); // qapplication.cpp;
extern Qt::MouseButton cocoaButton2QtButton(NSInteger buttonNum); // qcocoaview.mm
extern QWidget * mac_mouse_grabber;

void macWindowFade(void * /*OSWindowRef*/ window, float durationSeconds)
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
	if( wnd ) {
#if QT_MAC_USE_COCOA
        [NSAnimationContext beginGrouping];
        [[wnd animator] setAlphaValue:0.0];
        if (durationSeconds > 0) {
            [[NSAnimationContext currentContext] setDuration:NSTimeInterval(durationSeconds)];
        } else {
            durationSeconds = [[NSAnimationContext currentContext] duration];
        }
        [NSAnimationContext endGrouping];
        QTimer::singleShot(qRound(durationSeconds * 1000), [wnd qt_qwidget], SLOT(hide()));
#else
        if (durationSeconds <= 0)
            durationSeconds = 0.15;
		TransitionWindowOptions options = {0, durationSeconds, 0, 0};
		TransitionWindowWithOptions(wnd, kWindowFadeTransitionEffect, kWindowHideTransitionAction, 0, 1, &options);
#endif
	}
}



bool macWindowIsTextured( void * /*OSWindowRef*/ window )
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
#if QT_MAC_USE_COCOA
	return ( [wnd styleMask] & NSTexturedBackgroundWindowMask ) ? true : false;
#else
	WindowAttributes currentAttributes;
	GetWindowAttributes(wnd, &currentAttributes);
	return (currentAttributes & kWindowMetalAttribute) ? true : false;
#endif
}

void macWindowToolbarShow(const QWidget *widget, bool show )
{
    OSWindowRef wnd = qt_mac_window_for(widget);
#if QT_MAC_USE_COCOA
    NSToolbar *toolbar	= [wnd toolbar];
    if (toolbar) {
        if (show != [toolbar isVisible])
            [wnd toggleToolbarShown:wnd];
    }
#else
    ShowHideWindowToolbar(wnd, show, false);
#endif
}


void macWindowToolbarSet( void * /*OSWindowRef*/ window, void *toolbarRef  )
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
#if QT_MAC_USE_COCOA
	[wnd setToolbar:static_cast<NSToolbar *>(toolbarRef)];
#else
    SetWindowToolbar(wnd, static_cast<HIToolbarRef>(toolbarRef));
#endif
}

bool macWindowToolbarVisible( void * /*OSWindowRef*/ window )
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
#if QT_MAC_USE_COCOA
	NSToolbar *toolbar	= [wnd toolbar];
	if (toolbar)
        return [toolbar isVisible];
    return false;
#else
	return IsWindowToolbarVisible(wnd);
#endif
}

void macWindowSetHasShadow( void * /*OSWindowRef*/ window, bool hasShadow  )
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
#if QT_MAC_USE_COCOA
	[wnd setHasShadow:BOOL(hasShadow)];
#else
	if (hasShadow)
		ChangeWindowAttributes(wnd, 0, kWindowNoShadowAttribute);
	else
		ChangeWindowAttributes(wnd, kWindowNoShadowAttribute, 0);
#endif
}

void macWindowFlush(void * /*OSWindowRef*/ window)
{
    OSWindowRef wnd = static_cast<OSWindowRef>(window);
#if QT_MAC_USE_COCOA
	[wnd flushWindowIfNeeded];
#else
	HIWindowFlush(wnd);
#endif
}

void * /*NSImage */qt_mac_create_nsimage(const QPixmap &pm)
{
    QMacCocoaAutoReleasePool pool;
    if(QCFType<CGImageRef> image = pm.toMacCGImageRef()) {
        NSImage *newImage = 0;
        NSRect imageRect = NSMakeRect(0.0, 0.0, CGImageGetWidth(image), CGImageGetHeight(image));
        newImage = [[NSImage alloc] initWithSize:imageRect.size];
        [newImage lockFocus];
        {
            CGContextRef imageContext = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
            CGContextDrawImage(imageContext, *(CGRect*)&imageRect, image);
        }
        [newImage unlockFocus];
        return newImage;
    }
    return 0;
}

void qt_mac_update_mouseTracking(QWidget *widget)
{
#ifdef QT_MAC_USE_COCOA
    [qt_mac_nativeview_for(widget) updateTrackingAreas];
#else
    Q_UNUSED(widget);
#endif
}

OSStatus qt_mac_drawCGImage(CGContextRef inContext, const CGRect *inBounds, CGImageRef inImage)
{
    // Verbatim copy if HIViewDrawCGImage (as shown on Carbon-Dev)
    OSStatus err = noErr;

    require_action(inContext != NULL, InvalidContext, err = paramErr);
    require_action(inBounds != NULL, InvalidBounds, err = paramErr);
    require_action(inImage != NULL, InvalidImage, err = paramErr);

    CGContextSaveGState( inContext );
    CGContextTranslateCTM (inContext, 0, inBounds->origin.y + CGRectGetMaxY(*inBounds));
    CGContextScaleCTM(inContext, 1, -1);

    CGContextDrawImage(inContext, *inBounds, inImage);

    CGContextRestoreGState(inContext);
InvalidImage:
InvalidBounds:
InvalidContext:
	return err;
}

bool qt_mac_checkForNativeSizeGrip(const QWidget *widget)
{
#ifndef QT_MAC_USE_COCOA
    OSViewRef nativeSizeGrip = 0;
    HIViewFindByID(HIViewGetRoot(HIViewGetWindow(HIViewRef(widget->winId()))), kHIViewWindowGrowBoxID, &nativeSizeGrip);
    return (nativeSizeGrip != 0);
#else
    return [[reinterpret_cast<NSView *>(widget->winId()) window] showsResizeIndicator];
#endif
}

Q_GLOBAL_STATIC(QMacTabletHash, tablet_hash)
QMacTabletHash *qt_mac_tablet_hash()
{
    return tablet_hash();
}

#ifdef QT_MAC_USE_COCOA
void qt_dispatchTabletProximityEvent(void * /*NSEvent * */ tabletEvent)
{
    NSEvent *proximityEvent = static_cast<NSEvent *>(tabletEvent);
    // simply construct a Carbon proximity record and handle it all in one spot.
    TabletProximityRec carbonProximityRec = { [proximityEvent vendorID],
                                              [proximityEvent tabletID],
                                              [proximityEvent pointingDeviceID],
                                              [proximityEvent deviceID],
                                              [proximityEvent systemTabletID],
                                              [proximityEvent vendorPointingDeviceType],
                                              [proximityEvent pointingDeviceSerialNumber],
                                              [proximityEvent uniqueID],
                                              [proximityEvent capabilityMask],
                                              [proximityEvent pointingDeviceType],
                                              [proximityEvent isEnteringProximity] };
    qt_dispatchTabletProximityEvent(carbonProximityRec);
}
#endif // QT_MAC_USE_COCOA

void qt_dispatchTabletProximityEvent(const TabletProximityRec &proxRec)
{
    QTabletDeviceData proximityDevice;
    proximityDevice.tabletUniqueID = proxRec.uniqueID;
    proximityDevice.capabilityMask = proxRec.capabilityMask;

    switch (proxRec.pointerType) {
        case NSUnknownPointingDevice:
        default:
            proximityDevice.tabletPointerType = QTabletEvent::UnknownPointer;
            break;
        case NSPenPointingDevice:
            proximityDevice.tabletPointerType = QTabletEvent::Pen;
            break;
        case NSCursorPointingDevice:
            proximityDevice.tabletPointerType = QTabletEvent::Cursor;
            break;
        case NSEraserPointingDevice:
            proximityDevice.tabletPointerType = QTabletEvent::Eraser;
            break;
    }
    uint bits = proxRec.vendorPointerType;
    if (bits == 0 && proximityDevice.tabletUniqueID != 0) {
        // Fallback. It seems that the driver doesn't always include all the information.
        // High-End Wacom devices store their "type" in the uper bits of the Unique ID.
        // I'm not sure how to handle it for consumer devices, but I'll test that in a bit.
        bits = proximityDevice.tabletUniqueID >> 32;
    }
    // Defined in the "EN0056-NxtGenImpGuideX"
    // on Wacom's Developer Website (www.wacomeng.com)
    switch (bits & 0x0F06) {
        case 0x0802:
            proximityDevice.tabletDeviceType = QTabletEvent::Stylus;
            break;
        case 0x0902:
            proximityDevice.tabletDeviceType = QTabletEvent::Airbrush;
            break;
        case 0x0004:
            proximityDevice.tabletDeviceType = QTabletEvent::FourDMouse;
            break;
        case 0x0006:
            proximityDevice.tabletDeviceType = QTabletEvent::Puck;
            break;
        case 0x0804:
            proximityDevice.tabletDeviceType = QTabletEvent::RotationStylus;
            break;
        default:
            proximityDevice.tabletDeviceType = QTabletEvent::NoDevice;
    }
    // The deviceID is "unique" while in the proximity, it's a key that we can use for
    // linking up TabletDeviceData to an event (especially if there are two devices in action).
    bool entering = proxRec.enterProximity;
    if (entering) {
        qt_mac_tablet_hash()->insert(proxRec.deviceID, proximityDevice);
    } else {
        qt_mac_tablet_hash()->remove(proxRec.deviceID);
    }

    QTabletEvent qtabletProximity(entering ? QEvent::TabletEnterProximity
                                  : QEvent::TabletLeaveProximity,
                                  QPoint(), QPoint(), QPointF(), proximityDevice.tabletDeviceType,
                                  proximityDevice.tabletPointerType, 0., 0, 0, 0., 0., 0, 0,
                                  proximityDevice.tabletUniqueID);

    qt_sendSpontaneousEvent(qApp, &qtabletProximity);
}

#ifdef QT_MAC_USE_COCOA
// Use this method to keep all the information in the TextSegment. As long as it is ordered
// we are in OK shape, and we can influence that ourselves.
struct KeyPair
{
    QChar cocoaKey;
    Qt::Key qtKey;
};

bool operator==(const KeyPair &entry, QChar qchar)
{
    return entry.cocoaKey == qchar;
}

bool operator<(const KeyPair &entry, QChar qchar)
{
    return entry.cocoaKey < qchar;
}

bool operator<(QChar qchar, const KeyPair &entry)
{
    return qchar < entry.cocoaKey;
}

static Qt::Key cocoaKey2QtKey(QChar keyCode)
{
    static const int NumEntries = 57;
    static const KeyPair entries[NumEntries] = {
        { NSEnterCharacter, Qt::Key_Enter },
        { NSTabCharacter, Qt::Key_Tab },
        { NSCarriageReturnCharacter, Qt::Key_Return },
        { NSBackTabCharacter, Qt::Key_Backtab },
        { kEscapeCharCode, Qt::Key_Escape },
        { NSDeleteCharacter, Qt::Key_Backspace },
        { NSUpArrowFunctionKey, Qt::Key_Up },
        { NSDownArrowFunctionKey, Qt::Key_Down },
        { NSLeftArrowFunctionKey, Qt::Key_Left },
        { NSRightArrowFunctionKey, Qt::Key_Right },
        { NSF1FunctionKey, Qt::Key_F1 },
        { NSF2FunctionKey, Qt::Key_F2 },
        { NSF3FunctionKey, Qt::Key_F3 },
        { NSF4FunctionKey, Qt::Key_F4 },
        { NSF5FunctionKey, Qt::Key_F5 },
        { NSF6FunctionKey, Qt::Key_F6 },
        { NSF7FunctionKey, Qt::Key_F7 },
        { NSF8FunctionKey, Qt::Key_F8 },
        { NSF9FunctionKey, Qt::Key_F8 },
        { NSF10FunctionKey, Qt::Key_F10 },
        { NSF11FunctionKey, Qt::Key_F11 },
        { NSF12FunctionKey, Qt::Key_F12 },
        { NSF13FunctionKey, Qt::Key_F13 },
        { NSF14FunctionKey, Qt::Key_F14 },
        { NSF15FunctionKey, Qt::Key_F15 },
        { NSF16FunctionKey, Qt::Key_F16 },
        { NSF17FunctionKey, Qt::Key_F17 },
        { NSF18FunctionKey, Qt::Key_F18 },
        { NSF19FunctionKey, Qt::Key_F19 },
        { NSF20FunctionKey, Qt::Key_F20 },
        { NSF21FunctionKey, Qt::Key_F21 },
        { NSF22FunctionKey, Qt::Key_F22 },
        { NSF23FunctionKey, Qt::Key_F23 },
        { NSF24FunctionKey, Qt::Key_F24 },
        { NSF25FunctionKey, Qt::Key_F25 },
        { NSF26FunctionKey, Qt::Key_F26 },
        { NSF27FunctionKey, Qt::Key_F27 },
        { NSF28FunctionKey, Qt::Key_F28 },
        { NSF29FunctionKey, Qt::Key_F29 },
        { NSF30FunctionKey, Qt::Key_F30 },
        { NSF31FunctionKey, Qt::Key_F31 },
        { NSF32FunctionKey, Qt::Key_F32 },
        { NSF33FunctionKey, Qt::Key_F33 },
        { NSF34FunctionKey, Qt::Key_F34 },
        { NSF35FunctionKey, Qt::Key_F35 },
        { NSInsertFunctionKey, Qt::Key_Insert },
        { NSDeleteFunctionKey, Qt::Key_Delete },
        { NSHomeFunctionKey, Qt::Key_Home },
        { NSEndFunctionKey, Qt::Key_End },
        { NSPageUpFunctionKey, Qt::Key_PageUp },
        { NSPageDownFunctionKey, Qt::Key_PageDown },
        { NSPrintScreenFunctionKey, Qt::Key_Print },
        { NSScrollLockFunctionKey, Qt::Key_ScrollLock },
        { NSPauseFunctionKey, Qt::Key_Pause },
        { NSSysReqFunctionKey, Qt::Key_SysReq },
        { NSMenuFunctionKey, Qt::Key_Menu },
        { NSHelpFunctionKey, Qt::Key_Help },
    };
    static const KeyPair * const end = entries + NumEntries;
    const KeyPair *i = qBinaryFind(entries, end, keyCode);
    if (i == end)
        return Qt::Key(keyCode.unicode());
    return i->qtKey;
}

Qt::KeyboardModifiers qt_cocoaModifiers2QtModifiers(ulong modifierFlags)
{
    Qt::KeyboardModifiers qtMods =Qt::NoModifier;
    if (modifierFlags &  NSShiftKeyMask)
        qtMods |= Qt::ShiftModifier;
    if (modifierFlags & NSControlKeyMask)
        qtMods |= Qt::MetaModifier;
    if (modifierFlags & NSAlternateKeyMask)
        qtMods |= Qt::AltModifier;
    if (modifierFlags & NSCommandKeyMask)
        qtMods |= Qt::ControlModifier;
    if (modifierFlags & NSNumericPadKeyMask)
        qtMods |= Qt::KeypadModifier;
    return qtMods;
}

static inline QEvent::Type cocoaEvent2QtEvent(NSUInteger eventType)
{
    // Handle the trivial cases that can be determined from the type.
    switch (eventType) {
    case NSKeyDown:
        return QEvent::KeyPress;
    case NSKeyUp:
        return QEvent::KeyRelease;
    case NSLeftMouseDown:
    case NSRightMouseDown:
    case NSOtherMouseDown:
        return QEvent::MouseButtonPress;
    case NSLeftMouseUp:
    case NSRightMouseUp:
    case NSOtherMouseUp:
        return QEvent::MouseButtonRelease;
    case NSMouseMoved:
    case NSLeftMouseDragged:
    case NSRightMouseDragged:
    case NSOtherMouseDragged:
        return QEvent::MouseMove;
    case NSScrollWheel:
        return QEvent::Wheel;
    }
    return QEvent::None;
}

static bool mustUseCocoaKeyEvent()
{
    QCFType<TISInputSourceRef> source = TISCopyCurrentKeyboardInputSource();
    return TISGetInputSourceProperty(source, kTISPropertyUnicodeKeyLayoutData) == 0;
}

bool qt_dispatchKeyEventWithCocoa(void * /*NSEvent * */ keyEvent, QWidget *widgetToGetEvent)
{
    NSEvent *event = static_cast<NSEvent *>(keyEvent);
    NSString *keyChars = [event charactersIgnoringModifiers];
    int keyLength = [keyChars length];
    if (keyLength == 0)
        return false; // Dead Key, nothing to do!
    Qt::Key qtKey = Qt::Key_unknown;
    if (keyLength == 1) {
        qtKey = cocoaKey2QtKey(QChar([keyChars characterAtIndex:0]));
    }
    Qt::KeyboardModifiers keyMods = qt_cocoaModifiers2QtModifiers([event modifierFlags]);
    QString text;

    // To quote from the Carbon port: This is actually wrong--but it is the best that
    // can be done for now because of the Control/Meta mapping issues
    // (we always get text on the Mac)
    if (!(keyMods & (Qt::ControlModifier | Qt::MetaModifier)))
        text = QCFString::toQString(reinterpret_cast<CFStringRef>(keyChars));

    QKeyEvent ke(cocoaEvent2QtEvent([event type]), qtKey, keyMods, text, [event isARepeat], keyLength);
    qt_sendSpontaneousEvent(widgetToGetEvent, &ke);
    return ke.isAccepted();
}
#endif

// Helper to share code between QCocoaWindow and QCocoaView
bool qt_dispatchKeyEvent(void * /*NSEvent * */ keyEvent, QWidget *widgetToGetEvent)
{
#ifndef QT_MAC_USE_COCOA
    Q_UNUSED(keyEvent);
    Q_UNUSED(widgetToGetEvent);
    return false;
#else
    NSEvent *event = static_cast<NSEvent *>(keyEvent);
    EventRef key_event = static_cast<EventRef>(const_cast<void *>([event eventRef]));
    Q_ASSERT(key_event);
    if ([event type] == NSKeyDown) {
        qt_keymapper_private()->updateKeyMap(0, key_event, 0);
    }
    if (widgetToGetEvent == 0)
        return false;
    if (mustUseCocoaKeyEvent())
        return qt_dispatchKeyEventWithCocoa(keyEvent, widgetToGetEvent);
    return qt_keymapper_private()->translateKeyEvent(widgetToGetEvent, 0, key_event, 0, true);
#endif
}

void qt_dispatchModifiersChanged(void * /*NSEvent * */flagsChangedEvent, QWidget *widgetToGetEvent)
{
#ifndef QT_MAC_USE_COCOA
    Q_UNUSED(flagsChangedEvent);
    Q_UNUSED(widgetToGetEvent);
#else
    UInt32 modifiers = 0;
    // Sync modifiers with Qt
    NSEvent *event = static_cast<NSEvent *>(flagsChangedEvent);
    EventRef key_event = static_cast<EventRef>(const_cast<void *>([event eventRef]));
    Q_ASSERT(key_event);
    GetEventParameter(key_event, kEventParamKeyModifiers, typeUInt32, 0,
                      sizeof(modifiers), 0, &modifiers);
    extern void qt_mac_send_modifiers_changed(quint32 modifiers, QObject *object);
    qt_mac_send_modifiers_changed(modifiers, widgetToGetEvent);
#endif
}


QPointF flipPoint(const NSPoint &p)
{
    return QPointF(p.x, flipYCoordinate(p.y));
}

NSPoint flipPoint(const QPoint &p)
{
    return NSMakePoint(p.x(), flipYCoordinate(p.y()));
}

NSPoint flipPoint(const QPointF &p)
{
    return NSMakePoint(p.x(), flipYCoordinate(p.y()));
}

void qt_mac_dispatchNCMouseMessage(void * /* NSWindow* */eventWindow, void * /* NSEvent* */mouseEvent,
                                   QWidget *widgetToGetEvent, bool &leftButtonIsRightButton)
{
#ifndef QT_MAC_USE_COCOA
    Q_UNUSED(eventWindow);
    Q_UNUSED(mouseEvent);
    Q_UNUSED(widgetToGetEvent);
    Q_UNUSED(leftButtonIsRightButton);
#else
    if (widgetToGetEvent == 0)
        return;
    NSWindow *window = static_cast<NSWindow *>(eventWindow);
    NSEvent *event = static_cast<NSEvent *>(mouseEvent);
    NSEventType evtType = [event type];

    QPoint qlocalPoint;
    QPoint qglobalPoint;
    bool processThisEvent = false;
    bool fakeNCEvents = false;

    // Check if this is a mouse event.
    if (evtType == NSLeftMouseDown || evtType == NSLeftMouseUp
        || evtType == NSRightMouseDown || evtType == NSRightMouseUp
        || evtType == NSOtherMouseDown || evtType == NSOtherMouseUp
        || evtType == NSMouseMoved     || evtType == NSLeftMouseDragged
        || evtType == NSRightMouseDragged || evtType == NSOtherMouseDragged) {
        // Check if we want to pass this message to another window
        if (mac_mouse_grabber  && mac_mouse_grabber != widgetToGetEvent) {
            NSWindow *grabWindow = static_cast<NSWindow *>(qt_mac_window_for(mac_mouse_grabber));
            if (window != grabWindow) {
                window = grabWindow;
                widgetToGetEvent = mac_mouse_grabber;
                fakeNCEvents = true;
            }
        }
        // Dont generate normal NC mouse events for Left Button dragged
        if(evtType != NSLeftMouseDragged || fakeNCEvents) {
            NSPoint windowPoint = [event locationInWindow];
            NSPoint globalPoint = [[event window] convertBaseToScreen:windowPoint];
            NSRect frameRect = [window frame];
            if (fakeNCEvents || NSMouseInRect(globalPoint, frameRect, NO)) {
                NSRect contentRect = [window contentRectForFrameRect:frameRect];
                if (fakeNCEvents || !NSMouseInRect(globalPoint, contentRect, NO)) {
                    qglobalPoint = QPoint(flipPoint(globalPoint).toPoint());
                    qlocalPoint = widgetToGetEvent->mapFromGlobal(qglobalPoint);
                    processThisEvent = true;
                }
            }
        }
    }
    // This is not an NC area mouse message.
    if (!processThisEvent)
        return;
    Qt::MouseButton button;
    QEvent::Type eventType;
    // Convert to Qt::Event type
    switch (evtType) {
        case NSLeftMouseDown:
            button = Qt::LeftButton;
            eventType = QEvent::NonClientAreaMouseButtonPress;
            break;
        case NSLeftMouseUp:
            button = Qt::LeftButton;
            eventType = QEvent::NonClientAreaMouseButtonRelease;
            break;
        case NSRightMouseDown:
            button = Qt::RightButton;
            eventType = QEvent::NonClientAreaMouseButtonPress;
            break;
        case NSRightMouseUp:
            button = Qt::RightButton;
            eventType = QEvent::NonClientAreaMouseButtonRelease;
            break;
        case NSOtherMouseDown:
            button = cocoaButton2QtButton([event buttonNumber]);
            eventType = QEvent::NonClientAreaMouseButtonPress;
            break;
        case NSOtherMouseUp:
            button = cocoaButton2QtButton([event buttonNumber]);
            eventType = QEvent::NonClientAreaMouseButtonRelease;
            break;
        case NSMouseMoved:
            button = Qt::NoButton;
            eventType = QEvent::NonClientAreaMouseMove;
            break;
        case NSLeftMouseDragged:
            button = Qt::LeftButton;
            eventType = QEvent::NonClientAreaMouseMove;
            break;
        case NSRightMouseDragged:
            button = Qt::RightButton;
            eventType = QEvent::NonClientAreaMouseMove;
            break;
        case NSOtherMouseDragged:
            button = cocoaButton2QtButton([event buttonNumber]);
            eventType = QEvent::NonClientAreaMouseMove;
            break;
        default:
            qWarning("not handled! Non client area mouse message");
            return;
    }

    Qt::KeyboardModifiers keyMods = qt_cocoaModifiers2QtModifiers([event modifierFlags]);
    if (eventType == QEvent::NonClientAreaMouseButtonPress) {
        NSInteger clickCount = [event clickCount];
        if (clickCount % 2 == 0)
            eventType = QEvent::NonClientAreaMouseButtonDblClick;
        if (button == Qt::LeftButton && (keyMods & Qt::MetaModifier)) {
            button = Qt::RightButton;
            leftButtonIsRightButton = true;
        }
    } else if (eventType == QEvent::NonClientAreaMouseButtonRelease) {
        if (button == Qt::LeftButton && leftButtonIsRightButton) {
            button = Qt::RightButton;
            leftButtonIsRightButton = false;
        }
    }
    QMouseEvent qme(eventType, qlocalPoint, qglobalPoint, button, button, keyMods);
    qt_sendSpontaneousEvent(widgetToGetEvent, &qme);
#endif
}

void qt_mac_updateContentBorderMetricts(void * /*OSWindowRef */window, const HIContentBorderMetrics &metrics)
{
    OSWindowRef theWindow = static_cast<OSWindowRef>(window);
#if !defined(QT_MAC_USE_COCOA)
#  if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_5
    if (QSysInfo::MacintoshVersion >= QSysInfo::MV_10_5) {
        HIWindowSetContentBorderThickness(theWindow, &metrics);
    }
#  else
    Q_UNUSED(window);
    Q_UNUSED(metrics);
#  endif
#else
    if ([theWindow styleMask] & NSTexturedBackgroundWindowMask)
        [theWindow setContentBorderThickness:metrics.top forEdge:NSMaxYEdge];
    [theWindow setContentBorderThickness:metrics.bottom forEdge:NSMinYEdge];
#endif
}

QStringList qt_mac_NSArrayToQStringList(void *nsarray)
{
    QStringList result;
    NSArray *array = static_cast<NSArray *>(nsarray);
    for (NSUInteger i=0; i<[array count]; ++i)
        result << qt_mac_NSStringToQString([array objectAtIndex:i]);
    return result;
}

void *qt_mac_QStringListToNSMutableArrayVoid(const QStringList &list)
{
    NSMutableArray *result = [NSMutableArray arrayWithCapacity:list.size()];
    for (int i=0; i<list.size(); ++i){
        [result addObject:reinterpret_cast<const NSString *>(QCFString::toCFStringRef(list[i]))];
    }
    return result;
}

// Carbon: Make sure you call QDEndContext on the context when done with it.
CGContextRef qt_mac_graphicsContextFor(QWidget *widget)
{
    if (!widget)
        return 0;

#ifndef QT_MAC_USE_COCOA
    CGContextRef context;
    CGrafPtr port = GetWindowPort(qt_mac_window_for(widget));
    QDBeginCGContext(port, &context);
#else
    CGContextRef context = (CGContextRef)[[NSGraphicsContext graphicsContextWithWindow:qt_mac_window_for(widget)] graphicsPort];
#endif
    return context;
}
