/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may only be used for
** evaluation and testing purposes.  It may not be used for commercial
** development.  You may use this file in accordance with the terms and
** conditions contained in the either Technology Preview License
** Agreement or the Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "qfiledialog.h"

#ifndef QT_NO_FILEDIALOG

/*****************************************************************************
  QFileDialog debug facilities
 *****************************************************************************/
//#define DEBUG_FILEDIALOG_FILTERS

#include <qapplication.h>
#include <private/qapplication_p.h>
#include <private/qfiledialog_p.h>
#include <private/qt_mac_p.h>
#include <private/qt_cocoa_helpers_mac_p.h>
#include <qregexp.h>
#include <qbuffer.h>
#include <qdebug.h>
#include <qstringlist.h>
#include <qaction.h>
#include <qtextcodec.h>
#include <qvarlengtharray.h>
#include <qdesktopwidget.h>
#include <stdlib.h>
#include "ui_qfiledialog.h"

QT_BEGIN_NAMESPACE

extern QStringList qt_make_filter_list(const QString &filter); // qfiledialog.cpp
extern QStringList qt_clean_filter_list(const QString &filter); // qfiledialog.cpp
extern const char *qt_file_dialog_filter_reg_exp; // qfiledialog.cpp
extern bool qt_mac_is_macsheet(const QWidget *w); // qwidget_mac.mm

QT_END_NAMESPACE

@class QNSOpenSavePanelDelegate;

@interface QNSOpenSavePanelDelegate : NSObject {
    @public
    NSOpenPanel *mOpenPanel;
    NSSavePanel *mSavePanel;
    NSView *mAccessoryView;
    NSPopUpButton *mPopUpButton;
    NSTextField *mTextField;
    QFileDialogPrivate *mPriv;
    NSString *mCurrentDir;
    bool mConfirmOverwrite;
    int mReturnCode;

    QFileDialog::AcceptMode mAcceptMode;
    QDir::Filters *mQDirFilter;
    QFileDialog::FileMode mFileMode;
    QFileDialog::Options *mFileOptions;

    QString *mLastFilterCheckPath;
    QString *mCurrentSelection;
    QStringList *mQDirFilterEntryList;
    QStringList *mNameFilterDropDownList;
    QStringList *mSelectedNameFilter;
}

- (NSString *)strip:(const QString &)label;
- (BOOL)panel:(id)sender shouldShowFilename:(NSString *)filename;
- (void)filterChanged:(id)sender;
- (void)showModelessPanel;
- (BOOL)runApplicationModalPanel;
- (void)showWindowModalSheet:(QWidget *)docWidget;
- (void)updateProperties;
- (QStringList)acceptableExtensionsForSave;
- (QString)removeExtensions:(const QString &)filter;
- (void)createTextField;
- (void)createPopUpButton:(const QString &)selectedFilter hideDetails:(BOOL)hideDetails;
- (void)createAccessory;

@end

@implementation QNSOpenSavePanelDelegate

- (id)initWithAcceptMode:(QFileDialog::AcceptMode)acceptMode
    title:(const QString &)title
    nameFilters:(const QStringList &)nameFilters
    selectedNameFilter:(const QString &)selectedNameFilter
    hideNameFilterDetails:(bool)hideNameFilterDetails
    qDirFilter:(QDir::Filters)qDirFilter
    fileOptions:(QFileDialog::FileDialogOptions)fileOptions
    fileMode:(QFileDialog::FileMode)fileMode
    selectFile:(const QString &)selectFile
    confirmOverwrite:(bool)confirm
    priv:(QFileDialogPrivate *)priv
{
    self = [super init];

    mAcceptMode = acceptMode;
    if (mAcceptMode == QFileDialog::AcceptOpen){
        mOpenPanel = [NSOpenPanel openPanel];
        mSavePanel = mOpenPanel;
    } else {
        mSavePanel = [NSSavePanel savePanel];
        mOpenPanel = 0;
    }

    [mSavePanel setDelegate:self];
    mQDirFilter = new QDir::Filters(qDirFilter);
    mFileOptions = new QFileDialog::Options(fileOptions);
    mFileMode = fileMode;
    mConfirmOverwrite = confirm;
    mReturnCode = -1;
    mPriv = priv;
    mCurrentDir = 0;
    mLastFilterCheckPath = new QString;
    mQDirFilterEntryList = new QStringList;
    mNameFilterDropDownList = new QStringList(nameFilters);
    mSelectedNameFilter = new QStringList(qt_clean_filter_list(nameFilters.value(0)));
    mCurrentSelection = new QString(selectFile);

    [mSavePanel setTitle:qt_mac_QStringToNSString(title)];
    [self createPopUpButton:selectedNameFilter hideDetails:hideNameFilterDetails];
    [self createTextField];
    [self createAccessory];
    [mSavePanel setAccessoryView:mNameFilterDropDownList->size() > 1 ? mAccessoryView : nil];

    if (mPriv){
        [mSavePanel setPrompt:[self strip:mPriv->acceptLabel]];
        if (mPriv->fileNameLabelExplicitlySat)
            [mSavePanel setNameFieldLabel:[self strip:mPriv->qFileDialogUi->fileNameLabel->text()]];
    }

    [self updateProperties];
    [mSavePanel retain];
    return self;
}

- (void)dealloc
{
    delete mQDirFilter;
    delete mFileOptions;
    delete mLastFilterCheckPath;
    delete mQDirFilterEntryList;
    delete mNameFilterDropDownList;
    delete mSelectedNameFilter;
    delete mCurrentSelection;

    [mSavePanel setAccessoryView:nil];
    [mPopUpButton release];
    [mTextField release];
    [mAccessoryView release];
    [mSavePanel setDelegate:nil];
    [mSavePanel release];
    [mCurrentDir release];
    [super dealloc];
}

- (NSString *)strip:(const QString &)label
{
    QAction  a(label, 0);
    return qt_mac_QStringToNSString(a.iconText());
}

- (void)closePanel
{
    *mCurrentSelection = qt_mac_NSStringToQString([mSavePanel filename]); 
    [mSavePanel close];
}

- (void)showModelessPanel
{
    if (mOpenPanel){
        QFileInfo info(*mCurrentSelection);
        NSString *filename = qt_mac_QStringToNSString(info.fileName()); 
        bool selectable = [self panel:nil shouldShowFilename:filename];
        [mOpenPanel 
            beginForDirectory:qt_mac_QStringToNSString(info.absolutePath())
            file:selectable ? filename : nil
            types:nil
            modelessDelegate:self
            didEndSelector:@selector(openPanelDidEnd:returnCode:contextInfo:)
            contextInfo:nil];
    }
}

- (BOOL)runApplicationModalPanel
{
    QBoolBlocker nativeDialogOnTop(QApplicationPrivate::native_modal_dialog_active);
    QFileInfo info(*mCurrentSelection);
    NSString *filename = qt_mac_QStringToNSString(info.fileName()); 
    bool selectable = [self panel:nil shouldShowFilename:filename];
    mReturnCode = [mSavePanel 
        runModalForDirectory:qt_mac_QStringToNSString(info.absolutePath())
        file:selectable ? filename : nil];
    return (mReturnCode == NSOKButton);
}

- (QDialog::DialogCode)dialogResultCode
{
    return (mReturnCode == NSOKButton) ? QDialog::Accepted : QDialog::Rejected;
}

- (void)showWindowModalSheet:(QWidget *)docWidget
{
    Q_UNUSED(docWidget);
    QFileInfo info(*mCurrentSelection);
    NSString *filename = qt_mac_QStringToNSString(info.fileName()); 
    bool selectable = [self panel:nil shouldShowFilename:filename];
    [mSavePanel 
        beginSheetForDirectory:qt_mac_QStringToNSString(info.absolutePath())
        file:selectable ? filename : nil
#ifdef QT_MAC_USE_COCOA
        modalForWindow:qt_mac_window_for(docWidget)
#else
        modalForWindow:nil
#endif
        modalDelegate:self
        didEndSelector:@selector(openPanelDidEnd:returnCode:contextInfo:)
        contextInfo:nil];
}

- (BOOL)panel:(id)sender shouldShowFilename:(NSString *)filename
{
    Q_UNUSED(sender);
    QString qtFileName = qt_mac_NSStringToQString(filename); 
    QFileInfo info(qtFileName);
    QString path = info.absolutePath();
    if (path != *mLastFilterCheckPath){
        *mLastFilterCheckPath = path;
        *mQDirFilterEntryList = info.dir().entryList(*mQDirFilter); 
    }
    // Check if the QDir filter accepts the file:
    if (!mQDirFilterEntryList->contains(info.fileName()))
        return NO;

    // Always accept directories regardless of their names:
    BOOL isDir;
    if ([[NSFileManager defaultManager] fileExistsAtPath:filename isDirectory:&isDir] && isDir)
        return YES;

    // No filter means accept everything
    if (mSelectedNameFilter->isEmpty())
        return YES;
    // Check if the current file name filter accepts the file:
    for (int i=0; i<mSelectedNameFilter->size(); ++i) {
        if (QDir::match(mSelectedNameFilter->at(i), qtFileName))
            return YES;
    }
    return NO;
}

- (NSString *)panel:(id)sender userEnteredFilename:(NSString *)filename confirmed:(BOOL)okFlag
{
    Q_UNUSED(sender);
    if (!okFlag)
        return filename;
    if (mConfirmOverwrite)
        return filename;

    // User has clicked save, and no overwrite confirmation should occur.
    // To get the latter, we need to change the name we return (hence the prefix):
    return [@"___qt_very_unlikely_prefix_" stringByAppendingString:filename];
}

- (void)setNameFilters:(const QStringList &)filters hideDetails:(BOOL)hideDetails
{
    [mPopUpButton removeAllItems];
    *mNameFilterDropDownList = filters;
    if (filters.size() > 0){
        for (int i=0; i<filters.size(); ++i) {
            QString filter = hideDetails ? [self removeExtensions:filters.at(i)] : filters.at(i);
            [mPopUpButton addItemWithTitle:qt_mac_QStringToNSString(filter)];
        }
        [mPopUpButton selectItemAtIndex:0]; 
        [mSavePanel setAccessoryView:mAccessoryView];
    } else
        [mSavePanel setAccessoryView:nil];
    
    [self filterChanged:self];
}

- (void)filterChanged:(id)sender
{
    // This mDelegate function is called when the _name_ filter changes.
    Q_UNUSED(sender);
    QString selection = mNameFilterDropDownList->value([mPopUpButton indexOfSelectedItem]);
    *mSelectedNameFilter = qt_clean_filter_list(selection);
    [mSavePanel validateVisibleColumns];
    [self updateProperties];
    if (mPriv)
        mPriv->QNSOpenSavePanelDelegate_filterSelected([mPopUpButton indexOfSelectedItem]);
}

- (QString)currentNameFilter
{
    return mNameFilterDropDownList->value([mPopUpButton indexOfSelectedItem]);
}

- (QStringList)selectedFiles
{
    if (mOpenPanel)
        return qt_mac_NSArrayToQStringList([mOpenPanel filenames]);
    else{
        QStringList result;
        QString filename = qt_mac_NSStringToQString([mSavePanel filename]);
        result << filename.remove(QLatin1String("___qt_very_unlikely_prefix_"));
        return result; 
    }
}

- (void)updateProperties
{
    // Call this functions if mFileMode, mFileOptions,
    // mNameFilterDropDownList or mQDirFilter changes.
    // The savepanel does not contain the neccessary functions for this.
    bool chooseFilesOnly = mFileMode == QFileDialog::ExistingFile
        || mFileMode == QFileDialog::ExistingFiles;
    bool chooseDirsOnly = mFileMode == QFileDialog::Directory
        || mFileMode == QFileDialog::DirectoryOnly 
        || *mFileOptions & QFileDialog::ShowDirsOnly;

    [mOpenPanel setCanChooseFiles:!chooseDirsOnly];
    [mOpenPanel setCanChooseDirectories:!chooseFilesOnly];
    [mSavePanel setCanCreateDirectories:!(*mFileOptions & QFileDialog::ReadOnly)];
    [mOpenPanel setAllowsMultipleSelection:(mFileMode == QFileDialog::ExistingFiles)];
    [mOpenPanel setResolvesAliases:!(*mFileOptions & QFileDialog::DontResolveSymlinks)];

    QStringList ext = [self acceptableExtensionsForSave];
    if (mPriv && !ext.isEmpty() && !mPriv->defaultSuffix.isEmpty())
        ext.prepend(mPriv->defaultSuffix);
    [mSavePanel setAllowedFileTypes:ext.isEmpty() ? nil : qt_mac_QStringListToNSMutableArray(ext)];

    [mOpenPanel validateVisibleColumns];
}

- (void)panelSelectionDidChange:(id)sender
{
    Q_UNUSED(sender);
    *mCurrentSelection = qt_mac_NSStringToQString([mSavePanel filename]); 
    if (mPriv)
        mPriv->QNSOpenSavePanelDelegate_selectionChanged(*mCurrentSelection);
}

- (void)openPanelDidEnd:(NSOpenPanel *)panel returnCode:(int)returnCode  contextInfo:(void *)contextInfo
{
    Q_UNUSED(panel);
    Q_UNUSED(contextInfo);
    mReturnCode = returnCode;
    if (mPriv)
        mPriv->QNSOpenSavePanelDelegate_panelClosed(returnCode == NSOKButton);
}

- (void)panel:(id)sender directoryDidChange:(NSString *)path
{
    Q_UNUSED(sender);
    if (!mPriv)
        return;
    if ([path isEqualToString:mCurrentDir])
        return;

    [mCurrentDir release];
    mCurrentDir = [path retain];
    mPriv->QNSOpenSavePanelDelegate_directoryEntered(qt_mac_NSStringToQString(mCurrentDir));
}

/*
    Returns a list of extensions (e.g. "png", "jpg", "gif")
    for the current name filter. If a filter do not conform
    to the format *.xyz or * or *.*, an empty list
    is returned meaning accept everything.
*/
- (QStringList)acceptableExtensionsForSave
{
    QStringList result;
    for (int i=0; i<mSelectedNameFilter->count(); ++i) {
        const QString &filter = mSelectedNameFilter->at(i);
        if (filter.startsWith(QLatin1String("*."))
                && !filter.contains(QLatin1Char('?'))
                && filter.count(QLatin1Char('*')) == 1) {
            result += filter.mid(2);
        } else {
            return QStringList(); // Accept everything
        }
    }
    return result;
}

- (QString)removeExtensions:(const QString &)filter
{
    QRegExp regExp(QString::fromLatin1(qt_file_dialog_filter_reg_exp));
    if (regExp.indexIn(filter) != -1)
        return regExp.cap(1).trimmed();
    return filter;
}

- (void)createTextField
{
    NSRect textRect = { { 0.0, 3.0 }, { 100.0, 25.0 } };
    mTextField = [[NSTextField alloc] initWithFrame:textRect];
    [[mTextField cell] setFont:[NSFont systemFontOfSize:
            [NSFont systemFontSizeForControlSize:NSRegularControlSize]]];
    [mTextField setAlignment:NSRightTextAlignment];
    [mTextField setEditable:false];
    [mTextField setSelectable:false];
    [mTextField setBordered:false];
    [mTextField setDrawsBackground:false];
    if (mPriv){
        [mTextField setStringValue:[self strip:mPriv->qFileDialogUi->fileTypeLabel->text()]];
    } else
        [mTextField setStringValue:qt_mac_QStringToNSString(QFileDialog::tr("Files of type:"))];
}

- (void)createPopUpButton:(const QString &)selectedFilter hideDetails:(BOOL)hideDetails
{
    NSRect popUpRect = { { 100.0, 5.0 }, { 250.0, 25.0 } };
    mPopUpButton = [[NSPopUpButton alloc] initWithFrame:popUpRect pullsDown:NO];
    [mPopUpButton setTarget:self];
    [mPopUpButton setAction:@selector(filterChanged:)];
    
    QStringList *filters = mNameFilterDropDownList;
    if (filters->size() > 0){
        for (int i=0; i<mNameFilterDropDownList->size(); ++i) {
            if (filters->at(i) == selectedFilter)
                [mPopUpButton selectItemAtIndex:i]; 
            QString filter = hideDetails ? [self removeExtensions:filters->at(i)] : filters->at(i);
            [mPopUpButton addItemWithTitle:qt_mac_QStringToNSString(filter)];
        }
    }
   
    /*
        If there is no selected filter or we can't find it, we try to select the
        right filter based on the file name.
    */
    /*
    if (initial == -1 && !fileName.isEmpty()) {
        for (int i = 0; i < filters.size(); ++i) {
            QStringList cleanFilters = qt_clean_filter_list(filters.at(i));
            for (int j = 0; j < cleanFilters.size(); ++j) {
                if (QDir::match(cleanFilters.at(j), fileName)) {
                    initial = i;
                    goto break_outer_loop;
                }
            }
        }
    break_outer_loop:
        ;
    }

    if (initial != -1)
        [popUpButton selectItemAtIndex:NSInteger(initial)];
        */
}

- (void)createAccessory
{
    NSRect accessoryRect = { { 0.0, 0.0 }, { 450.0, 33.0 } };
    mAccessoryView = [[NSView alloc] initWithFrame:accessoryRect];
    [mAccessoryView addSubview:mTextField];
    [mAccessoryView addSubview:mPopUpButton];
}

@end

QT_BEGIN_NAMESPACE

void QFileDialogPrivate::QNSOpenSavePanelDelegate_selectionChanged(const QString &newPath)
{
    emit q_func()->currentChanged(newPath);
}

void QFileDialogPrivate::QNSOpenSavePanelDelegate_panelClosed(bool accepted)
{
    if (accepted)
        q_func()->accept();
    else
        q_func()->reject();
}

void QFileDialogPrivate::QNSOpenSavePanelDelegate_directoryEntered(const QString &newDir)
{
    emit q_func()->directoryEntered(newDir);
}

void QFileDialogPrivate::QNSOpenSavePanelDelegate_filterSelected(int menuIndex) 
{
    emit q_func()->filterSelected(nameFilters.at(menuIndex));
}

void QFileDialogPrivate::setDirectory_sys(const QString &directory)
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);
    [delegate->mSavePanel setDirectory:qt_mac_QStringToNSString(directory)];
}

QString QFileDialogPrivate::directory_sys() const
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);
    return qt_mac_NSStringToQString([delegate->mSavePanel directory]);
}

void QFileDialogPrivate::selectFile_sys(const QString &filename)
{
    // There seems to no way to select a file once the dialog is
    // running. But I can set the directory at least:
    setDirectory_sys(QFileInfo(filename).absolutePath());
}

QStringList QFileDialogPrivate::selectedFiles_sys() const
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);
    return [delegate selectedFiles];
}

void QFileDialogPrivate::setNameFilters_sys(const QStringList &filters)
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);
    bool hideDetails = q_func()->testOption(QFileDialog::HideNameFilterDetails);
    [delegate setNameFilters:filters hideDetails:hideDetails];
}

void QFileDialogPrivate::setFilter_sys()
{
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);
    *(delegate->mQDirFilter) = model->filter();
    [delegate updateProperties];
}

void QFileDialogPrivate::selectNameFilter_sys(const QString &filter)
{
    int index = nameFilters.indexOf(filter);
    if (index != -1){
        QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);    
        [delegate->mPopUpButton selectItemAtIndex:index]; 
        [delegate filterChanged:nil];
    }
}

QString QFileDialogPrivate::selectedNameFilter_sys() const
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);    
    int index = [delegate->mPopUpButton indexOfSelectedItem]; 
    return index != -1 ? nameFilters.at(index) : QString();
}

void QFileDialogPrivate::createNSOpenSavePanelDelegate()
{
    Q_Q(QFileDialog);
    if (mDelegate)
        return;

    QNSOpenSavePanelDelegate *delegate = [[QNSOpenSavePanelDelegate alloc]
        initWithAcceptMode:acceptMode
        title:q->windowTitle()
        nameFilters:q->nameFilters()
        selectedNameFilter:q->selectedNameFilter()
        hideNameFilterDetails:q->testOption(QFileDialog::HideNameFilterDetails)
        qDirFilter:model->filter()
        fileOptions:opts
        fileMode:fileMode
        selectFile:q->selectedFiles().value(0)
        confirmOverwrite:!q->testOption(QFileDialog::DontConfirmOverwrite)
        priv:this];

    mDelegate = delegate;
}

void QFileDialogPrivate::deleteNativeDialog_sys()
{
    QMacCocoaAutoReleasePool pool;
    [reinterpret_cast<QNSOpenSavePanelDelegate *>(mDelegate) release];
    mDelegate = 0;
    nativeDialogInUse = false;
}

bool QFileDialogPrivate::setVisible_sys(bool visible)
{
    Q_Q(QFileDialog);
    QMacCocoaAutoReleasePool pool;
    if (!visible && !mDelegate)
        return false;
    createNSOpenSavePanelDelegate();
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);    

    if (!visible){
        [delegate closePanel];
        return true;
    }

    if (q->parentWidget() && (qt_mac_is_macsheet(q) || q->windowModality() == Qt::WindowModal)){
        // QFileDialog::Options might tell us to QFileDialog::DontUseSheet,
        // but there is no way around it if we are to be window modal.
#ifdef QT_MAC_USE_COCOA
        [delegate showWindowModalSheet:q->parentWidget()];
#else
        return false;
#endif
    } else
        [delegate showModelessPanel];
    return true;
}

bool QFileDialogPrivate::exec_sys()
{
    // If this dialog is told to exec after a QEventLoop has just finished,
    // the main cocoa run loop will be stopped (no matter how many times
    // it has been recursed) until control returns back to the run loop.
    // In that case showing a native dialog app modal will not work correctly.
    // So to work around this case, recurse into a new event loop that restarts the
    // main cocoa run loop while this dialog executes:
    QMacCocoaAutoReleasePool pool;
    createNSOpenSavePanelDelegate();
    QTimer::singleShot(1, q_func(), SLOT(_q_cocoaAppModalHelp()));
    eventLoopToEnsureNSAppRuns.exec();
    return true;
}

void QFileDialogPrivate::_q_cocoaAppModalHelp()
{
    QMacCocoaAutoReleasePool pool;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);    
    [delegate runApplicationModalPanel];
    eventLoopToEnsureNSAppRuns.exit();
}

QDialog::DialogCode QFileDialogPrivate::dialogResultCode_sys()
{
    if (!mDelegate)
        return QDialog::Rejected;
    QNSOpenSavePanelDelegate *delegate = static_cast<QNSOpenSavePanelDelegate *>(mDelegate);    
    return [delegate dialogResultCode];
}

#ifndef QT_MAC_USE_COCOA
static UInt8 *str_buffer = 0;
static void cleanup_str_buffer()
{
    if (str_buffer) {
        free(str_buffer);
        str_buffer = 0;
    }
}

// Returns the wildcard part of a filter.
struct qt_mac_filter_name {
    QString description, regxp, filter;
};

static qt_mac_filter_name *qt_mac_extract_filter(const QString &rawFilter)
{
    qt_mac_filter_name *ret = new qt_mac_filter_name;
    ret->filter = rawFilter;
    QString result = rawFilter;
    QRegExp r(QString::fromLatin1(qt_file_dialog_filter_reg_exp));
    int index = r.indexIn(result);
    if (index >= 0) {
        ret->description = r.cap(1).trimmed();
        result = r.cap(2);
    }
    if (ret->description.isEmpty())
        ret->description = result;
    ret->regxp = result.replace(QLatin1Char(' '), QLatin1Char(';'));
    return ret;
}

// Makes a list of filters from ;;-separated text.
static QList<qt_mac_filter_name*> qt_mac_make_filters_list(const QString &filter)
{
#ifdef DEBUG_FILEDIALOG_FILTERS
    qDebug("QFileDialog:%d - Got filter (%s)", __LINE__, filter.latin1());
#endif
    QString f(filter);
    if (f.isEmpty())
        f = QFileDialog::tr("All Files (*)");
    if (f.isEmpty())
        return QList<qt_mac_filter_name*>();
/*
    QString sep(";;");
    int i = f.indexOf(sep, 0);
    if (i == -1) {
        sep = "\n";
        if (f.indexOf(sep, 0) != -1)
            i = f.indexOf(sep, 0);
    }
    QStringList filts = f.split(sep);
*/
    QStringList filts = qt_make_filter_list(f);
    QList<qt_mac_filter_name*> ret;
    for (QStringList::Iterator it = filts.begin(); it != filts.end(); ++it) {
        qt_mac_filter_name *filter = qt_mac_extract_filter((*it));
#ifdef DEBUG_FILEDIALOG_FILTERS
        qDebug("QFileDialog:%d Split out filter (%d) '%s' '%s' [%s]", __LINE__, ret.count(),
               filter->regxp.latin1(), filter->description.latin1(), (*it).latin1());
#endif
        ret.append(filter);
    }
    return ret;
}

struct qt_mac_nav_filter_type {
    int index;
    bool saveDialog;
    QList<qt_mac_filter_name*> *filts;
};

static Boolean qt_mac_nav_filter(AEDesc *theItem, void *info,
                                             void *myd, NavFilterModes)
{
    qt_mac_nav_filter_type *t = (qt_mac_nav_filter_type *)myd;
    if (!t || !t->filts || t->index >= t->filts->count())
        return true;

    NavFileOrFolderInfo *theInfo = (NavFileOrFolderInfo *)info;
    QString file;
    qt_mac_filter_name *fn = t->filts->at(t->index);
    if (!fn)
        return true;
    if (theItem->descriptorType == typeFSRef) {
        FSRef ref;
        AEGetDescData(theItem, &ref, sizeof(ref));
        if (!str_buffer) {
            qAddPostRoutine(cleanup_str_buffer);
            str_buffer = (UInt8 *)malloc(1024);
        }
        FSRefMakePath(&ref, str_buffer, 1024);
        file = QString::fromUtf8((const char *)str_buffer);
        int slsh = file.lastIndexOf(QLatin1Char('/'));
        if (slsh != -1)
            file = file.right(file.length() - slsh - 1);
    }
    QStringList reg = fn->regxp.split(QLatin1String(";"));
    for (QStringList::Iterator it = reg.begin(); it != reg.end(); ++it) {
        QRegExp rg(*it, Qt::CaseInsensitive, QRegExp::Wildcard);
#ifdef DEBUG_FILEDIALOG_FILTERS
        qDebug("QFileDialog:%d, asked to filter.. %s (%s)", __LINE__,
               file.latin1(), (*it).latin1());
#endif
        if (rg.exactMatch(file))
            return true;
    }
    return (theInfo->isFolder && !file.endsWith(QLatin1String(".app")));
}

//filter UPP stuff
static NavObjectFilterUPP mac_navFilterUPP = 0;
static void cleanup_navFilterUPP()
{
    DisposeNavObjectFilterUPP(mac_navFilterUPP);
    mac_navFilterUPP = 0;
}
static const NavObjectFilterUPP make_navFilterUPP()
{
    if (mac_navFilterUPP)
        return mac_navFilterUPP;
    qAddPostRoutine(cleanup_navFilterUPP);
    return mac_navFilterUPP = NewNavObjectFilterUPP(qt_mac_nav_filter);
}
//event UPP stuff
static NavEventUPP mac_navProcUPP = 0;
static void cleanup_navProcUPP()
{
    DisposeNavEventUPP(mac_navProcUPP);
    mac_navProcUPP = 0;
}
static bool g_nav_blocking=true;
static void qt_mac_filedialog_event_proc(const NavEventCallbackMessage msg,
                                                     NavCBRecPtr p, NavCallBackUserData myd)
{
    switch(msg) {
    case kNavCBPopupMenuSelect: {
        qt_mac_nav_filter_type *t = (qt_mac_nav_filter_type *)myd;
        NavMenuItemSpec *s = (NavMenuItemSpec*)p->eventData.eventDataParms.param;
        t->index = s->menuType;
        if (t->saveDialog) {
            QString base = QCFString::toQString(NavDialogGetSaveFileName(p->context));
            QFileInfo fi(base);
            base = fi.completeBaseName();
            qt_mac_filter_name *fn = t->filts->at(t->index);
            QStringList reg = fn->regxp.split(QLatin1String(";"), QString::SkipEmptyParts);
            QString r = reg.first();
            r  = r.right(r.length()-1);      // Strip the *
            base += r;                        //"." + QString::number(s->menuType);
            NavDialogSetSaveFileName(p->context, QCFString::toCFStringRef(base));
        }
#ifdef DEBUG_FILEDIALOG_FILTERS
        qDebug("QFileDialog:%d - Selected a filter: %ld", __LINE__, s->menuType);
#endif
        break; }
    case kNavCBStart:
        g_nav_blocking=true;
        break;
    case kNavCBUserAction:
        g_nav_blocking=false;
        break;
    }
}
static const NavEventUPP make_navProcUPP()
{
    if (mac_navProcUPP)
        return mac_navProcUPP;
    qAddPostRoutine(cleanup_navProcUPP);
    return mac_navProcUPP = NewNavEventUPP(qt_mac_filedialog_event_proc);
}

extern OSErr qt_mac_create_fsref(const QString &, FSRef *); // qglobal.cpp
static QStringList qt_mac_get_open_file_names_carbon(const QFileDialogArgs &args, QString *pwd, QString *selectedFilter)
{
    QWidget *parent = args.parent;
    OSErr err;
    QStringList retstrl;

    NavDialogCreationOptions options;
    NavGetDefaultDialogCreationOptions(&options);
    options.modality = kWindowModalityAppModal;
    options.optionFlags |= kNavSupportPackages;
    if (args.options & QFileDialog::DontConfirmOverwrite)
        options.optionFlags |= kNavDontConfirmReplacement;
    if (args.mode != QFileDialog::ExistingFiles)
        options.optionFlags &= ~kNavAllowMultipleFiles;

    if (!args.caption.isEmpty())
        options.windowTitle = QCFString::toCFStringRef(args.caption);

    static const int w = 450, h = 350;
    options.location.h = options.location.v = -1;
    if (parent && parent->isVisible()) {
        WindowClass wclass;
        GetWindowClass(qt_mac_window_for(parent), &wclass);
        parent = parent->window();
        QString s = parent->windowTitle();
        options.clientName = QCFString::toCFStringRef(s);
        options.location.h = (parent->x() + (parent->width() / 2)) - (w / 2);
        options.location.v = (parent->y() + (parent->height() / 2)) - (h / 2);

        QRect r = QApplication::desktop()->screenGeometry(
                QApplication::desktop()->screenNumber(parent));
        const int border = 10;
        if (options.location.h + w > r.right())
            options.location.h -= (options.location.h + w) - r.right() + border;
        if (options.location.v + h > r.bottom())
            options.location.v -= (options.location.v + h) - r.bottom() + border;
        if (options.location.h < r.left())
            options.location.h = r.left() + border;
        if (options.location.v < r.top())
            options.location.v = r.top() + border;
    }

    QList<qt_mac_filter_name*> filts = qt_mac_make_filters_list(args.filter);
    qt_mac_nav_filter_type t;
    t.saveDialog = false;
    t.index = 0;
    t.filts = &filts;
    if (filts.count() > 1) {
        int i = 0;
        CFStringRef *arr = static_cast<CFStringRef *>(malloc(sizeof(CFStringRef) * filts.count()));
        for (QList<qt_mac_filter_name*>::const_iterator it = filts.constBegin();
             it != filts.constEnd(); ++it)
            arr[i++] = QCFString::toCFStringRef((*it)->description);
        options.popupExtension = CFArrayCreate(0, reinterpret_cast<const void **>(arr), filts.count(), 0);
    }

    NavDialogRef dlg;
    if (args.mode == QFileDialog::DirectoryOnly ||
        args.mode == QFileDialog::Directory) {
        if (NavCreateChooseFolderDialog(&options, make_navProcUPP(), 0, 0, &dlg)) {
            qDebug("Shouldn't happen %s:%d", __FILE__, __LINE__);
            return retstrl;
        }
    } else {
        if (NavCreateGetFileDialog(&options, 0, make_navProcUPP(), 0,
                                  make_navFilterUPP(), (void *) (filts.isEmpty() ? 0 : &t),
                                  &dlg)) {
            qDebug("Shouldn't happen %s:%d", __FILE__, __LINE__);
            return retstrl;
        }
    }
    if (pwd && !pwd->isEmpty()) {
        FSRef fsref;
        if (qt_mac_create_fsref(*pwd, &fsref) == noErr) {
            AEDesc desc;
            if (AECreateDesc(typeFSRef, &fsref, sizeof(FSRef), &desc) == noErr)
                NavCustomControl(dlg, kNavCtlSetLocation, (void*)&desc);
        }
    }

    QWidget modal_widg(parent, Qt::Sheet);
    if (options.modality == kWindowModalityWindowModal || options.modality == kWindowModalityAppModal) {
        //simulate modality prepare:
        modal_widg.createWinId();
        QApplicationPrivate::enterModal(&modal_widg);
        QApplicationPrivate::native_modal_dialog_active = true;
    }

    NavDialogRun(dlg);

    if (selectedFilter) {
        NavMenuItemSpec navSpec;
        bzero(&navSpec, sizeof(NavMenuItemSpec));
        qt_mac_filter_name *sel_filt_name = qt_mac_make_filters_list(*selectedFilter).at(0);
        for (int i = 0; i < filts.count(); ++i) {
            const qt_mac_filter_name *filter = filts.at(i);
            if (sel_filt_name->description == filter->description
                    && sel_filt_name->regxp == filter->regxp
                    && sel_filt_name->filter == filter->filter) {
                navSpec.menuType = i;
                break;
            }
        }
        NavCustomControl(dlg, kNavCtlSelectCustomType, &navSpec);
    }

    if (options.modality == kWindowModalityWindowModal || options.modality == kWindowModalityAppModal) {
        //simulate modality prepare:
        while (g_nav_blocking)
            qApp->processEvents(QEventLoop::WaitForMoreEvents);
        QApplicationPrivate::leaveModal(&modal_widg);
        QApplicationPrivate::native_modal_dialog_active = false;
    }

    if (!(NavDialogGetUserAction(dlg) &
          (kNavUserActionOpen | kNavUserActionChoose | kNavUserActionNewFolder))) {
        NavDialogDispose(dlg);
        return retstrl;
    }
    NavReplyRecord ret;
    NavDialogGetReply(dlg, &ret);
    NavDialogDispose(dlg);

    long count;
    err = AECountItems(&(ret.selection), &count);
    if (!ret.validRecord || err != noErr || !count) {
        NavDisposeReply(&ret);
        return retstrl;
    }

    for (long index = 1; index <= count; index++) {
        FSRef ref;
        err = AEGetNthPtr(&(ret.selection), index, typeFSRef, 0, 0, &ref, sizeof(ref), 0);
        if (err != noErr)
            break;

        if (!str_buffer) {
            qAddPostRoutine(cleanup_str_buffer);
            str_buffer = (UInt8 *)malloc(1024);
        }
        FSRefMakePath(&ref, str_buffer, 1024);
        retstrl.append(QString::fromUtf8((const char *)str_buffer));
    }
    NavDisposeReply(&ret);
    if (selectedFilter)
        *selectedFilter = filts.at(t.index)->filter;
    while (!filts.isEmpty())
        delete filts.takeFirst();
    return retstrl;
}
#else
static QStringList qt_mac_get_open_file_names_cocoa(const QFileDialogArgs &args, QString *pwd, QString *selectedFilter)
{
    QFileDialog d;
    d.setWindowTitle(args.caption);
    d.setNameFilter(args.filter);
    d.setOptions(args.options);
    d.setFileMode(args.mode);
    d.setConfirmOverwrite(!(args.options & QFileDialog::DontConfirmOverwrite));
    if (pwd)
        d.selectFile(*pwd);
    if (selectedFilter)
        d.selectNameFilter(*selectedFilter);

    if (d.exec() == QDialog::Accepted){
        if (selectedFilter)
            *selectedFilter = d.selectedNameFilter();
        return d.selectedFiles();
    }
    return QStringList();
}
#endif

QStringList qt_mac_get_open_file_names(
    const QFileDialogArgs &args, QString *pwd, QString *selectedFilter)
{
    return
#ifndef QT_MAC_USE_COCOA
        qt_mac_get_open_file_names_carbon(args, pwd, selectedFilter);
#else
        qt_mac_get_open_file_names_cocoa(args, pwd, selectedFilter);
#endif
}


#ifndef QT_MAC_USE_COCOA
static QString qt_mac_get_save_file_name_carbon(const QFileDialogArgs &args, QString *pwd,
                                         QString *selectedFilter)
{
    QWidget *parent = args.parent;
    OSErr err;
    QString retstr;
    NavDialogCreationOptions options;
    NavGetDefaultDialogCreationOptions(&options);
    static const int w = 450, h = 350;
    if (args.options & QFileDialog::DontConfirmOverwrite)
        options.optionFlags |= kNavDontConfirmReplacement;
    options.modality = kWindowModalityAppModal;
    options.location.h = options.location.v = -1;
    if (!args.directory.isEmpty())
        options.saveFileName = QCFString::toCFStringRef(args.selection);
    if (!args.caption.isEmpty())
        options.windowTitle = QCFString::toCFStringRef(args.caption);
    if (parent && parent->isVisible()) {
        WindowClass wclass;
        GetWindowClass(qt_mac_window_for(parent), &wclass);
        parent = parent->window();
        QString s = parent->windowTitle();
        options.clientName = CFStringCreateWithCharacters(0, (UniChar *)s.unicode(), s.length());
        options.location.h = (parent->x() + (parent->width() / 2)) - (w / 2);
        options.location.v = (parent->y() + (parent->height() / 2)) - (h / 2);

        QRect r = QApplication::desktop()->screenGeometry(
                QApplication::desktop()->screenNumber(parent));
        if (options.location.h + w > r.right())
            options.location.h -= (options.location.h + w) - r.right() + 10;
        if (options.location.v + h > r.bottom())
            options.location.v -= (options.location.v + h) - r.bottom() + 10;
#if 0
    } else if (QWidget *p = qApp->mainWidget()) {
        static int last_screen = -1;
        int scr = QApplication::desktop()->screenNumber(p);
        if (last_screen != scr) {
            QRect r = QApplication::desktop()->screenGeometry(scr);
            options.location.h = (r.x() + (r.width() / 2)) - (w / 2);
            options.location.v = (r.y() + (r.height() / 2)) - (h / 2);
        }
#endif
    }

    QList<qt_mac_filter_name*> filts = qt_mac_make_filters_list(args.filter);
    qt_mac_nav_filter_type t;
    t.saveDialog = true;
    t.index = 0;
    t.filts = &filts;
    if (filts.count() > 1) {
        int i = 0;
        CFStringRef *arr = static_cast<CFStringRef *>(malloc(sizeof(CFStringRef) * filts.count()));
        for (QList<qt_mac_filter_name*>::const_iterator it = filts.constBegin();
             it != filts.constEnd(); ++it)
            arr[i++] = QCFString::toCFStringRef((*it)->description);
        options.popupExtension = CFArrayCreate(0, reinterpret_cast<const void **>(arr), filts.count(), 0);
    }

    NavDialogRef dlg;
    if (NavCreatePutFileDialog(&options, 'cute', kNavGenericSignature, make_navProcUPP(),
                               static_cast<void *>(filts.isEmpty() ? 0 : &t), &dlg)) {
        qDebug("Shouldn't happen %s:%d", __FILE__, __LINE__);
        return retstr;
    }
    if (pwd && !pwd->isEmpty()) {
        FSRef fsref;
        if (qt_mac_create_fsref(*pwd, &fsref) == noErr) {
            AEDesc desc;
            if (AECreateDesc(typeFSRef, &fsref, sizeof(FSRef), &desc) == noErr)
                NavCustomControl(dlg, kNavCtlSetLocation, (void*)&desc);
        }
    }
    NavDialogRun(dlg);

    if (selectedFilter) {
        NavMenuItemSpec navSpec;
        bzero(&navSpec, sizeof(NavMenuItemSpec));
        qt_mac_filter_name *sel_filt_name = qt_mac_make_filters_list(*selectedFilter).at(0);
        for (int i = 0; i < filts.count(); ++i) {
            const qt_mac_filter_name *filter = filts.at(i);
            if (sel_filt_name->description == filter->description
                    && sel_filt_name->regxp == filter->regxp
                    && sel_filt_name->filter == filter->filter) {
                navSpec.menuType = i;
                break;
            }
        }
        NavCustomControl(dlg, kNavCtlSelectCustomType, &navSpec);
    }
    if (NavDialogGetUserAction(dlg) != kNavUserActionSaveAs) {
        NavDialogDispose(dlg);
        return retstr;
    }
    NavReplyRecord ret;
    NavDialogGetReply(dlg, &ret);
    NavDialogDispose(dlg);

    long count;
    err = AECountItems(&(ret.selection), &count);
    if (!ret.validRecord || err != noErr || !count) {
        NavDisposeReply(&ret);
        return retstr;
    }

    AEKeyword        keyword;
    DescType    type;
    Size        size;
    FSRef ref;
    err = AEGetNthPtr(&(ret.selection), 1, typeFSRef, &keyword,
                      &type, &ref, sizeof(ref), &size);
    if (err == noErr) {
        if (!str_buffer) {
            qAddPostRoutine(cleanup_str_buffer);
            str_buffer = (UInt8 *)malloc(1024);
        }
        FSRefMakePath(&ref, str_buffer, 1024);
        retstr = QString::fromUtf8((const char *)str_buffer);
        //now filename
        CFStringGetCString(ret.saveFileName, (char *)str_buffer, 1024, kCFStringEncodingUTF8);
        retstr += QLatin1String("/") + QString::fromUtf8((const char *)str_buffer);
    }
    NavDisposeReply(&ret);
    if (selectedFilter)
        *selectedFilter = filts.at(t.index)->filter;
    while (!filts.isEmpty())
        delete filts.takeFirst();
    return retstr;
}
#else
static QString qt_mac_get_save_file_name_cocoa(const QFileDialogArgs &args, QString *pwd,
                                        QString *selectedFilter)
{
    QFileDialog d;
    d.setAcceptMode(QFileDialog::AcceptSave);
    d.setWindowTitle(args.caption);
    d.setNameFilter(args.filter);
    d.setOptions(args.options);
    d.setFileMode(args.mode);
    d.setConfirmOverwrite(!(args.options & QFileDialog::DontConfirmOverwrite));
    if (pwd)
        d.selectFile(*pwd);
    if (selectedFilter)
        d.selectNameFilter(*selectedFilter);

    if (d.exec() == QDialog::Accepted){
        if (selectedFilter)
            *selectedFilter = d.selectedNameFilter();
        return d.selectedFiles().at(0);
    }
    return QString();
}
#endif

QString qt_mac_get_save_file_name(
        const QFileDialogArgs &args, QString *pwd, QString *selectedFilter)
{
    return
#ifndef QT_MAC_USE_COCOA
        qt_mac_get_save_file_name_carbon(args, pwd, selectedFilter);
#else
        qt_mac_get_save_file_name_cocoa(args, pwd, selectedFilter);
#endif
}

QT_END_NAMESPACE

#endif // QT_NO_FILEDIALOG

