/*  This file is part of the KDE project.

Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).

This library is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 2.1 or 3 of the License.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mediaobject.h"
#include "audiooutput.h"

#include <QtCore/QVector>
#include <QtCore/QTimerEvent>
#include <QtCore/QTimer>
#include <QtCore/QTime>
#include <QtCore/QLibrary>

#include <QtCore/QDebug>

#define TIMER_INTERVAL 16 //... ms for the timer that polls the current state (we use the multimedia timer

#define WAVEHEADER_OFFSET_FORMATTAG        20
#define WAVEHEADER_OFFSET_CHANNELS         22
#define WAVEHEADER_OFFSET_SAMPLESPERSEC    24
#define WAVEHEADER_OFFSET_AVGBYTESPERSEC   28
#define WAVEHEADER_OFFSET_BLOCKALIGN       32
#define WAVEHEADER_OFFSET_BITSPERSAMPLE    34
#define WAVEHEADER_OFFSET_DATA             44
#define WAVEHEADER_SIZE                    WAVEHEADER_OFFSET_DATA

QT_BEGIN_NAMESPACE

namespace Phonon
{
    namespace WaveOut
    {        
        static int buffer_size = 16 * 1024;

        class WorkerThread : public QThread
        {
         Q_OBJECT
         public slots:
              void stream(QIODevice *file, QByteArray *buffer, bool *finished);
        };

        void WorkerThread::stream(QIODevice *ioStream, QByteArray *buffer, bool *finished)
        {
            (*finished) = false;
            qint64 i = ioStream->read(buffer->data(), buffer_size);
            buffer->resize(i);
            (*finished) = true;
        }


        void CALLBACK MediaObject::WaveOutCallBack(HWAVEOUT m_hWaveOut, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
        {
            Q_UNUSED(m_hWaveOut);
            Q_UNUSED(dwInstance);
            Q_UNUSED(dwParam2);

            switch(uMsg)
            {
            case WOM_OPEN:
                break;
            case WOM_DONE:
                {
                    WAVEHDR *waveHeader = (WAVEHDR*)dwParam1;
                    MediaObject* mediaObject = reinterpret_cast<MediaObject *>(waveHeader->dwUser);
                    if (mediaObject)
                        mediaObject->swapBuffers();
                }
                break;
            case WOM_CLOSE:
                break;
            }
        }


        MediaObject::MediaObject(QObject *parent) : m_file(0), m_stream(0),
                                                    m_hWaveOut(0), m_nextBufferIndex(1), 
                                                    m_mediaSize(-1), m_bufferingFinished(0),
                                                    m_paused(0), m_tickInterval(100),
                                                    m_hasNextSource(0), m_hasSource(0),
                                                    m_sourceIsValid(0), m_errorType(Phonon::NoError),
                                                    m_currentTime(0), m_transitionTime(0),
                                                    m_prefinishMark(0), m_stopped(0), m_bufferPrepared(0), m_volume(100)
        {
            m_thread = new WorkerThread();
            m_thread->setParent(this);
            connect(this, SIGNAL(outOfData(QIODevice*, QByteArray*, bool*)), m_thread, SLOT(stream(QIODevice*, QByteArray*, bool*)));
            m_thread->start();
            m_thread->setPriority(QThread::LowestPriority);
            m_soundBuffer1.waveHeader = new WAVEHDR;
            m_soundBuffer2.waveHeader = new WAVEHDR;
            setParent(parent);
        }

        MediaObject::~MediaObject()
        {
            delete m_soundBuffer1.waveHeader;
            delete m_soundBuffer2.waveHeader;
            deleteValidWaveOutDevice();
            unPrepareBuffers();
        }

        Phonon::State MediaObject::state() const
        {

           return m_state;
        }

        bool MediaObject::hasVideo() const
        {
            return false;
        }

        bool MediaObject::isSeekable() const
        {
            return true;
        }

        qint64 MediaObject::totalTime() const
        {
            return m_totalTime;
        }

        qint64 MediaObject::currentTime() const
        {
            //this handles inaccuracy when stopping on a title
            return m_currentTime;
        }

        qint32 MediaObject::tickInterval() const
        {
            return m_tickInterval;
        }

        void MediaObject::setTickInterval(qint32 newTickInterval)
        {
            m_tickInterval = newTickInterval;
        }

        void MediaObject::pause()
        {
            if (!(m_state == Phonon::PlayingState))
            {
                setError(Phonon::NormalError, QLatin1String("cannot pause while not playing"));
            }
            if (!m_paused) {
                m_paused = true;
                setState(Phonon::PausedState);
                if (!(waveOutPause(m_hWaveOut) == MMSYSERR_NOERROR))
                {
                    setError(Phonon::NormalError, QLatin1String("cannot pause (system error)"));
                }
            }
        }

        void MediaObject::stop()
        {
            setState(Phonon::StoppedState);
            m_stopped = true;
            if (!(waveOutReset(m_hWaveOut) == MMSYSERR_NOERROR))
                setError(Phonon::NormalError, QLatin1String("cannot stop (system error)"));
        }

        void MediaObject::play()
        {
            if  ((m_state == Phonon::LoadingState) ||
                 (m_state == Phonon::BufferingState) ||
                 (m_state == Phonon::ErrorState)) {
                    setError(Phonon::FatalError, QLatin1String("illegale state for playback"));
                    return;
            }

            if (m_sourceIsValid) {
                m_stopped = false;
                setState(Phonon::PlayingState);
                if (!m_paused) {
                    playBuffer(m_soundBuffer1.waveHeader);
                    playBuffer(m_soundBuffer2.waveHeader);
                } else {
                    if (!(waveOutRestart(m_hWaveOut) == MMSYSERR_NOERROR))
                        setError(Phonon::NormalError, QLatin1String("cannot resume (system)"));
                }
            } else {
                setError(Phonon::FatalError, QLatin1String("cannot playback invalid source"));
            }
        }

        QString MediaObject::errorString() const
        {
            
            return m_errorString;
        }

        Phonon::ErrorType MediaObject::errorType() const
        {
            return Phonon::ErrorType();
        }

        qint32 MediaObject::prefinishMark() const
        {
            return m_prefinishMark;
        }

        void MediaObject::setPrefinishMark(qint32 newPrefinishMark)
        {
            m_prefinishMark = newPrefinishMark;
        }

        qint32 MediaObject::transitionTime() const
        {
            return m_transitionTime;
        }

        void MediaObject::setTransitionTime(qint32 time)
        {
           m_transitionTime = time;
        }

        qint64 MediaObject::remainingTime() const
        {
            return m_totalTime - m_currentTime;
        }

        Phonon::MediaSource MediaObject::source() const
        {
            return Phonon::MediaSource();
        }

        void MediaObject::setNextSource(const Phonon::MediaSource &source)
        {
            m_nextSource = source;
            m_hasNextSource = true;
        }

        void MediaObject::setSource(const Phonon::MediaSource &source)
        {
            if (m_state == Phonon::PlayingState)
            {
                setError(Phonon::NormalError, QLatin1String("source changed while playing"));
                stop();
            }

            m_source = source;
            m_hasSource = true;
            m_sourceIsValid = false;

            emit currentSourceChanged(source);

            if (source.type() == Phonon::MediaSource::LocalFile) {
                if (!openWaveFile(source.fileName())) {
                  setError(Phonon::FatalError, QLatin1String("cannot open media file"));
                  return ;
                }
            } else if (source.type() == Phonon::MediaSource::Stream) {
                m_stream = 0;
                setError(Phonon::FatalError, QLatin1String("streams not supported"));
                return ;
            } else {
                setError(Phonon::FatalError, QLatin1String("type of source not supported"));
                return ;
            }
            setState(Phonon::LoadingState);

            if (!readHeader())
                setError(Phonon::FatalError, QLatin1String("invalid header"));
            else if (!getWaveOutDevice())
                setError(Phonon::FatalError, QLatin1String("No waveOut device available"));
            else if (!fillBuffers())
                setError(Phonon::FatalError, QLatin1String("no data for buffering"));
            else if (!prepareBuffers())
                setError(Phonon::FatalError, QLatin1String("cannot prepare buffers"));
            else
                m_sourceIsValid = true;

            if (m_sourceIsValid)
                setState(Phonon::StoppedState);
        }

        void MediaObject::seek(qint64 time)
        {
            if ((time > 0) && (time < m_totalTime)) {
                stop();
                while (!m_bufferingFinished) {
                  setState(Phonon::BufferingState);
                  Sleep(20);
                }
                m_stream->seek(WAVEHEADER_SIZE + time * m_waveFormatEx.nSamplesPerSec * m_waveFormatEx.wBitsPerSample * m_waveFormatEx.nChannels / 8 / 1000);
                play();
            } else {
                setError(Phonon::NormalError, QLatin1String("seeking out of range"));
            }
        }

        void MediaObject::unPrepareBuffers()
        {
            if (m_bufferPrepared)
                if ((waveOutUnprepareHeader(m_hWaveOut, m_soundBuffer1.waveHeader, sizeof(WAVEHDR)) == MMSYSERR_NOERROR) &&
                    (waveOutUnprepareHeader(m_hWaveOut, m_soundBuffer2.waveHeader, sizeof(WAVEHDR)) == MMSYSERR_NOERROR))
                    setError(Phonon::NormalError, QLatin1String("cannot unprepare buffer"));
            m_bufferPrepared = false;
        }

        bool MediaObject::prepareBuffers()
        {

            ZeroMemory((void*)m_soundBuffer1.waveHeader, sizeof(WAVEHDR));
            m_soundBuffer1.waveHeader->lpData = m_soundBuffer1.data.data();
            m_soundBuffer1.waveHeader->dwBufferLength = m_soundBuffer1.data.size();
            m_soundBuffer1.waveHeader->dwUser = (DWORD_PTR) this;
            m_soundBuffer1.waveHeader->dwFlags = WHDR_BEGINLOOP;

            ZeroMemory((void*)m_soundBuffer2.waveHeader, sizeof(WAVEHDR));
            m_soundBuffer2.waveHeader->lpData = m_soundBuffer2.data.data();
            m_soundBuffer2.waveHeader->dwBufferLength = m_soundBuffer1.data.size();
            m_soundBuffer2.waveHeader->dwUser = (DWORD_PTR) this;
            m_soundBuffer2.waveHeader->dwFlags = WHDR_BEGINLOOP;

            m_bufferPrepared = true;

            return (waveOutPrepareHeader(m_hWaveOut, m_soundBuffer1.waveHeader, sizeof(WAVEHDR)) == MMSYSERR_NOERROR)
                && (waveOutPrepareHeader(m_hWaveOut, m_soundBuffer2.waveHeader, sizeof(WAVEHDR)) == MMSYSERR_NOERROR);
        }

        void MediaObject::deleteValidWaveOutDevice()
        {
            if (m_hWaveOut) {
                unPrepareBuffers();
                if (!(waveOutClose(m_hWaveOut)  == MMSYSERR_NOERROR))
                    setError(Phonon::NormalError, QLatin1String("cannot close wave device"));
            }
        }

        bool MediaObject::getWaveOutDevice()
        {
            deleteValidWaveOutDevice();

            for(UINT deviceId = 0; deviceId < waveOutGetNumDevs(); deviceId++)
            {
                if(deviceId == waveOutGetNumDevs())
                    return false;
                if(waveOutOpen(&m_hWaveOut, WAVE_MAPPER, &m_waveFormatEx, (DWORD)WaveOutCallBack, 0, CALLBACK_FUNCTION) == MMSYSERR_NOERROR)
                    return true;
            }
            return false;
        }

        bool  MediaObject::openWaveFile(QString fileName)
        {
            if (m_file)
                delete m_file;
            m_file = new QFile(fileName);
            m_file->setParent(this);
            m_stream = m_file;
            m_mediaSize = m_file->size();
            return (m_file->open(QIODevice::ReadOnly));
        }

        bool MediaObject::readHeader()
        {
            QByteArray header = m_stream->read(WAVEHEADER_SIZE);

            if (header.size() == WAVEHEADER_SIZE) {

                m_waveFormatEx.wFormatTag         = *((WORD* )(header.data() + WAVEHEADER_OFFSET_FORMATTAG     ));
                m_waveFormatEx.nChannels          = *((WORD* )(header.data() + WAVEHEADER_OFFSET_CHANNELS      ));
                m_waveFormatEx.nSamplesPerSec     = *((DWORD*)(header.data() + WAVEHEADER_OFFSET_SAMPLESPERSEC ));
                m_waveFormatEx.nAvgBytesPerSec    = *((DWORD*)(header.data() + WAVEHEADER_OFFSET_AVGBYTESPERSEC));
                m_waveFormatEx.nBlockAlign        = *((WORD* )(header.data() + WAVEHEADER_OFFSET_BLOCKALIGN    ));
                m_waveFormatEx.wBitsPerSample     = *((WORD* )(header.data() + WAVEHEADER_OFFSET_BITSPERSAMPLE ));

                if (m_mediaSize > 0)
                   m_totalTime = ((m_mediaSize - WAVEHEADER_SIZE) * 8 * 1000) / m_waveFormatEx.nSamplesPerSec / m_waveFormatEx.wBitsPerSample / m_waveFormatEx.nChannels;
                else
                  m_totalTime = -1;
                return true;
            } else {
                return false;
            }
        }
        
        bool MediaObject::fillBuffers()
        {
            
            setState(Phonon::BufferingState);
            m_soundBuffer1.data = m_stream->read(buffer_size);
            m_soundBuffer2.data = m_stream->read(buffer_size);
            setState(Phonon::StoppedState);

            m_bufferingFinished = true;

            if (!(m_soundBuffer1.data.size() > 0))
                setError(Phonon::NormalError, QLatin1String("cannot read source"));
            return true;
        }

        void MediaObject::setState(Phonon::State newState)
        {
            if (m_state == newState)
                return;
            emit stateChanged(newState, m_state);
            m_state = newState;
        }

        void MediaObject::setError(ErrorType errorType, QString errorMessage)
        {
            m_errorType = errorType;
            setState(Phonon::ErrorState);
            m_errorString = errorMessage;
        }

        void MediaObject::setAudioOutput(QObject *audioOutput)
        {

            m_audioOutput = qobject_cast<AudioOutput*>(audioOutput);

            if (m_audioOutput) {
                m_volume = m_audioOutput->volume();
                connect(m_audioOutput, SIGNAL(volumeChanged(qreal)), this, SLOT(setVolume(qreal)));
            }
        }

        void MediaObject::setVolume(qreal newVolume)
        {
            m_volume = newVolume;
        }

        void MediaObject::swapBuffers()
        {
            if (m_stopped)
                return;

            while (!m_bufferingFinished) {
                setState(Phonon::BufferingState);
                qWarning() << QLatin1String("buffer underun");
                Sleep(20);
            }

            setState(Phonon::PlayingState);

            //if size == o then stop...
            if (m_nextBufferIndex) {
                int size = m_soundBuffer1.waveHeader->dwBufferLength = m_soundBuffer1.data.size();
                if (size) {
                    playBuffer(m_soundBuffer1.waveHeader);
                    emit outOfData(m_stream, &m_soundBuffer1.data, &m_bufferingFinished);
                } else {
                    m_stopped = true;
                    setState(Phonon::StoppedState);
                    emit finished();
                }
            } else {
                int size = m_soundBuffer2.waveHeader->dwBufferLength = m_soundBuffer2.data.size();
                if (size) {
                    playBuffer(m_soundBuffer2.waveHeader);
                    emit outOfData(m_stream, &m_soundBuffer2.data, &m_bufferingFinished);
                } else {
                    m_stopped = true;
                    setState(Phonon::StoppedState);
                    emit finished();
                }
            }
            m_nextBufferIndex =! m_nextBufferIndex;
        }


        void MediaObject::playBuffer(WAVEHDR *waveHeader)
        {
            if (!(waveOutWrite(m_hWaveOut, waveHeader, sizeof(WAVEHDR)) == MMSYSERR_NOERROR)) {
                setError(Phonon::FatalError, QLatin1String("cannot play sound buffer (system)"));
                m_stopped = true;
            }
        }
    }
}

QT_END_NAMESPACE

#include "mediaobject.moc"
