/*  This file is part of the KDE project.

    Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).

    This library is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 2.1 or 3 of the License.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QtCore/qglobal.h>
#ifdef QT_MAC_USE_COCOA
#import <QTKit/QTMovieLayer.h>
#endif
#undef check

#include "videowidget.h"
#include "backendheader.h"
#include "quicktimevideoplayer.h"
#include "medianode.h"
#include "medianodeevent.h"
#include "mediaobject.h"

#include <QGLWidget>
#include <QTime>

#import <AppKit/NSImage.h>
#import <QTKit/QTMovieView.h>

/////////////////////////////////////////////////////////////////////////////////////////

#ifdef QT_MAC_USE_COCOA // Rendering to a QTMovieView can only be done in Cocoa
@interface QTSharedMovieView : QTMovieView
{
@public
    Phonon::QT7::VideoFrame *m_videoFrame;
}
- (void) movieRepaint;
- (bool) setVideoFrame:(Phonon::QT7::VideoFrame &)videoFrame;
- (void) setVideoImageSharing:(BOOL)share;
@end

/////////////////////////////////////////////////////////////////////////////////////////

extern NSModalSession gSession;

@implementation QTSharedMovieView

-(id) init
{
    self = [super initWithFrame:NSZeroRect];
	if (self){
		m_videoFrame = new Phonon::QT7::VideoFrame();
        [self setControllerVisible:NO];
	}
	return self;
}

- (void) dealloc
{
   	Phonon::QT7::QuickTimeVideoPlayer *player = m_videoFrame->videoPlayer();
	if (player && self == player->m_primaryRenderingTarget)
		player->m_primaryRenderingTarget = 0;
	delete m_videoFrame;
	[super dealloc];
}

- (void) movieRepaint
{
	QTMovie *movie = [self movie];
   	Phonon::QT7::QuickTimeVideoPlayer *player = m_videoFrame->videoPlayer();
	if (movie && player){ 
		float rate = [movie rate];
		if (!rate)
			[movie setRate:1];
		player->setPrimaryRenderingCIImage(0);
		QTime t; t.start();
		while (!player->primaryRenderingCIImage() && t.elapsed() < 2000)
			qApp->processEvents(QEventLoop::ExcludeUserInputEvents | QEventLoop::ExcludeSocketNotifiers);
		[movie setRate:rate];
	} else {
		[self setNeedsDisplay:YES];
		[self display];
	}
}

- (void) drawRect:(NSRect)aRect
{
	if ([self movie])
		[super drawRect:aRect];
	else
		m_videoFrame->drawCIImage(NSRectToCGRect([self bounds]));
}

- (bool) setVideoFrame:(Phonon::QT7::VideoFrame &)videoFrame
{
	bool grabsOwnership = false;
    Phonon::QT7::QuickTimeVideoPlayer *player = videoFrame.videoPlayer();
	if (player && !player->m_primaryRenderingTarget){
		// Take ownership of the movie:
		player->m_primaryRenderingTarget = self;
    	[self setMovie:player->qtMovie()];
		grabsOwnership = true;
	}
	
	*m_videoFrame = videoFrame;
	if (!player || self != player->m_primaryRenderingTarget)
		[self setNeedsDisplay:YES];
	return grabsOwnership;
}

- (void) setVideoImageSharing:(BOOL)share
{
	if (share){
    	[self setDelegate:self];
		[self movieRepaint];
	} else
    	[self setDelegate:0];
}

- (CIImage *) view:(QTMovieView *)view willDisplayImage:(CIImage *)img
{	
	// This method is called from QTMovieView just
	// before the image will be drawn.
    Q_UNUSED(view);
    Phonon::QT7::QuickTimeVideoPlayer *player = m_videoFrame->videoPlayer();
	if (player)
		player->setPrimaryRenderingCIImage(img);
	return img;
}

#define QTSharedMovieView_TRANSPARENT(m) -(void)m:(NSEvent *)e{[[self superview] m:e];}
QTSharedMovieView_TRANSPARENT(mouseDown)
QTSharedMovieView_TRANSPARENT(mouseMoved)
QTSharedMovieView_TRANSPARENT(rightMouseDown)
QTSharedMovieView_TRANSPARENT(rightMouseDragged)
QTSharedMovieView_TRANSPARENT(otherMouseDown)
QTSharedMovieView_TRANSPARENT(otherMouseDragged)
QTSharedMovieView_TRANSPARENT(scrollWheel)
QTSharedMovieView_TRANSPARENT(keyDown)
QTSharedMovieView_TRANSPARENT(keyUp)

@end
#endif // QT_MAC_USE_COCOA

/////////////////////////////////////////////////////////////////////////////////////////

QT_BEGIN_NAMESPACE

namespace Phonon
{
namespace QT7
{

class IVideoRenderDrawWidget
{
public:
	virtual ~IVideoRenderDrawWidget(){}
	virtual void setVideoFrame(VideoFrame &) = 0;
	virtual void setDrawFrameRect(const QRect &) = 0;
	virtual void updateVideoOutputCount(int){}
	virtual void setMovieIsPaused(bool){}
};

/////////////////////////////////////////////////////////////////////////////////////////

QGLWidget *PhononSharedQGLWidget(){
	static QGLWidget *glWidget = 0;
	if (!glWidget)
		glWidget = new QGLWidget();
	return glWidget;
}

/////////////////////////////////////////////////////////////////////////////////////////

class RenderOpenGL : public QGLWidget, public IVideoRenderDrawWidget
{    
public:
    VideoFrame m_currentFrame;
    QRect m_drawFrameRect;

    RenderOpenGL(QWidget *parent, const QGLFormat &format) : QGLWidget(format, parent, PhononSharedQGLWidget())
    {
        setAutoFillBackground(false);        
    }
    
    void initializeGL()
    {
        glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
    }
    
    void resizeGL(int w, int h)
    {
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        glViewport(0, 0, GLsizei(w), GLsizei(h));
        gluOrtho2D(0, GLsizei(w), 0, GLsizei(h));    
        updateGL();
    }

    void paintGL()
    {
        glClear(GL_COLOR_BUFFER_BIT);
        m_currentFrame.drawCVTexture(m_drawFrameRect);
    }

    void setVideoFrame(VideoFrame &frame)
    {
        m_currentFrame = frame;
        if (isVisible()){
            makeCurrent();
            paintGL();
            swapBuffers();
        }
    }
    
    void setDrawFrameRect(const QRect &rect)
    {
        m_drawFrameRect = rect;
    }
};

/////////////////////////////////////////////////////////////////////////////////////////

class RenderQTMovieView : public QWidget, public IVideoRenderDrawWidget
{    
public:
#ifdef QT_MAC_USE_COCOA
    QTSharedMovieView *m_movieView;
    QTSharedMovieView *m_slaveView;
	bool m_useSlaveView;
	int m_removeSlaveViewDelay;
	int m_videoOutputCount;
	QRect m_drawRect;
	VideoFrame m_videoFrame;
	
    RenderQTMovieView(QWidget *parent) : QWidget(parent)
    {
		PhononAutoReleasePool pool;
        setAutoFillBackground(false);
		m_useSlaveView = false;
		m_videoOutputCount = 0;
        m_movieView = [[QTSharedMovieView alloc] init];
        [(NSView *)winId() addSubview:m_movieView];
        m_slaveView = [[QTSharedMovieView alloc] init];
        [(NSView *)winId() addSubview:m_slaveView];
    }

    ~RenderQTMovieView()
    {
        [m_movieView release];
        [m_slaveView release];
    }

    void paintEvent(QPaintEvent *)
    {
		QPainter p(this);
		p.fillRect(rect(), Qt::black);
    }

	bool needSlaveRendering()
	{
		// Slave rendering means that we'll draw frame images our
		// selves, bypassing QTMovieView. We need to do this if we
		// render frames to more than one widget, or if the frames
		// contains color adjustments (then CIFilters needs to be applied).
		// QTSharedMovieView-s that has no movie attached will always
		// draw the images directly by using a shared image.
		return (m_videoOutputCount > 1 || m_videoFrame.hasColorAdjustments()) && [m_movieView movie];
	}
	
	void updateVideoOutputCount(int count)
	{
		m_videoOutputCount = count;
		setVideoFrame(m_videoFrame);
	}

	void setMovieIsPaused(bool paused)
	{
		Q_UNUSED(paused);
		if (m_useSlaveView && !needSlaveRendering()){
			// A good time for a switch back to normal:
			m_useSlaveView = false;
	        [m_movieView setVideoImageSharing:NO];
			setDrawFrameRect(m_drawRect);
		}
	}
	
    void setVideoFrame(VideoFrame &frame)
    {
		m_videoFrame = frame;
		if (!m_useSlaveView && needSlaveRendering()){
			m_useSlaveView = true;
		    [m_movieView setVideoImageSharing:YES];
			[m_movieView setVideoFrame:frame];
			[m_slaveView setVideoFrame:frame];
			setDrawFrameRect(m_drawRect);			
		} else {
			[m_movieView setVideoFrame:frame];
			if (m_useSlaveView)
				[m_slaveView setVideoFrame:frame];
		}
    }

    void setDrawFrameRect(const QRect &rect)
    {
		PhononAutoReleasePool pool;
		m_drawRect = rect;
        NSRect frame;
        frame.origin.x = rect.x();
        frame.origin.y = rect.y();

		QTSharedMovieView *hidden = m_useSlaveView ? m_movieView : m_slaveView;
	    QTSharedMovieView *visible = m_useSlaveView ? m_slaveView : m_movieView;
	    frame.size.width = rect.width();
	    frame.size.height = rect.height();
	    [visible setFrame:frame];
		frame.size.width = 1;
		frame.size.height = 1;
	    [hidden setFrame:frame];
    }

#else // QT_MAC_USE_COCOA == false
	RenderQTMovieView(QWidget *){}
	void setVideoFrame(VideoFrame &){}
	void setDrawFrameRect(const QRect &){}
#endif
};

/////////////////////////////////////////////////////////////////////////////////////////

class RenderCIImageProducer : public QWidget, public IVideoRenderDrawWidget
{    
public:
#if defined(QT_MAC_USE_COCOA) && !defined(QT_ALLOW_QUICKTIME)
    QTSharedMovieView *m_movieView;
	VideoFrame m_videoFrame;
	
    RenderCIImageProducer(QWidget * /*parent*/) : QWidget(0/*parent*/, Qt::Window)
    {
		PhononAutoReleasePool pool;
        m_movieView = [[QTSharedMovieView alloc] init];
        [(NSView *)winId() addSubview:m_movieView];
	    [m_movieView setVideoImageSharing:YES];
        NSRect frame;
        frame.origin.x = 0;
        frame.origin.y = 0;
		frame.size.width = 1;
		frame.size.height = 1;
	    [m_movieView setFrame:frame];
    }

    ~RenderCIImageProducer()
    {
        [m_movieView release];
    }

	void updateVideoOutputCount(int count)
	{
		Q_UNUSED(count);
		setVideoFrame(m_videoFrame);
	}

    void setVideoFrame(VideoFrame &frame)
    {
		m_videoFrame = frame;
		if ([m_movieView setVideoFrame:frame]){
			show();
	    	[m_movieView setVideoImageSharing:YES];
		}
    }

    void setDrawFrameRect(const QRect &)
    {
		return;
    }

#else // (QT_MAC_USE_COCOA == false || QT_ALLOW_QUICKTIME == true
	RenderCIImageProducer(QWidget *){}
	void setVideoFrame(VideoFrame &){}
	void setDrawFrameRect(const QRect &){}
#endif
};

/////////////////////////////////////////////////////////////////////////////////////////

class RenderQTMovieLayer : public QWidget, public IVideoRenderDrawWidget
{    
public:
#ifdef QT_MAC_USE_COCOA
    QTMovieLayer *m_movieLayer;

    RenderQTMovieLayer(QWidget *parent) : QWidget(parent)
    {
		PhononAutoReleasePool pool;
        setAutoFillBackground(false);
        m_movieLayer = 0;
        [(NSView *)winId() setWantsLayer:YES];
    }

    void setVideoFrame(VideoFrame &frame)
    {
        QuickTimeVideoPlayer *player = frame.videoPlayer();
        if (!player || player->qtMovie() == [m_movieLayer movie])
            return;

        if (m_movieLayer)
            [m_movieLayer setMovie:player->qtMovie()];
        else {
            m_movieLayer = [QTMovieLayer layerWithMovie:player->qtMovie()];
            [(NSView *)winId() setLayer:m_movieLayer];            
        }
    }

    void setDrawFrameRect(const QRect &rect)
    {
        m_movieLayer.frame.origin.x = rect.x();
        m_movieLayer.frame.origin.y = rect.y();
        m_movieLayer.frame.size.width = rect.width();
        m_movieLayer.frame.size.height = rect.height();
    }

#else // QT_MAC_USE_COCOA == false
	RenderQTMovieLayer(QWidget *){}
	void setVideoFrame(VideoFrame &){}
	void setDrawFrameRect(const QRect &){}
#endif
};

/////////////////////////////////////////////////////////////////////////////////////////

class VideoRenderWidget : public QWidget
{    
public:
    enum RenderSystem {	RS_NoRendering	= 0,
						RS_QGLWidget	= 1,
						RS_QPainter		= 2,
						RS_CIImage		= 3,
						RS_CVTexture	= 4,
						RS_QImage		= 5,
						RS_QTMovieView	= 6,
						RS_QTMovieLayer = 7
	} m_renderSystem;
    
    VideoFrame m_currentFrame;
    QRect m_movieFrameRect;
    QRect m_drawFrameRect;
    Phonon::VideoWidget::ScaleMode m_scaleMode;
    Phonon::VideoWidget::AspectRatio m_aspect;    
	IVideoRenderDrawWidget *m_renderDrawWidget;
    
    qreal m_brightness;
    qreal m_contrast;
    qreal m_hue;
    qreal m_saturation;
    qreal m_opacity;

    VideoRenderWidget() : QWidget(0),
        m_scaleMode(Phonon::VideoWidget::FitInView), m_aspect(Phonon::VideoWidget::AspectRatioAuto)
    {
		PhononAutoReleasePool pool;
        m_brightness = 0;
        m_contrast = 0;
        m_hue = 0;
        m_saturation = 0;
        m_opacity = 1;
		m_renderDrawWidget = 0;
		m_renderSystem = RS_NoRendering;
				
        setRenderSystem(selectBestRenderSystem());       
        setAutoFillBackground(false);        
        updateDrawFrameRect();
    }
    
    RenderSystem selectBestRenderSystem(){
        if (parentWidget() && parentWidget()->testAttribute(Qt::WA_DontShowOnScreen))
            return RS_QPainter;
        else{
#if defined(QT_ALLOW_QUICKTIME)
            return RS_QGLWidget;
#elif defined(QT_MAC_USE_COCOA)
            return RS_QTMovieView;
#else
			return RS_NoRendering;
#endif
        }
    }
    
    void setRenderSystem(RenderSystem renderSystem){
		QString userSystem = qgetenv("PHONON_RENDER_SYSTEM");
	    if (!userSystem.isEmpty())
			renderSystem = RenderSystem(userSystem.toInt());
		
        if (m_renderSystem == renderSystem)
            return;
      
        m_renderSystem = renderSystem;

		if (m_renderDrawWidget){
        	delete m_renderDrawWidget;
			m_renderDrawWidget = 0;
		}

        switch (m_renderSystem){
            case RS_QGLWidget:{
			    QGLFormat format = QGLFormat::defaultFormat();
			    format.setSwapInterval(1); // Vertical sync (avoid tearing)
			    m_renderDrawWidget = new RenderOpenGL(this, format);
                break;}
            case RS_QTMovieView:{
			    m_renderDrawWidget = new RenderQTMovieView(this);
                break;}
            case RS_QTMovieLayer:{
			    m_renderDrawWidget = new RenderQTMovieLayer(this);
				break;}
            case RS_QPainter:
			case RS_CIImage:
			case RS_CVTexture:
			case RS_QImage:
				m_renderDrawWidget = new RenderCIImageProducer(this);
				break;
          	case RS_NoRendering:
                break;
        }

		if (m_renderDrawWidget)
			m_renderDrawWidget->setVideoFrame(m_currentFrame);
    }
    
    QSize sizeHint() const
    {
        return m_movieFrameRect.size();
    }

    void resizeEvent(QResizeEvent */*event*/)
    {
        updateDrawFrameRect();
		if (m_renderDrawWidget)
			dynamic_cast<QWidget *>(m_renderDrawWidget)->resize(size());
    }     

    bool event(QEvent *event)
    {
        switch (event->type()){
            // Try to detect if one of this objects
            // anchestors might have changed:
            case QEvent::Resize:
            case QEvent::Move:
            case QEvent::ParentChange:
                setRenderSystem(selectBestRenderSystem());
                break;
            case QEvent::Paint:{
                float opacity = parentWidget() ? parentWidget()->windowOpacity() : 1;
                switch (m_renderSystem){
                    case RS_QPainter:{
                        QPainter p(this);
                        if (p.paintEngine()->type() == QPaintEngine::OpenGL)
                         	m_currentFrame.drawCVTexture(m_drawFrameRect, opacity, false);
                        else
							m_currentFrame.drawQImage(&p, m_drawFrameRect, opacity, false);
                        break; }
					case RS_CIImage:
                        m_currentFrame.drawCIImage(m_drawFrameRect, opacity, false);
						break;
 					case RS_CVTexture:
	                   m_currentFrame.drawCVTexture(m_drawFrameRect, opacity, false);
					   break;
 					case RS_QImage:{
                        QPainter p(this);
						m_currentFrame.drawQImage(&p, m_drawFrameRect, opacity, false);
                        break; }
                   	case RS_QGLWidget:
                    case RS_QTMovieView:
                    case RS_QTMovieLayer:
                    case RS_NoRendering:
                        // draw in separate widget
                        break;
                }
                break; }
            default:
                break;
        }

        return QWidget::event(event);
    }

    void setVideoFrame(VideoFrame &frame)
    {
		PhononAutoReleasePool pool;
        m_currentFrame = frame;
        m_currentFrame.setColors(m_brightness, m_contrast, m_hue, m_saturation);

		if (m_renderDrawWidget)
			m_renderDrawWidget->setVideoFrame(m_currentFrame);

        switch (m_renderSystem){
            case RS_QGLWidget:
            case RS_QTMovieView:
            case RS_QTMovieLayer:
            case RS_NoRendering:
                break;
			case RS_CIImage:
			case RS_CVTexture:
			case RS_QImage:
            case RS_QPainter:
                update();
                break;
        }
    }
    
    void updateVideoFrame()
    {
        setVideoFrame(m_currentFrame);
    }
    
    void setMovieRect(const QRect &mrect)
    {
        if (mrect == m_movieFrameRect)
            return;
        m_movieFrameRect = mrect;
        updateDrawFrameRect();
        updateGeometry();
        qApp->processEvents();        
    }
    
    void setScaleMode(Phonon::VideoWidget::ScaleMode scaleMode)
    {
        m_scaleMode = scaleMode;
        updateDrawFrameRect();
        updateVideoFrame();
        repaint();
    }

    void setAspectRatio(Phonon::VideoWidget::AspectRatio aspect)
    {
        m_aspect = aspect;
        updateDrawFrameRect();
        updateVideoFrame();
        repaint();
    }

	void VideoRenderWidget::updateVideoOutputCount(int count)
	{
		if (m_renderDrawWidget)
			m_renderDrawWidget->updateVideoOutputCount(count);
	}
	
	void VideoRenderWidget::setMovieIsPaused(bool paused)
	{
		if (m_renderDrawWidget)
			m_renderDrawWidget->setMovieIsPaused(paused);
	}

    void VideoRenderWidget::updateDrawFrameRect()
    {
        if (m_movieFrameRect.width() <= 0 || m_movieFrameRect.height() <= 0)
            m_movieFrameRect = QRect(0, 0, 640, 480);
    
        // Set m_drawFrameRect to be the size of the smallest possible
        // rect conforming to the aspect and containing the whole frame:
        switch(m_aspect){
        case Phonon::VideoWidget::AspectRatioWidget:
            m_drawFrameRect = rect();
            break;
        case Phonon::VideoWidget::AspectRatio4_3:
            m_drawFrameRect = scaleToAspect(m_movieFrameRect, 4, 3);
            break;
        case Phonon::VideoWidget::AspectRatio16_9:
            m_drawFrameRect = scaleToAspect(m_movieFrameRect, 16, 9);
            break;
        case Phonon::VideoWidget::AspectRatioAuto:
        default:
            m_drawFrameRect = m_movieFrameRect;
            break;
        }
    
        // Scale m_drawFrameRect to fill the widget
        // without breaking aspect:
        int widgetWidth = rect().width();
        int widgetHeight = rect().height();
        int frameWidth = widgetWidth;
        int frameHeight = m_drawFrameRect.height() * float(widgetWidth) / float(m_drawFrameRect.width());
    
        switch(m_scaleMode){
        case Phonon::VideoWidget::ScaleAndCrop:
            if (frameHeight < widgetHeight){
                frameWidth *= float(widgetHeight) / float(frameHeight);
                frameHeight = widgetHeight;
            }
            break;
        case Phonon::VideoWidget::FitInView:
        default:
            if (frameHeight > widgetHeight){
                frameWidth *= float(widgetHeight) / float(frameHeight);
                frameHeight = widgetHeight;
            }
            break;
        }
    
        m_drawFrameRect.setSize(QSize(frameWidth, frameHeight));
        m_drawFrameRect.moveTo((widgetWidth - frameWidth) / 2.0f, (widgetHeight - frameHeight) / 2.0f);
        
		if (m_renderDrawWidget)
			m_renderDrawWidget->setDrawFrameRect(m_drawFrameRect);
    }
    
    QRect scaleToAspect(QRect srcRect, int w, int h)
    {
        int width = srcRect.width();
        int height = srcRect.width() * (float(h) / float(w));
        if (height > srcRect.height()){
            height = srcRect.height();
            width = srcRect.height() * (float(w) / float(h));
        }
        return QRect(0, 0, width, height);
    }    
};

/////////////////////////////////////////////////////////////////////////////////////////

VideoWidget::VideoWidget(QObject *parent) : MediaNode(VideoSink, parent)
{
    m_videoRenderWidget = new VideoRenderWidget();
}

VideoWidget::~VideoWidget()
{
    delete m_videoRenderWidget;
}

QWidget *VideoWidget::widget()
{
    IMPLEMENTED;
    return m_videoRenderWidget;
}

Phonon::VideoWidget::AspectRatio VideoWidget::aspectRatio() const
{
    IMPLEMENTED;
    return  m_videoRenderWidget->m_aspect;
}

void VideoWidget::setAspectRatio(Phonon::VideoWidget::AspectRatio aspect)
{
    IMPLEMENTED;
    m_videoRenderWidget->setAspectRatio(aspect);
}

Phonon::VideoWidget::ScaleMode VideoWidget::scaleMode() const
{
    IMPLEMENTED;
    return m_videoRenderWidget->m_scaleMode;
}

void VideoWidget::setScaleMode(Phonon::VideoWidget::ScaleMode scaleMode)
{
    IMPLEMENTED;
    m_videoRenderWidget->setScaleMode(scaleMode);
}

qreal VideoWidget::brightness() const
{
    IMPLEMENTED;
    return m_videoRenderWidget->m_brightness;
}

void VideoWidget::setBrightness(qreal value)
{
    IMPLEMENTED;
    m_videoRenderWidget->m_brightness = value;
    if (m_owningMediaObject && m_owningMediaObject->state() == Phonon::PausedState)
        m_videoRenderWidget->updateVideoFrame();
}

qreal VideoWidget::contrast() const
{
    IMPLEMENTED;
    return m_videoRenderWidget->m_contrast;
}

void VideoWidget::setContrast(qreal value)
{
    IMPLEMENTED;
    m_videoRenderWidget->m_contrast = value;
    if (m_owningMediaObject && m_owningMediaObject->state() == Phonon::PausedState)
        m_videoRenderWidget->updateVideoFrame();
}

qreal VideoWidget::hue() const
{
    IMPLEMENTED;
    return m_videoRenderWidget->m_hue;
}

void VideoWidget::setHue(qreal value)
{
    IMPLEMENTED;
    m_videoRenderWidget->m_hue = value;
    if (m_owningMediaObject && m_owningMediaObject->state() == Phonon::PausedState)
        m_videoRenderWidget->updateVideoFrame();
}

qreal VideoWidget::saturation() const
{
    IMPLEMENTED;
    return m_videoRenderWidget->m_saturation;
}

void VideoWidget::setSaturation(qreal value)
{
    IMPLEMENTED;
    m_videoRenderWidget->m_saturation = value;
    if (m_owningMediaObject && m_owningMediaObject->state() == Phonon::PausedState)
        m_videoRenderWidget->updateVideoFrame();
}

void VideoWidget::mediaNodeEvent(const MediaNodeEvent *event)
{
    switch (event->type()){
    case MediaNodeEvent::VideoFrameSizeChanged:
        m_videoRenderWidget->setMovieRect(*static_cast<QRect *>(event->data()));
        break;
	case MediaNodeEvent::VideoOutputCountChanged:
	     m_videoRenderWidget->updateVideoOutputCount(*static_cast<int *>(event->data()));
	     break;
	case MediaNodeEvent::MediaPlaying:
	     m_videoRenderWidget->setMovieIsPaused(!(*static_cast<bool *>(event->data())));
	     break;
	default:
        break;
    }
}

void VideoWidget::updateVideo(VideoFrame &frame){
	PhononAutoReleasePool pool;
    m_videoRenderWidget->setVideoFrame(frame);
    MediaNode::updateVideo(frame);
}

}} // namespace Phonon::QT7

QT_END_NAMESPACE

#include "moc_videowidget.cpp"
