/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtScriptTools module of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the either Technology Preview License Agreement or the
** Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "qscriptenginedebuggerfrontend_p.h"
#include "qscriptdebuggerfrontend_p_p.h"
#include "qscriptdebuggerbackend_p.h"
#include "qscriptdebuggerbackend_p_p.h"
#include "qscriptdebuggerevent_p.h"
#include "qscriptdebuggercommandexecutor_p.h"
#include "qscriptdebuggerresponse_p.h"

#include <QtCore/qcoreapplication.h>
#include <QtCore/qeventloop.h>
#include <QtCore/qlist.h>
#include <QtScript/qscriptvalue.h>

QT_BEGIN_NAMESPACE

/*!
  \class QScriptEngineDebuggerFrontend
  \since 4.5
  \internal

  \brief The QScriptEngineDebuggerFrontend class provides an in-process debugger frontend.

  This type of frontend is used when the QScriptEngine being debugged
  lives in the same process as the debugger.

  Call the attachTo() function to attach to an engine.
*/

class QScriptEngineDebuggerBackendPrivate;
class QScriptEngineDebuggerBackend : public QScriptDebuggerBackend
{
public:
    QScriptEngineDebuggerBackend(QScriptEngineDebuggerFrontendPrivate *frontend);
    ~QScriptEngineDebuggerBackend();

protected:
    void event(const QScriptDebuggerEvent &event);
    void resume();

private:
    Q_DECLARE_PRIVATE(QScriptEngineDebuggerBackend)
    Q_DISABLE_COPY(QScriptEngineDebuggerBackend)
};

class QScriptEngineDebuggerBackendPrivate
    : public QScriptDebuggerBackendPrivate
{
    Q_DECLARE_PUBLIC(QScriptEngineDebuggerBackend)
public:
    QScriptEngineDebuggerBackendPrivate();
    ~QScriptEngineDebuggerBackendPrivate();

    QScriptEngineDebuggerFrontendPrivate *frontend;
};

class QScriptEngineDebuggerFrontendPrivate
    : public QScriptDebuggerFrontendPrivate
{
    Q_DECLARE_PUBLIC(QScriptEngineDebuggerFrontend)
public:
    QScriptEngineDebuggerFrontendPrivate();
    ~QScriptEngineDebuggerFrontendPrivate();

    void event(const QScriptDebuggerEvent &event);
    void resume();

    QScriptEngineDebuggerBackend *backend;
    QList<QEventLoop*> eventLoopPool;
    QList<QEventLoop*> eventLoopStack;
};

QScriptEngineDebuggerBackendPrivate::QScriptEngineDebuggerBackendPrivate()
{
    frontend = 0;
}

QScriptEngineDebuggerBackendPrivate::~QScriptEngineDebuggerBackendPrivate()
{
}

/*!
  \internal

  Creates a new QScriptEngineDebuggerBackend object for the given \a
  engine.  The back-end will forward events to the given \a frontend.
*/
QScriptEngineDebuggerBackend::QScriptEngineDebuggerBackend(
    QScriptEngineDebuggerFrontendPrivate *frontend)
    : QScriptDebuggerBackend(*new QScriptEngineDebuggerBackendPrivate)
{
    Q_D(QScriptEngineDebuggerBackend);
    d->frontend = frontend;
}

QScriptEngineDebuggerBackend::~QScriptEngineDebuggerBackend()
{
}

/*!
  \reimp
*/
void QScriptEngineDebuggerBackend::event(const QScriptDebuggerEvent &event)
{
    Q_D(QScriptEngineDebuggerBackend);
    // forward to front-end
    d->frontend->event(event);
}

/*!
  \reimp
*/
void QScriptEngineDebuggerBackend::resume()
{
    Q_D(QScriptEngineDebuggerBackend);
    // forward to front-end
    d->frontend->resume();
}

QScriptEngineDebuggerFrontendPrivate::QScriptEngineDebuggerFrontendPrivate()
{
    backend = 0;
}

QScriptEngineDebuggerFrontendPrivate::~QScriptEngineDebuggerFrontendPrivate()
{
    delete backend;
    eventLoopPool << eventLoopStack;
    eventLoopStack.clear();
    while (!eventLoopPool.isEmpty()) {
        QEventLoop *eventLoop = eventLoopPool.takeFirst();
        if (eventLoop->isRunning()) {
            eventLoop->quit();
            eventLoop->deleteLater();
        } else {
            delete eventLoop;
        }
    }
}

/*!
  \internal

  Handles an event that has happened in the back-end. We notify the
  client, then enter an event loop until the front-end sends a command
  telling us to resume.
*/
void QScriptEngineDebuggerFrontendPrivate::event(const QScriptDebuggerEvent &event)
{
    Q_Q(QScriptEngineDebuggerFrontend);
    if (eventLoopPool.isEmpty())
        eventLoopPool.append(new QEventLoop());
    QEventLoop *eventLoop = eventLoopPool.takeFirst();
    Q_ASSERT(!eventLoop->isRunning());
    eventLoopStack.prepend(eventLoop);

    bool handled = q->notifyEvent(event);
    if (eventLoopStack.isEmpty()) {
        // delivery of the event caused a resume() already
        return;
    } else if (handled) {
        eventLoopStack.takeFirst();
        eventLoopPool.append(eventLoop);
        return;
    }

    // Run an event loop until resume() is called.
    // This effectively stalls script execution and makes it possible
    // for the debugger to inspect the execution state in the meantime.
    eventLoop->exec();
    if (!eventLoopStack.isEmpty()) {
        // the event loop was quit directly (i.e. not via resume())
        eventLoopStack.takeFirst();
    }
    eventLoopPool.append(eventLoop);
    backend->doPendingEvaluate(/*postEvent=*/false);
}

void QScriptEngineDebuggerFrontendPrivate::resume()
{
    // quitting the event loops will cause event() to return (see above)
    while (!eventLoopStack.isEmpty()) {
        QEventLoop *eventLoop = eventLoopStack.takeFirst();
        if (eventLoop->isRunning())
            eventLoop->quit();
    }
}

QScriptEngineDebuggerFrontend::QScriptEngineDebuggerFrontend()
    : QScriptDebuggerFrontend(*new QScriptEngineDebuggerFrontendPrivate)
{
}

QScriptEngineDebuggerFrontend::~QScriptEngineDebuggerFrontend()
{
    detach();
}

/*!
  Attaches this front-end to the given \a engine.
*/
void QScriptEngineDebuggerFrontend::attachTo(QScriptEngine *engine)
{
    Q_D(QScriptEngineDebuggerFrontend);
    if (d->backend)
        d->backend->detach();
    else
        d->backend = new QScriptEngineDebuggerBackend(d);
    d->backend->attachTo(engine);
}

/*!
  Detaches this front-end from the current script engine.
*/
void QScriptEngineDebuggerFrontend::detach()
{
    Q_D(QScriptEngineDebuggerFrontend);
    if (d->backend)
        d->backend->detach();
}

QScriptValue QScriptEngineDebuggerFrontend::traceFunction() const
{
    Q_D(const QScriptEngineDebuggerFrontend);
    if (d->backend)
        d->backend->traceFunction();
    return QScriptValue();
}

QScriptDebuggerBackend *QScriptEngineDebuggerFrontend::backend() const
{
    Q_D(const QScriptEngineDebuggerFrontend);
    return d->backend;
}

/*!
  \reimp
*/
void QScriptEngineDebuggerFrontend::processCommand(int id, const QScriptDebuggerCommand &command)
{
    Q_D(QScriptEngineDebuggerFrontend);
    Q_ASSERT(d->backend != 0);
    QScriptDebuggerCommandExecutor *executor = d->backend->commandExecutor();
    QScriptDebuggerResponse response = executor->execute(d->backend, command);
    notifyCommandFinished(id, response);
}

QT_END_NAMESPACE
