/****************************************************************************
**
** Copyright (C) 2008 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the demonstration applications of the Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the either Technology Preview License Agreement or the
** Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include "trackball.h"

//============================================================================//
//                                  TrackBall                                 //
//============================================================================//

TrackBall::TrackBall(TrackMode mode)
    : m_angularVelocity(0)
    , m_paused(false)
    , m_pressed(false)
    , m_mode(mode)
{
    m_axis = gfx::Vector3f::vector(0, 1, 0);
    m_rotation = gfx::Quaternionf::quaternion(1.0f, 0.0f, 0.0f, 0.0f);
    m_lastTime = QTime::currentTime();
}

TrackBall::TrackBall(float angularVelocity, const gfx::Vector3f& axis, TrackMode mode)
    : m_axis(axis)
    , m_angularVelocity(angularVelocity)
    , m_paused(false)
    , m_pressed(false)
    , m_mode(mode)
{
    m_rotation = gfx::Quaternionf::quaternion(1.0f, 0.0f, 0.0f, 0.0f);
    m_lastTime = QTime::currentTime();
}

void TrackBall::push(const QPointF& p)
{
    m_rotation = rotation();
    m_pressed = true;
    m_lastTime = QTime::currentTime();
    m_lastPos = p;
    m_angularVelocity = 0.0f;
}

void TrackBall::move(const QPointF& p)
{
    if (!m_pressed)
        return;

    QTime currentTime = QTime::currentTime();
    int msecs = m_lastTime.msecsTo(currentTime);
    if (msecs <= 20)
        return;

    switch (m_mode) {
    case Plane:
        {
            QLineF delta(m_lastPos, p);
            m_angularVelocity = delta.length() / msecs;
            m_axis = gfx::Vector3f::vector(delta.dy(), -delta.dx(), 0.0f).normalized();
            m_rotation *= gfx::Quaternionf::rotation(delta.length(), m_axis);
        }
        break;
    case Sphere:
        {
            gfx::Vector3f lastPos3D = gfx::Vector3f::vector(m_lastPos.x(), m_lastPos.y(), 0);
            float sqrZ = 1 - lastPos3D.sqrNorm();
            if (sqrZ > 0)
                lastPos3D[2] = sqrt(sqrZ);
            else
                lastPos3D.normalize();

            gfx::Vector3f currentPos3D = gfx::Vector3f::vector(p.x(), p.y(), 0);
            sqrZ = 1 - currentPos3D.sqrNorm();
            if (sqrZ > 0)
                currentPos3D[2] = sqrt(sqrZ);
            else
                currentPos3D.normalize();

            m_axis = gfx::Vector3f::cross(currentPos3D, lastPos3D);
            float angle = asin(sqrt(m_axis.sqrNorm()));

            m_angularVelocity = angle / msecs;
            m_axis.normalize();
            m_rotation *= gfx::Quaternionf::rotation(angle, m_axis);
        }
        break;
    }

    m_lastPos = p;
    m_lastTime = currentTime;
}

void TrackBall::release(const QPointF& p)
{
    // Calling move() caused the rotation to stop if the framerate was too low.
    move(p);
    m_pressed = false;
}

void TrackBall::start()
{
    m_lastTime = QTime::currentTime();
    m_paused = false;
}

void TrackBall::stop()
{
    m_rotation = rotation();
    m_paused = true;
}

gfx::Quaternionf TrackBall::rotation() const
{
    if (m_paused || m_pressed)
        return m_rotation;

    QTime currentTime = QTime::currentTime();
    float angle = m_angularVelocity * m_lastTime.msecsTo(currentTime);
    return m_rotation * gfx::Quaternionf::rotation(angle, m_axis);
}

