/****************************************************************************
**
** Copyright (C) 1992-2008 Trolltech ASA. All rights reserved.
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License versions 2.0 or 3.0 as published by the Free Software
** Foundation and appearing in the files LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file.  Alternatively you may (at
** your option) use any later version of the GNU General Public
** License if such license has been publicly approved by Trolltech ASA
** (or its successors, if any) and the KDE Free Qt Foundation. In
** addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.2, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** Please review the following information to ensure GNU General
** Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/. If
** you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech, as the sole
** copyright holder for Qt Designer, grants users of the Qt/Eclipse
** Integration plug-in the right for the Qt/Eclipse Integration to
** link to functionality provided by Qt Designer and its related
** libraries.
**
** This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
** INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE. Trolltech reserves all rights not expressly
** granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QT_NO_GRAPHICSVIEW

#include "qapplication.h"
#include "qgraphicslayout.h"
#include "qgraphicslayout_p.h"
#include "qgraphicslayoutitem.h"
#include "qgraphicswidget.h"
#include "qgraphicswidget_p.h"
#include "qgraphicsscene.h"

QT_BEGIN_NAMESPACE

/*!
    \class QGraphicsLayout
    \brief The QGraphicsLayout class provides the base class for all layouts
    in Graphics View.
    \since 4.4
    \ingroup multimedia

    QGraphicsLayout is an abstract class that defines a virtual API for
    arranging QGraphicsWidget children and other QGraphicsLayoutItem items for a
    QGraphicsWidget. QGraphicsWidget assigns responsibility to a
    QGraphicsWidget through QGraphicsWidget::setLayout(). As the widget is
    resized, the layout will automatically arrange the widget's children.

    You can use this class as a base when writing your own custom layout
    (e.g., a flowlayout), but it's more common to use one of its subclasses,
    QGraphicsLinearLayout and QGraphicsGridLayout, directly instead.

    When the geometry changes, QGraphicsLayout immediately rearranges the
    QGraphicsLayoutItem items it is managing by calling setGeometry() on each
    item. This rearrangement is what we refer to as "activating" the
    layout. QGraphicsLayout always updates its own geometry to match the
    contentsRect() of the QGraphicsLayoutItem it is managing (i.e., a QGraphicsLayout
    will always have the same geometry as the contents rect of the
    QGraphicsWidget it is assigned to), and because of this, it will
    automatically rearrange all its items when the widget is
    resized. QGraphicsLayout caches the sizes of all its managed items to
    avoid calling setGeometry() too often.

    There are also two implicit ways to trigger an activation. By calling
    activate(), you activate the layout immediately. In contrast, calling
    invalidate() is delayed, it will post a
    \l{QEvent::LayoutRequest}{LayoutRequest} event to the managed widget, and
    through event compression this ensures that activate() will be called only
    once after control has returned to the event loop. This is referred to as
    "invalidating" the layout. Activation is always immediate, invalidating is
    always delayed. Invalidating the layout also invalidates any cached
    information - invalidate() is a virtual function, so you can invalidate
    your own cache in a subclass of QGraphicsLayout by reimplementing this
    function.

    QGraphicsLayout listens to events for its managed widget through the
    virtual widgetEvent() event handler. When assigned to a widget, all events
    that are delivered to the widget are first processed by
    widgetEvent(). This allows the layout to become aware of any relevant
    state changes to the widget (e.g., visibility changes or layout direction
    changes).

    When creating a subclass of QGraphicsLayout, you must implement three pure
    virtual functions for managing QGraphicsLayoutItem items. The count() function
    should return the number of items that are currently managed. itemAt()
    should return a pointer any of the managed items, and removeAt() removes
    an item at an index. These virtual functions are called by QGraphicsLayout
    and used when rearranging the managed items. The subclass decides how
    these items are stored.

    QGraphicsLayout provides content margin management through
    reimplementations of setContentsMargins() and getContentsMargins().

    Because it inherits QGraphicsLayoutItem, QGraphicsLayout can itself also be
    managed by any layout, including its own subclasses.
*/

/*!
    Contructs a QGraphicsLayout object. \a parent is passed to QGraphicsLayoutItem's
    constructor (and QGraphicsLayoutItem's isLayout argument is set to \e true).
*/
QGraphicsLayout::QGraphicsLayout(QGraphicsLayoutItem *parent)
    : QGraphicsLayoutItem(parent, true)
{
    if (parent && !parent->isLayout()) {
        //### Ugly, but keeps old Qt-style
        static_cast<QGraphicsWidget *>(parent)->d_func()->setLayout_helper(this);
    }
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding, QSizePolicy::DefaultType);
}

/*!
    \internal
*/
QGraphicsLayout::QGraphicsLayout(QGraphicsLayoutPrivate &dd, QGraphicsLayoutItem *parent)
    : QGraphicsLayoutItem(dd)
{
    setParentLayoutItem(parent);
    if (parent && !parent->isLayout())
        static_cast<QGraphicsWidget*>(parent)->d_func()->setLayout_helper(this);     //### Ugly, but keeps old Qt-style
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding, QSizePolicy::DefaultType);
}

/*!
    Destroys the QGraphicsLayout object.
*/
QGraphicsLayout::~QGraphicsLayout()
{
}

/*!
    Sets the contents margins to \a left, \a top, \a right and \a bottom. The
    default contents margins for toplevel layouts are style dependent (by quering the pixelMetric 
    for QStyle::PM_LayoutLeftMargin, QStyle::PM_LayoutTopMargin, QStyle::PM_LayoutRightMargin
    and QStyle::PM_LayoutBottomMargin
    For sublayouts the default margins are 0.

    Changing the contents margins automatically invalidates the layout.

    \sa invalidate()
*/
void QGraphicsLayout::setContentsMargins(qreal left, qreal top, qreal right, qreal bottom)
{
    Q_D(QGraphicsLayout);
    if (d->left == left && d->top == top && d->right == right && d->bottom == bottom)
        return;
    d->left = left;
    d->right = right;
    d->top = top;
    d->bottom = bottom;
    invalidate();
}

/*!
    \reimp
*/
void QGraphicsLayout::getContentsMargins(qreal *left, qreal *top, qreal *right, qreal *bottom) const
{
    Q_D(const QGraphicsLayout);
    d->getMargin(left, d->left, QStyle::PM_LayoutLeftMargin);
    d->getMargin(top, d->top, QStyle::PM_LayoutTopMargin);
    d->getMargin(right, d->right, QStyle::PM_LayoutRightMargin);
    d->getMargin(bottom, d->bottom, QStyle::PM_LayoutBottomMargin);
}

/*!
    \reimp
*/
QRectF QGraphicsLayout::geometry() const
{
    // this works for most use cases
    Q_D(const QGraphicsLayout);
    QRectF geom;
    if (QGraphicsWidget *par = d->parentWidget()) {
        qreal left, top, right, bottom;
        par->getContentsMargins(&left, &top, &right, &bottom);
        geom = par->geometry().adjusted(+left, +top, -right, -bottom);
    }
    return geom;
}

/*!
    Activates the layout, causing an immediate rearrangement of all items in
    the layout. This function is based on calling count() and itemAt(), and
    then calling setGeometry() in sequence on all items.
    When activated, the layout will adjust its geometry to its parent's
    contentsRect(). The parent will then invalidate any layout of its own.

    If called in sequence or recursively, (e.g., by one of the arranged items
    in response to being resized,) this function will do nothing.

    Note that the layout is free to use geometry caching to optimize this
    process.  To forcefully invalidate any such cache, you can call
    invalidate() before calling activate().

    \sa invalidate()
*/
void QGraphicsLayout::activate()
{
    Q_D(QGraphicsLayout);
    if (d->activated)
        return;

    // we don't call activate on a sublayout, but somebody might.
    // Therefore, we walk to the parentitem of the toplevel layout.
    QGraphicsLayoutItem *parentItem = this;
    while (parentItem && parentItem->isLayout())
        parentItem = parentItem->parentLayoutItem();
    if (!parentItem)
        return;
    Q_ASSERT(!parentItem->isLayout());

    d->activated = true;                        // yes, we are now starting a relayout of our children
    setGeometry(parentItem->contentsRect());    // relayout children
    d->activated = false;
    parentLayoutItem()->updateGeometry();            // bubble up; will set activated to false
    // ### too many resizes? maybe we should walk up the chain to the
    // ### top-level layouted layoutItem and call activate there.
}

/*!
    Returns true if the layout is currently being activated; otherwise,
    returns false. If the layout is being activated, this means that it is
    currently in the process of rearranging its items (i.e., the activate()
    function has been called, and has not yet returned).

    \sa activate(), invalidate()
*/
bool QGraphicsLayout::isActivated() const
{
    Q_D(const QGraphicsLayout);
    return d->activated;
}

/*!
    Clears any cached geometry and size hint information in the layout, and
    posts a \l{QEvent::LayoutRequest}{LayoutRequest} event to the managed
    parent QGraphicsLayoutItem.

    \sa activate(), setGeometry()
*/
void QGraphicsLayout::invalidate()
{
    QGraphicsLayoutItem *layoutItem = this;
    while (layoutItem && layoutItem->isLayout()) {
        static_cast<QGraphicsLayout*>(layoutItem)->d_func()->activated = false;
        layoutItem = layoutItem->parentLayoutItem();
    }
    if (layoutItem && !layoutItem->isLayout())
        QApplication::postEvent(static_cast<QGraphicsWidget *>(layoutItem), new QEvent(QEvent::LayoutRequest));
}

/*!
    This virtual event handler receives all events for the managed
    widget. QGraphicsLayout uses this event handler to listen for layout
    related events such as geometry changes, layout changes and layout
    direction changes. \a e is a pointer to the event.

    You can reimplement this event handler to track similar events for your
    own custom layout.

    \sa QGraphicsWidget::event(), QGraphicsItem::sceneEvent()
*/
void QGraphicsLayout::widgetEvent(QEvent *e)
{
    switch (e->type()) {
    case QEvent::GraphicsSceneResize:
        if (isActivated()) {
            setGeometry(parentLayoutItem()->contentsRect());
        } else {
            activate(); // relies on that activate() will call updateGeometry()
        }
        break;
    case QEvent::LayoutRequest:
        activate();
        break;
    case QEvent::LayoutDirectionChange:
        invalidate();
        break;
    default:
        break;
    }
}

/*!
    \fn virtual int QGraphicsLayout::count() const = 0

    This pure virtual function must be reimplemented in a subclass of
    QGraphicsLayout to return the number of items in the layout.

    The subclass is free to decide how to store the items.

    \sa itemAt(), removeAt()
*/

/*!
    \fn virtual QGraphicsLayoutItem *QGraphicsLayout::itemAt(int i) const = 0

    This pure virtual function must be reimplemented in a subclass of
    QGraphicsLayout to return a pointer to the item at index \a i. The
    reimplementation can assume that \a i is valid (i.e., it respects the
    value of count()).

    The subclass is free to decide how to store the items.

    \sa count(), removeAt()
*/

/*!
    \fn virtual void QGraphicsLayout::removeAt(int i) = 0

    This pure virtual function must be reimplemented in a subclass of
    QGraphicsLayout to remove the item at index \a i. The reimplementation can
    assume that \a i is valid (i.e., it respects the value of count()).

    The subclass is free to decide how to store the items.

    \sa itemAt(), count()
*/

QT_END_NAMESPACE

#endif //QT_NO_GRAPHICSVIEW
