/****************************************************************************
**
** Copyright (C) 2005-2008 Trolltech ASA. All rights reserved.
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** This file may be used under the terms of the GNU General Public
** License versions 2.0 or 3.0 as published by the Free Software
** Foundation and appearing in the files LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file.  Alternatively you may (at
** your option) use any later version of the GNU General Public
** License if such license has been publicly approved by Trolltech ASA
** (or its successors, if any) and the KDE Free Qt Foundation. In
** addition, as a special exception, Trolltech gives you certain
** additional rights. These rights are described in the Trolltech GPL
** Exception version 1.2, which can be found at
** http://www.trolltech.com/products/qt/gplexception/ and in the file
** GPL_EXCEPTION.txt in this package.
**
** Please review the following information to ensure GNU General
** Public Licensing requirements will be met:
** http://trolltech.com/products/qt/licenses/licensing/opensource/. If
** you are unsure which license is appropriate for your use, please
** review the following information:
** http://trolltech.com/products/qt/licenses/licensing/licensingoverview
** or contact the sales department at sales@trolltech.com.
**
** In addition, as a special exception, Trolltech, as the sole
** copyright holder for Qt Designer, grants users of the Qt/Eclipse
** Integration plug-in the right for the Qt/Eclipse Integration to
** link to functionality provided by Qt Designer and its related
** libraries.
**
** This file is provided "AS IS" with NO WARRANTY OF ANY KIND,
** INCLUDING THE WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE. Trolltech reserves all rights not expressly
** granted herein.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QTCONCURRENT_RESULTSTORE_H
#define QTCONCURRENT_RESULTSTORE_H

#include <QtCore/qglobal.h>

#ifndef QT_NO_QFUTURE

#include <QtCore/qmap.h>

QT_BEGIN_HEADER
QT_BEGIN_NAMESPACE

QT_MODULE(Core)

/*
    ResultStore stores indexed results. Results can be added and retrieved
    either individually batched in a QVector. Retriveing results and checking
    which indexes are in the store can be done either by iterating or by random
    accees. In addition results kan be removed from the front of the store,
    either individually or in batches.
*/

#ifndef qdoc

namespace QtConcurrent {

#if 0
template <typename DataType>
class Result {
public:
    // Constructs a valid result.
    Result(const DataType &data)
    : m_data(data), valid(true) {}

    // Construct an invalid result.
    Result()
    : m_data(DataType()), valid(false) {}

    // Returns wether the result is valid
    inline bool isValid() const
    { return valid; }

    // Returns the value for the result.
    inline DataType data() const
    { return m_data; }
private:
    DataType m_data;
    bool valid;
};
#endif

class ResultItem
{
public:
    ResultItem(const void *_result, int _count) : count(_count), result(_result) { } // contruct with vector of results
    ResultItem(const void *_result) : count(0), result(_result) { } // construct with result
    ResultItem() : count(0), result(0) { }
    bool isVector() const { return count != 0; }
    int count;          // result is either a pointer to a result or to a vector of results,
    const void *result; // if count is 0 it's a result, otherwise it's a vector.
};

class ResultIteratorBase
{
public:
    inline ResultIteratorBase() : mapIterator(QMap<int, ResultItem>::const_iterator()), m_vectorIndex(0) { }
    inline ResultIteratorBase(QMap<int, ResultItem>::const_iterator _mapIterator, int _vectorIndex = 0)
           : mapIterator(_mapIterator), m_vectorIndex(_vectorIndex) { }

    int vectorIndex() const { return m_vectorIndex; }
    inline int resultIndex() const { return mapIterator.key() + m_vectorIndex; }

    ResultIteratorBase operator++()
    {
        if (canIncrementVectorIndex()) {
            ++m_vectorIndex;
        } else {
            ++mapIterator;
            m_vectorIndex = 0;
        }
        return *this;
    }

    int batchSize() const
    {
        if (isVector() == false)
            return 1;
        return mapIterator.value().count;
    }

    void batchedAdvance()
    {
        ++mapIterator;
        m_vectorIndex = 0;
    }

    bool operator==(const ResultIteratorBase &other) const
    {
        return (mapIterator == other.mapIterator && m_vectorIndex == other.m_vectorIndex);
    }

    bool operator!=(const ResultIteratorBase &other) const
    {
        return !operator==(other);
    }

    inline bool isVector() const
    {
        return mapIterator.value().isVector();
    }

    inline bool canIncrementVectorIndex() const
    {
        return (m_vectorIndex + 1 < mapIterator.value().count);
    }

    QMap<int, ResultItem>::const_iterator mapIterator;
    int m_vectorIndex;
};


template <typename T>
class ResultIterator : public ResultIteratorBase
{
public:
    ResultIterator(const ResultIteratorBase &base)
    : ResultIteratorBase(base) { }

    const T &value() const
    {
        return *pointer();
    }

    const T *pointer() const
    {
        if (mapIterator.value().isVector())
            return &(reinterpret_cast<const QVector<T> *>(mapIterator.value().result)->at(m_vectorIndex));
        else
            return reinterpret_cast<const T *>(mapIterator.value().result);
    }
};

class ResultStoreBase
{
public:
    ResultStoreBase() : vectorBeginIndex(0), insertIndex(0), resultCount(0) { }

    int addResult(int index, const void *result)
    {
        const int storeIndex = updateInsertIndex(index, 1);
        m_results[storeIndex] = ResultItem(result, 0); // 0 means "not a vector"
        ++resultCount;
        return storeIndex;
    }

    int addResults(int index, const void *results, int _count)
    {
        const int storeIndex = updateInsertIndex(index, _count);
        m_results[storeIndex] = ResultItem(results, _count);
        resultCount+=_count;
        return storeIndex;
    }

    ResultIteratorBase begin() const
    {
        return ResultIteratorBase(m_results.begin(), vectorBeginIndex);
    }

    ResultIteratorBase end() const
    {
        return ResultIteratorBase(m_results.end());
    }

    bool hasNextResult() const
    {
        return begin() != end();
    }

    ResultIteratorBase resultAt(int index) const
    {
        if (m_results.isEmpty())
            return ResultIteratorBase(m_results.end());
        QMap<int, ResultItem>::const_iterator it = m_results.lowerBound(index);

        // lowerBound returns either an iterator to the result or an iterator
        // to the nearest greater index. If the latter happens it might be
        // that the result is stored in a vector at the previous index.
        if (it == m_results.end()) {
            --it;
            if (it.value().isVector() == false) {
                return ResultIteratorBase(m_results.end());
            }
        } else {
            if (it.key() > index)
                --it;
        }

        const int vectorIndex = index - it.key();
        if (it.value().isVector() && vectorIndex >= it.value().count)
            return ResultIteratorBase(m_results.end());
        return ResultIteratorBase(it, vectorIndex);
    }

    bool contains(int index) const
    {
        return (resultAt(index) != end());
    }

    int count() const
    {
        return resultCount;
    }

    // returns the insert index, calling this funciton with
    // index equal to -1 returns the next available index.
    int updateInsertIndex(int index, int _count)
    {
        if (index == -1) {
            index = insertIndex;
            insertIndex += _count;
        } else {
            insertIndex = qMax(index + _count, insertIndex);
        }
        return index;
    }

    virtual ~ResultStoreBase() { };

protected:
    QMap<int, ResultItem> m_results;
    int vectorBeginIndex;  // Indicates how many items that have been erased
                           // from the beginning of the first result vector.
    int insertIndex;       // The index where the next results(s) will be inserted.
    int resultCount;       // The result count is maintanied manually.
};

template <typename T>
class ResultStore : public ResultStoreBase
{
public:
    ResultStore() { }

    ResultStore(const ResultStoreBase &base)
    : ResultStoreBase(base) { }

    int addResult(int index, const T  *result)
    {
        return ResultStoreBase::addResult(index, new T(*result));
    }

    int addResults(int index, const QVector<T> *results)
    {
        return ResultStoreBase::addResults(index, new QVector<T>(*results), results->count());
    }

    ResultIterator<T> begin() const
    {
        return static_cast<ResultIterator<T> >(ResultStoreBase::begin());
    }

    ResultIterator<T> end() const
    {
        return static_cast<ResultIterator<T> >(ResultStoreBase::end());
    }

    ResultIterator<T> resultAt(int index) const
    {
        return static_cast<ResultIterator<T> >(ResultStoreBase::resultAt(index));
    }

#if 0
    // Remove and return the first result in the store
    Result<T> takeFirstResult()
    {
        ResultIterator<T> it = begin();
        if (it == end())
            return QtConcurrent::Result<T>();
        const QtConcurrent::Result<T> result(it.value());
        // Special case for vectors, either take one result from the front of the vector
        // or advance to the next index in the map if the vector is empty.
        if (it.isVector()) {
            if (it.canIncrementVectorIndex()) {
                ++vectorBeginIndex;
            } else {
                vectorBeginIndex = 0;
                delete reinterpret_cast<const QVector<T> *>(it.mapIterator.value().result);
                m_results.erase(*reinterpret_cast<QMap<int, ResultItem>::iterator *>(&it.mapIterator));
            }
        } else {
            delete reinterpret_cast<const T *>(it.mapIterator.value().result);
            m_results.erase(*reinterpret_cast<QMap<int, ResultItem>::iterator *>(&it.mapIterator));
        }
        --resultCount;
        return result;
    }

    // Remove and return the first batch of results in the store.
    QtConcurrent::Result<QVector<T> > takeFirstResults()
    {
        ResultIterator<T> it = begin();
        if (it == end())
            return QtConcurrent::Result<QVector<T> >();
        if (it.isVector()) {
            const QVector<T> *ptr = reinterpret_cast<const QVector<T> *>(it.mapIterator.value().result);
            QVector<T> vector = *ptr;
            delete ptr;
            m_results.erase(*reinterpret_cast<QMap<int, ResultItem>::iterator *>(&it.mapIterator));

            // Some of the results in the vector might have already been taken by
            // calls to takeFirstResult(), so erase those from the vector.
            const int taken = vector.end() - vector.begin() + vectorBeginIndex;
            vector.erase(vector.begin(), vector.begin() + vectorBeginIndex);
            resultCount -= taken;

            vectorBeginIndex = 0;
            return Result<QVector<T> >(vector);
        } else {
            // If the first item in the map is not a vector, return  a vector
            // containing that first result.
            QtConcurrent::Result<QVector<T> > result(QVector<T>() << it.value());
            delete reinterpret_cast<const T *>(it.mapIterator.value().result);
            m_results.erase(*reinterpret_cast<QMap<int, ResultItem>::iterator *>(&it.mapIterator));
            --resultCount;
            return result;
        }
    }
#endif

    void clear()
    {
        QMap<int, ResultItem>::const_iterator mapIterator = m_results.begin();
        while (mapIterator != m_results.end()) {
            if (mapIterator.value().isVector())
                delete reinterpret_cast<const QVector<T> *>(mapIterator.value().result);
            else
                delete reinterpret_cast<const T *>(mapIterator.value().result);
            ++mapIterator;
        }
        resultCount = 0;
        m_results.clear();
    }

    ~ResultStore()
    {
        clear();
    }

};

} // namespace QtConcurrent

#endif //qdoc

QT_END_NAMESPACE
QT_END_HEADER

#endif // QT_NO_CONCURRENT

#endif
