/*  This file is part of the KDE project.

Copyright (C) 2007 Trolltech ASA. All rights reserved.

This library is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 2.1 or 3 of the License.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mediaobject.h"
#include "videowidget.h"
#include "audiooutput.h"

#include <QtCore/QVector>
#include <QtCore/QTimerEvent>
#include <QtCore/QTimer>
#include <QtGui/QWidget>

#include <qdebug.h>

#include <initguid.h>
#include <qnetwork.h>
#include <comdef.h>
#include <evcode.h>

#define TIMER_INTERVAL 10 //... ms for the timer that polls the current state (we use the multimedia timer
#define PRELOAD_TIME 2000 // 2 seconds to load a source

QT_BEGIN_NAMESPACE

namespace Phonon
{
    namespace DS9
    {

        MediaObject::MediaObject(QObject *parent) : BackendNode(parent),
            m_errorType(Phonon::NoError),
            m_state(Phonon::LoadingState),
            m_nextState(Phonon::StoppedState),
            m_prefinishMark(0),
            m_tickInterval(0),
            m_buffering(false),
            m_oldHasVideo(false),
            m_prefinishMarkSent(false),
            m_aboutToFinishSent(false),
            m_autoplayTitles(true),
            m_currentTitle(0),
            m_targetTick(INFINITE)
        {
            for(int i = 0; i < 2; ++i) {
                MediaGraph *graph = new MediaGraph(this, i);
                connect(graph, SIGNAL(loadingFinished(MediaGraph*, HRESULT)), SLOT(loadingFinished(MediaGraph*, HRESULT)));
                connect(graph, SIGNAL(stateReady(MediaGraph*, Phonon::State)), SLOT(slotStateReady(MediaGraph*, Phonon::State)));
                m_graphs << graph;
            }

            //really special case
            m_mediaObject = this;
        }

        MediaObject::~MediaObject()
        {
            qDeleteAll(m_graphs);
        }

        QWidget *MediaObject::dummyWidget()
        {
            return MediaGraph::dummyWidget();
        }

        MediaGraph *MediaObject::currentGraph() const
        {
            return m_graphs.first();
        }

        MediaGraph *MediaObject::nextGraph() const
        {
            return m_graphs.last();
        }

        //utility function to save the graph to a file
        void MediaObject::timerEvent(QTimerEvent *e)
        {
            if (e->timerId() == m_tickTimer.timerId()) {

                const qint64 current = currentTime();
                const qint64 remaining = remainingTime();

                if ( m_tickInterval != 0 && current > m_targetTick) {
                        updateTargetTick();
                        emit tick(current);
                }

                //check that the title hasn't changed
                if (m_titles.count() > 1 && m_currentTitle < m_titles.count() - 1) {

                    if (remaining <= 0) {
                        //the current title is finished
                        if (m_autoplayTitles) {
                            //we go to the next title
                            _iface_setTitle(m_currentTitle + 1, false);
                            emit tick(currentTime());
                        } else {
                            handleComplete(currentGraph(), false /*not a real finish*/);
                            updateTargetTick();
                        }
                    }
                    return;
                }

                if (m_transitionTime < 0 && nextGraph()->mediaSource().type() != Phonon::MediaSource::Invalid) {
                    //we are currently crossfading
                    foreach(AudioOutput *audio, m_audioOutputs) {
                        audio->setCrossFadingProgress( currentGraph()->index(), qMin(1., qreal(current) / qreal(-m_transitionTime)));
                    }
                }

                if (m_transitionTime < 0 && remaining < m_transitionTime + TIMER_INTERVAL/2) {
                    //we need to switch graphs to run the next source in the queue (with cross-fading)
                    switchToNextSource();
                }

                if ( m_prefinishMark > 0 && !m_prefinishMarkSent && remaining < m_prefinishMark + TIMER_INTERVAL/2) {
#ifdef GRAPH_DEBUG
                    qDebug() << "DS9: emit prefinishMarkReached" << remaining << QTime::currentTime().toString();
#endif
                    m_prefinishMarkSent = true;

                    emit prefinishMarkReached( remaining );
                }

                if (!m_aboutToFinishSent && remaining < PRELOAD_TIME - m_transitionTime  + TIMER_INTERVAL/2) {
                    //let's take a 2 seconds time time to actually load the next file
#ifdef GRAPH_DEBUG
                    qDebug() << "DS9: emit aboutToFinish" << remaining << QTime::currentTime().toString();
#endif
                    m_aboutToFinishSent = true;
                    emit aboutToFinish();
                }

                if (m_buffering) {
                    ComPointer<IAMNetworkStatus> status(currentGraph()->realSource(), IID_IAMNetworkStatus);
                    if (status) {
                        long l;
                        status->get_BufferingProgress(&l);
                        emit bufferStatus(l);
#ifdef GRAPH_DEBUG
                        qDebug() << "emit bufferStatus(" << l << ")";
#endif
                    }
                }
            }
        }

        void MediaObject::switchToNextSource()
        {
            m_prefinishMarkSent = false;
            m_aboutToFinishSent = false;

            m_oldHasVideo = currentGraph()->hasVideo();

            m_graphs.swap(0,1); //swap the graphs

            //we tell the video widgets to switch now to the new source
            foreach(VideoWidget *video, m_videoWidgets) {
                video->setCurrentGraph(currentGraph()->index());
            }

            emit currentSourceChanged(currentGraph()->mediaSource());

            if (currentGraph()->isLoading()) {
                //will simply tell that when loading is finished 
                //it should start the playback
                play(); 
                return;
            }



            emit metaDataChanged(currentGraph()->metadata());

            if (nextGraph()->hasVideo() != currentGraph()->hasVideo()) {
                emit hasVideoChanged(currentGraph()->hasVideo());
            }

            emit totalTimeChanged(totalTime());
            emit tick(0);

            setTitles(currentGraph()->titles());


            //this manages only gapless transitions
            if (currentGraph()->mediaSource().type() != Phonon::MediaSource::Invalid) {
                play();
            }
        }

        Phonon::State MediaObject::state() const
        {
            if (m_buffering) {
                return Phonon::BufferingState;
            } else {
                return m_state;
            }
        }

        bool MediaObject::hasVideo() const
        {
            return currentGraph()->hasVideo();
        }

        bool MediaObject::isSeekable() const
        {
            return currentGraph()->isSeekable();
        }

        qint64 MediaObject::totalTime() const
        {
            //1st, check if there is more titles after
            const qint64 ret = (m_currentTitle < m_titles.count() - 1) ? 
                titleAbsolutePosition(m_currentTitle+1) : currentGraph()->absoluteTotalTime();

            //this is the duration of the current title
            return ret - titleAbsolutePosition(m_currentTitle);
        }

        qint64 MediaObject::currentTime() const
        {
            //this handles inaccuracy when stopping on a title
            return qMin(totalTime(), 
                currentGraph()->absoluteCurrentTime() - titleAbsolutePosition(m_currentTitle));
        }

        qint32 MediaObject::tickInterval() const
        {
            return m_tickInterval;
        }

        void MediaObject::setTickInterval(qint32 newTickInterval)
        {
            m_tickInterval = newTickInterval;
            updateTargetTick();
        }

        void MediaObject::pause()
        {
            if (currentGraph()->isLoading()) {
                m_nextState = Phonon::PausedState;
            } else {
                currentGraph()->pause();
            }
        }

        void MediaObject::stop()
        {
            if (currentGraph()->isLoading()) {
                m_nextState = Phonon::StoppedState;
            } else {
                currentGraph()->stop();
                seek(0); //we just seek back to 0 position
            }
        }

        void MediaObject::play()
        {
            if (currentGraph()->isLoading()) {
                m_nextState = Phonon::PlayingState;
            } else  if (currentTime() <= totalTime()) {
                //the condition above is to handle gracefully the titles
                currentGraph()->play();
            }
        }

        QString MediaObject::errorString() const
        {
            return m_errorString;
        }

        Phonon::ErrorType MediaObject::errorType() const
        {
            return m_errorType;
        }


        void MediaObject::updateTargetTick()
        {
            if (m_tickInterval) {
                const qint64 current = currentTime();
                m_targetTick = current / m_tickInterval * m_tickInterval;
                if (current == 0 || m_targetTick < current) {
                    m_targetTick += m_tickInterval;
                }
            }
        }

        void MediaObject::setState(Phonon::State newstate)
        {
            if (newstate == Phonon::PlayingState) {
                updateTargetTick();
            }

            if (newstate == m_state) {
                return;
            }

            //manage the timer
            if (newstate == Phonon::PlayingState) {
                m_tickTimer.start(TIMER_INTERVAL, this);
            } else {
                m_tickTimer.stop();
            }

            Phonon::State oldstate = state();
            m_state = newstate;
            emit stateChanged(newstate, oldstate);
        }


        qint32 MediaObject::prefinishMark() const
        {
            return m_prefinishMark;
        }

        void MediaObject::setPrefinishMark(qint32 newPrefinishMark)
        {
            m_prefinishMark = newPrefinishMark;
        }

        qint32 MediaObject::transitionTime() const
        {
            return m_transitionTime;
        }

        void MediaObject::setTransitionTime(qint32 time)
        {
            m_transitionTime = time;
        }

        qint64 MediaObject::remainingTime() const
        {
            return totalTime() - currentTime();
        }


        Phonon::MediaSource MediaObject::source() const
        {
            return currentGraph()->mediaSource();
        }

        void MediaObject::setNextSource(const Phonon::MediaSource &source)
        {
            const bool shouldSwitch = m_state == Phonon::StoppedState || m_state == Phonon::ErrorState;
            nextGraph()->loadSource(source); //let's preload the source

            if (shouldSwitch) {
                switchToNextSource();
            }
        }

        void MediaObject::setSource(const Phonon::MediaSource &source)
        {
            m_prefinishMarkSent = false;
            m_aboutToFinishSent = false;

            m_oldHasVideo = currentGraph()->hasVideo();
            setState(Phonon::LoadingState);
            //After loading we go into stopped state
            m_nextState = Phonon::StoppedState; 
            catchComError(currentGraph()->loadSource(source));

        }

        void MediaObject::slotStateReady(MediaGraph *mg, Phonon::State newState)
        {
            if (mg == currentGraph() && !currentGraph()->isLoading()) {
                setState(newState);
            }
        }

        void MediaObject::loadingFinished(MediaGraph *mg, HRESULT hr)
        {
            if (mg == currentGraph()) { 
                //Title interface
                m_currentTitle = 0;
                setTitles(currentGraph()->titles());

                if (catchComError(hr)) {
                    return;
                }

                if (m_oldHasVideo != currentGraph()->hasVideo()) {
                    emit hasVideoChanged(currentGraph()->hasVideo());
                }

                emit metaDataChanged(currentGraph()->metadata());
                emit totalTimeChanged(totalTime());

                //let's put the next state
                switch(m_nextState)
                {
                case Phonon::PausedState:
                    pause();
                    break;
                case Phonon::StoppedState:
                    stop();
                    break;
                case Phonon::PlayingState:
                    play();
                    break;
                }
            }
        }

        void MediaObject::seek(qint64 time)
        {
            //we seek into the current title
            currentGraph()->absoluteSeek(time + titleAbsolutePosition(m_currentTitle));
            updateTargetTick();
            if (time < totalTime() - m_prefinishMark) {
                m_prefinishMarkSent = false;
            }

            if (time < totalTime() - PRELOAD_TIME + m_transitionTime) {
                m_aboutToFinishSent = false;
            }

            //this helps the update of the application (seekslider for example)
            if (m_state == PausedState || m_state == PlayingState) {
                emit tick(currentTime());
            }
        }

        bool MediaObject::catchComError(HRESULT hr)
        {

            m_errorString.clear();
            m_errorType = Phonon::NoError;

            switch(hr)
            {
            case S_OK:
                m_errorType = Phonon::NoError;
                break;
            default:
                if (FAILED(hr)) {
                    qWarning() << "an error occurred" << QString::fromWCharArray(_com_error(hr).ErrorMessage());
                    m_errorType = Phonon::FatalError;
                    m_errorString = QString::fromWCharArray(_com_error(hr).ErrorMessage());
                    setState(Phonon::ErrorState);
                }
                break;
            }


            return m_errorType != Phonon::NoError;
        }


        void MediaObject::grabNode(BackendNode *node)
        {
            foreach(MediaGraph *graph, m_graphs) {
                graph->grabNode(node);
            }
            node->setMediaObject(this);
        }

        bool MediaObject::connectNodes(BackendNode *source, BackendNode *sink)
        {
            bool ret = true;
            foreach(MediaGraph *graph, m_graphs) {
                ret = ret && graph->connectNodes(source, sink);
            }
            if (ret) {
                if (VideoWidget *video = qobject_cast<VideoWidget*>(sink)) {
                    m_videoWidgets += video;
                } else if (AudioOutput *audio = qobject_cast<AudioOutput*>(sink)) {
                    m_audioOutputs += audio;
                }
            }
            return ret;
        }

        bool MediaObject::disconnectNodes(BackendNode *source, BackendNode *sink)
        {
            bool ret = true;
            foreach(MediaGraph *graph, m_graphs) {
                ret = ret && graph->disconnectNodes(source, sink);
            }
            if (ret) {
                if (VideoWidget *video = qobject_cast<VideoWidget*>(sink)) {
                    m_videoWidgets -= video;
                } else if (AudioOutput *audio = qobject_cast<AudioOutput*>(sink)) {
                    m_audioOutputs -= audio;
                }
            }
            return ret;
        }

        void MediaObject::handleComplete(MediaGraph *graph, bool realFinish)
        {
            if (graph == currentGraph()) {
                if (m_transitionTime >= PRELOAD_TIME || m_aboutToFinishSent == false) {
                    emit aboutToFinish(); //give a chance to the frontend to give a next source
                    m_aboutToFinishSent = true;
                }

                if (nextGraph()->mediaSource().type() == Phonon::MediaSource::Invalid) {

                    //this is the last source, we simply finish
                    emit finished();
                    if (!realFinish) {
                        //we really need to stop data flow
                        pause(); //finish means that we pause
                        setState(Phonon::PausedState); //ensure it is done immediately
                    }

                } else if (m_transitionTime == 0) {
                    //gapless transition
                    switchToNextSource(); //let's call the function immediately
                } else if (m_transitionTime > 0) {
                    //management of the transition (if it is >= 0)
                    QTimer::singleShot(m_transitionTime, this, SLOT(switchToNextSource()));
                }
            } else {
                //it is just the end of the previous source (in case of cross-fading)
                graph->cleanup();
            }
            foreach(AudioOutput *audio, m_audioOutputs) {
                audio->setCrossFadingProgress( currentGraph()->index(), 1.); //cross-fading is in any case finished
            }
        }

        void MediaObject::handleEvents(MediaGraph *graph, long eventCode, long param1)
        {
            QString eventDescription;
            QTextStream o(&eventDescription);
            switch (eventCode)
            {
            case EC_BUFFERING_DATA:
                if (graph == currentGraph()) {
                    m_buffering = param1;
                    emit stateChanged(state(), m_state);
                }
                break;
            case EC_LENGTH_CHANGED: 
                emit totalTimeChanged( totalTime() );
                break;

            case EC_COMPLETE:
                handleComplete(graph);
                break;
            case EC_ACTIVATE: o << "EC_ACTIVATE: A video window is being " << (param1 ? "ACTIVATED" : "DEACTIVATED"); break;
            case EC_BUILT: o << "EC_BUILT: Send by the Video Control when a graph has been built. Not forwarded to applications."; break;
            case EC_CLOCK_CHANGED: o << "EC_CLOCK_CHANGED"; break;
            case EC_CLOCK_UNSET: o << "EC_CLOCK_UNSET: The clock provider was disconnected."; break;
            case EC_CODECAPI_EVENT: o << "EC_CODECAPI_EVENT: Sent by an encoder to signal an encoding event."; break;
            case EC_DEVICE_LOST: o << "EC_DEVICE_LOST: A Plug and Play device was removed or has become available again."; break;
            case EC_DISPLAY_CHANGED: o << "EC_DISPLAY_CHANGED: The display mode has changed."; break;
            case EC_END_OF_SEGMENT: o << "EC_END_OF_SEGMENT: The end of a segment has been reached."; break;
            case EC_ERROR_STILLPLAYING: o << "EC_ERROR_STILLPLAYING: An asynchronous command to run the graph has failed."; break;
            case EC_ERRORABORT: o << "EC_ERRORABORT: An operation was aborted because of an error."; break;
            case EC_EXTDEVICE_MODE_CHANGE: o << "EC_EXTDEVICE_MODE_CHANGE: Not supported."; break;
            case EC_FULLSCREEN_LOST: o << "EC_FULLSCREEN_LOST: The video renderer is switching out of full-screen mode."; break;
            case EC_GRAPH_CHANGED: o << "EC_GRAPH_CHANGED: The filter graph has changed."; break;
            case EC_NEED_RESTART: o << "EC_NEED_RESTART: A filter is requesting that the graph be restarted."; break;
            case EC_NOTIFY_WINDOW: o << "EC_NOTIFY_WINDOW: Notifies a filter of the video renderer's window."; break;
            case EC_OLE_EVENT: o << "EC_OLE_EVENT: A filter is passing a text string to the application."; break;
            case EC_OPENING_FILE: o << "EC_OPENING_FILE: The graph is opening a file, or has finished opening a file."; break;
            case EC_PALETTE_CHANGED: o << "EC_PALETTE_CHANGED: The video palette has changed."; break;
            case EC_PAUSED: o << "EC_PAUSED: A pause request has completed."; break;
            case EC_PREPROCESS_COMPLETE: o << "EC_PREPROCESS_COMPLETE: Sent by the WM ASF Writer filter when it completes the pre-processing for multipass encoding."; break;
            case EC_QUALITY_CHANGE: o << "EC_QUALITY_CHANGE: The graph is dropping samples, for quality control."; break;
            case EC_REPAINT: o << "EC_REPAINT: A video renderer requires a repaint."; break;
            case EC_SEGMENT_STARTED: o << "EC_SEGMENT_STARTED: A new segment has started."; break;
            case EC_SHUTTING_DOWN: o << "EC_SHUTTING_DOWN: The filter graph is shutting down, prior to being destroyed."; break;
            case EC_SNDDEV_IN_ERROR: o << "EC_SNDDEV_IN_ERROR: A device error has occurred in an audio capture filter."; break;
            case EC_SNDDEV_OUT_ERROR: o << "EC_SNDDEV_OUT_ERROR: A device error has occurred in an audio renderer filter."; break;
            case EC_STARVATION: o << "EC_STARVATION: A filter is not receiving enough data."; break;
            case EC_STATE_CHANGE: o << "EC_STATE_CHANGE: The filter graph has changed state."; break;
            case EC_STEP_COMPLETE: o << "EC_STEP_COMPLETE: A filter performing frame stepping has stepped the specified number of frames."; break;
            case EC_STREAM_CONTROL_STARTED: o << "EC_STREAM_CONTROL_STARTED: A stream-control start command has taken effect."; break;
            case EC_STREAM_CONTROL_STOPPED: o << "EC_STREAM_CONTROL_STOPPED: A stream-control stop command has taken effect."; break;
            case EC_STREAM_ERROR_STILLPLAYING: o << "EC_STREAM_ERROR_STILLPLAYING: An error has occurred in a stream. The stream is still playing."; break;
            case EC_STREAM_ERROR_STOPPED: o << "EC_STREAM_ERROR_STOPPED: A stream has stopped because of an error."; break;
            case EC_TIMECODE_AVAILABLE: o << "EC_TIMECODE_AVAILABLE: Not supported."; break;
            case EC_UNBUILT: o << "Sent by the Video Control when a graph has been torn down. Not forwarded to applications."; break;
            case EC_USERABORT: o << "EC_USERABORT: Send by the Video Control when a graph has been torn down. Not forwarded to applications."; break;
            case EC_VIDEO_SIZE_CHANGED: o << "EC_VIDEO_SIZE_CHANGED"; break;
            case EC_VMR_RECONNECTION_FAILED: o << "EC_VMR_RECONNECTION_FAILED: Sent by the VMR-7 and the VMR-9 when it was unable to accept a dynamic format change request from the upstream decoder."; break;
            case EC_VMR_RENDERDEVICE_SET: o << "EC_VMR_RENDERDEVICE_SET: Sent when the VMR has selected its rendering mechanism."; break;
            case EC_VMR_SURFACE_FLIPPED: o << "EC_VMR_SURFACE_FLIPPED: Sent when the VMR-7's allocator presenter has called the DirectDraw Flip method on the surface being presented."; break;
            case EC_WINDOW_DESTROYED: o << "EC_WINDOW_DESTROYED: The video renderer was destroyed or removed from the graph"; break;
            case EC_WMT_EVENT: o << "EC_WMT_EVENT: Sent by the Windows Media Format SDK when an application uses the ASF Reader filter to play ASF files protected by digital rights management (DRM)."; break;
            case EC_WMT_INDEX_EVENT: o << "EC_WMT_INDEX_EVENT: Sent by the Windows Media Format SDK when an application uses the ASF Writer to index Windows Media Video files."; break;

                //documented by Microsoft but not supported in the Platform SDK
                //              case EC_BANDWIDTHCHANGE : o << "EC_BANDWIDTHCHANGE: not supported"; break;
                //              case EC_CONTENTPROPERTY_CHANGED: o << "EC_CONTENTPROPERTY_CHANGED: not supported."; break;
                //              case EC_EOS_SOON: o << "EC_EOS_SOON: not supported"; break;
                //              case EC_ERRORABORTEX: o << "EC_ERRORABORTEX: An operation was aborted because of an error."; break;
                //              case EC_FILE_CLOSED: o << "EC_FILE_CLOSED: The source file was closed because of an unexpected event."; break;
                //              case EC_LOADSTATUS: o << "EC_LOADSTATUS: Notifies the application of progress when opening a network file."; break;
                //              case EC_MARKER_HIT: o << "EC_MARKER_HIT: not supported."; break;
                //              case EC_NEW_PIN: o << "EC_NEW_PIN: not supported."; break;
                //              case EC_PLEASE_REOPEN: o << "EC_PLEASE_REOPEN: The source file has changed."; break;
                //              case EC_PROCESSING_LATENCY: o << "EC_PROCESSING_LATENCY: Indicates the amount of time that a component is taking to process each sample."; break;
                //              case EC_RENDER_FINISHED: o << "EC_RENDER_FINISHED: Not supported."; break;
                //              case EC_SAMPLE_LATENCY: o << "EC_SAMPLE_LATENCY: Specifies how far behind schedule a component is for processing samples."; break;
                //              case EC_SAMPLE_NEEDED: o << "EC_SAMPLE_NEEDED: Requests a new input sample from the Enhanced Video Renderer (EVR) filter."; break;
                //              case EC_SCRUB_TIME: o << "EC_SCRUB_TIME: Specifies the time stamp for the most recent frame step."; break;
                //              case EC_STATUS: o << "EC_STATUS: Contains two arbitrary status strings."; break;
                //              case EC_VIDEOFRAMEREADY: o << "EC_VIDEOFRAMEREADY: A video frame is ready for display."; break;

            default: o << "Unknown event" << eventCode << "(" << param1 << ")"; break;
            }
#ifdef GRAPH_DEBUG
            qDebug() << eventDescription;
#endif
        }


        //interface management
        bool MediaObject::hasInterface(Interface iface) const
        {
            return iface == AddonInterface::TitleInterface;
        }

        QVariant MediaObject::interfaceCall(Interface iface, int command, const QList<QVariant> &params)
        {
            if (hasInterface(iface)) {

                switch (iface)
                {
                case TitleInterface:
                    switch (command)
                    {
                        case availableTitles:
                            return _iface_availableTitles();
                        case title:
                            return _iface_currentTitle();
                        case setTitle:
                            _iface_setTitle(params.first().toInt());
                            break;
                        case autoplayTitles:
                            return m_autoplayTitles;
                        case setAutoplayTitles:
                            m_autoplayTitles = params.first().toBool();
                            break;
                    }
                    break;
                default:
                    break;

                }
            }
            return QVariant();
        }


        //TitleInterface
        
        //this is called to set the time for the different titles
        qint64 MediaObject::titleAbsolutePosition(int title) const
        {
            if (title >= 0 && title < m_titles.count()) {
                return m_titles.at(title);
            } else {
                return 0;
            }
        }

        void MediaObject::setTitles(const QList<qint64> &titles)
        {
            const bool emitSignal = m_titles.count() != titles.count();
            m_titles = titles;
            if (emitSignal) {
                emit availableTitlesChanged(titles.count());
            }
        }


        int MediaObject::_iface_availableTitles() const
        {
            return m_titles.count();
        }

        int MediaObject::_iface_currentTitle() const
        {
            return m_currentTitle;
        }

        void MediaObject::_iface_setTitle(int title, bool bseek)
        {
            const int oldTitle = m_currentTitle;
            m_currentTitle = title;
            if (bseek) {
                //let's seek to the beginning of the song
                seek(0);
            } else {
                updateTargetTick();
            }
            if (oldTitle != title) {
                emit titleChanged(title);
                emit totalTimeChanged(totalTime());
            }
        }



    }
}

QT_END_NAMESPACE


#include "moc_mediaobject.cpp"
