/*  This file is part of the KDE project.

Copyright (C) 2007 Trolltech ASA. All rights reserved.

This library is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 2.1 or 3 of the License.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "fakesource.h"
#include "iodevicereader.h"
#include "qaudiocdreader.h"

#include "mediagraph.h"
#include "mediaobject.h"


#include <QtCore/QThread>
#include <QtCore/QUrl>
#include <QtGui/QWidget>

#include <qnetwork.h>


QT_BEGIN_NAMESPACE

uint qHash (const Phonon::DS9::Filter &f)
{
    return uint(static_cast<IBaseFilter*>(f));
}

namespace Phonon
{
    namespace DS9
    {
        class DummyWidget;

        static const long WM_GRAPHNOTIFY = WM_APP + 1;
        static DummyWidget *common = 0;

        //this class is used to get the events and redirect them to the objects
        class DummyWidget : public QWidget
        {
        public:
            DummyWidget()
            {
            }

            bool winEvent(MSG *msg, long *result)
            {
                if (msg->message == WM_GRAPHNOTIFY) {
                    MediaGraph *mg = reinterpret_cast<MediaGraph*>(msg->lParam);
                    if (mg && activeGraphs.contains(mg)) {
                        mg->handleEvents();
                    }
                }
                return QWidget::winEvent(msg, result);
            }

            QSet<MediaGraph*> activeGraphs;
        };

        class WorkerThread : public QThread
        {
            Q_OBJECT
        public:
            WorkerThread(const ComPointer<IGraphBuilder> graph) : QThread(), m_graph(graph), 
                m_checkingState(false), m_finished(false), m_currentWorkId(0)
            {
            }

            virtual void run()
            {
                m_waitMutex.lock();
                while (1) {

                    m_mutex.lock();
                    if (!m_finished && m_checkingState == false && m_queue.isEmpty()) {
                        m_mutex.unlock();
                        //let's wait on the wait condition
                        m_waitCondition.wait(&m_waitMutex);
                        m_mutex.lock();
                    }

                    if (m_finished) {
                        //that's the end if the thread execution
                        return;
                    }

                    if (m_queue.isEmpty()) {
                        Q_ASSERT(m_checkingState);
                        //we can get the state
                        ComPointer<IMediaControl> mc(m_graph, IID_IMediaControl);
                        OAFilterState s;
                        //blocking call
                        m_mutex.unlock();
                        HRESULT hr = S_OK;
                        while(m_checkingState) {
                            m_checkingState = false;
                            hr = mc->GetState(INFINITE, &s);
                        }

                        if (SUCCEEDED(hr)) {
                            if (s == State_Stopped) {
                                emit stateReady(Phonon::StoppedState);
                            } else if (s == State_Paused) {
                                emit stateReady(Phonon::PausedState);
                            } else /*if (s == State_Running)*/ {
                                emit stateReady(Phonon::PlayingState);
                            }
                        }
                    } else {
                        Work w = m_queue.dequeue();
                        m_mutex.unlock();
                        QSet<Filter> previous = MediaGraph::getAllFilters(m_graph);
                        HRESULT hr = S_OK;
                        if (w.filter) {
                            //let's render pins
                            foreach(OutputPin out, BackendNode::pins(w.filter, PINDIR_OUTPUT)) {
                                //blocking call
                                hr = m_graph->Render(out);
                                if (FAILED(hr)) {
                                    break;
                                }
                            }
                        } else if (!w.url.isEmpty()) {
                            //let's render a url (blocking call)
                            hr = m_graph->RenderFile(reinterpret_cast<const wchar_t *>(w.url.utf16()), 0);
                        }

                        if (hr != E_ABORT) {
                            //if it was aborted or the queue is not empty, another source will be loaded
                            emit asyncRenderFinished(w.id, hr, MediaGraph::getAllFilters(m_graph) - previous);
                        }
                    }
                }
            }

            //wants to know as soon as the state is set
            void addStateRequest()
            {
                QMutexLocker locker(&m_mutex);
                m_checkingState = true;
                m_waitCondition.wakeOne();
            }

            quint16 addUrlToRender(const QString &url)
            {
                QMutexLocker locker(&m_mutex);
                Work w;
                w.url = url;
                w.url.detach();
                w.id = m_currentWorkId++;
                m_queue.enqueue(w);
                m_waitCondition.wakeOne();
                return w.id;
            }

            quint16 addFilterToRender(const Filter &filter)
            {
                QMutexLocker locker(&m_mutex);
                Work w;
                w.filter = filter;
                w.id = m_currentWorkId++;
                m_queue.enqueue(w);
                m_waitCondition.wakeOne();
                return w.id;
            }

            //tells the thread to stop processing
            void signalStop()
            {
                QMutexLocker locker(&m_mutex);
                m_queue.clear();
                m_checkingState = false;
                m_finished = true;
                m_waitCondition.wakeOne();
            }


        Q_SIGNALS:
            void asyncRenderFinished(quint16, HRESULT, QSet<Filter>);
            void stateReady(Phonon::State);

        private:
            struct Work
            {
                quint16 id;
                Filter filter;
                QString url;
            };

            ComPointer<IGraphBuilder> m_graph;
            QQueue<Work> m_queue;
            bool m_checkingState;
            bool m_finished;
            quint16 m_currentWorkId;
            QWaitCondition m_waitCondition;
            QMutex m_mutex, m_waitMutex;
        };

        MediaGraph::MediaGraph(MediaObject *mo, short index) : m_fakeSource(new FakeSource()),
            m_isLoading(false), m_hasVideo(false), m_hasAudio(false), m_connectionsDirty(false),
            m_index(index), m_mediaObject(mo), 
            m_graph(CLSID_FilterGraph, IID_IGraphBuilder),
            m_thread(new WorkerThread(m_graph))
        {
            m_mediaControl = ComPointer<IMediaControl>(m_graph, IID_IMediaControl);
            Q_ASSERT(m_mediaControl);
            m_mediaSeeking = ComPointer<IMediaSeeking>(m_graph, IID_IMediaSeeking);
            Q_ASSERT(m_mediaSeeking);
            m_mediaEvent = ComPointer<IMediaEventEx>(m_graph, IID_IMediaEventEx);
            Q_ASSERT(m_mediaEvent);

            //event management
            dummyWidget(); // ensures it is created;

            common->activeGraphs += this;

            //we route the events to the same widget
            HRESULT hr = m_mediaEvent->SetNotifyWindow(reinterpret_cast<OAHWND>(common->winId()), WM_GRAPHNOTIFY,
                reinterpret_cast<LONG_PTR>(this) );

            if (m_mediaObject->catchComError(hr)) {
                return;
            }

            hr = m_graph->AddFilter(m_fakeSource, L"Fake Source");
            if (m_mediaObject->catchComError(hr)) {
                return;
            }

            connect(m_thread, SIGNAL(asyncRenderFinished(quint16, HRESULT, QSet<Filter>)),
                SLOT(finishLoading(quint16, HRESULT, QSet<Filter>)));
            connect(m_thread, SIGNAL(stateReady(Phonon::State)),
                SLOT(stateReady(Phonon::State)));
            m_thread->start();
        }

        MediaGraph::~MediaGraph()
        {
            m_graph->Abort(); //abort any current operation
            m_thread->signalStop();
            m_thread->wait();
            delete m_thread;
            common->activeGraphs -= this;
        }

        QWidget *MediaGraph::dummyWidget()
        {
            if (common == 0) {
                common = new DummyWidget;
            }
            return common;
        }


        short MediaGraph::index() const
        {
            return m_index;
        }

        void MediaGraph::grabNode(BackendNode *node)
        {
            FILTER_INFO info;
            const Filter filter = node->filter(m_index);
            if (filter) {
                filter->QueryFilterInfo(&info);
                if (info.pGraph == 0) {
                    HRESULT hr = m_graph->AddFilter(filter, 0);
                    m_mediaObject->catchComError(hr);
                } else {
                    info.pGraph->Release();
                }
            }
        }

        void MediaGraph::ensureSourceDisconnected()
        {
            foreach(BackendNode *node, m_sinkConnections) {
                const Filter currentFilter = node->filter(m_index);
                foreach(const InputPin inpin, BackendNode::pins(currentFilter, PINDIR_INPUT)) {
                    foreach(const OutputPin outpin, BackendNode::pins(m_fakeSource, PINDIR_OUTPUT)) {
                        tryDisconnect(outpin, inpin);
                    }

                    foreach(const Filter &filter, m_decoders) {
                        const OutputPin outpin = BackendNode::pins(filter, PINDIR_OUTPUT).first();
                        tryDisconnect(outpin, inpin);
                    }
                }
            }
        }

        void MediaGraph::ensureSourceConnectedTo(bool force)
        {
            if (m_connectionsDirty == false && force == false) {
                return;
            }

            m_connectionsDirty = false;
            ensureSourceDisconnected();

            //reconnect the pins
            foreach(BackendNode *node, m_sinkConnections) {
                const Filter currentFilter = node->filter(m_index);
                foreach(const InputPin &inpin, BackendNode::pins(currentFilter, PINDIR_INPUT)) {
                    //we ensure the filter belongs to the graph
                    FILTER_INFO info;
                    currentFilter->QueryFilterInfo(&info);
                    if (info.pGraph == 0) {
                        m_graph->AddFilter(currentFilter, 0);
                    } else {
                        info.pGraph->Release();
                    }

                    foreach(const Filter &filter, m_decoders) {
                        //a decoder has only one output
                        const OutputPin outpin = BackendNode::pins(filter, PINDIR_OUTPUT).first();
                        if (tryConnect(outpin, inpin)) {
                            break;
                        }
                    }
                }
            }
        }

        QSet<Filter> MediaGraph::getAllFilters(ComPointer<IGraphBuilder> graph)
        {
            QSet<Filter> ret;
            ComPointer<IEnumFilters> enumFilters;
            graph->EnumFilters(&enumFilters);
            Filter current;
            while( enumFilters->Next(1, &current, 0) == S_OK) {
                ret += current;
            }
            return ret;
        }

        QSet<Filter> MediaGraph::getAllFilters() const
        {
            return getAllFilters(m_graph);
        }


        bool MediaGraph::isSeekable()
        {
            if (m_mediaSeeking == 0) {
                return false;
            }

            removeUnusedFilters();
            DWORD caps = AM_SEEKING_CanSeekAbsolute;
            HRESULT hr = m_mediaSeeking->CheckCapabilities(&caps);
            restoreUnusedFilters();
            return SUCCEEDED(hr);
        }

        qint64 MediaGraph::absoluteTotalTime() const
        {
            qint64 ret = -1;
            if (m_mediaSeeking) {
                HRESULT hr = m_mediaSeeking->GetDuration(&ret);
                if (FAILED(hr)) {
                    ret = absoluteCurrentTime(); //that's the least we know
                } else {
                    ret /= 10000; //convert to milliseconds
                }
            }
            return ret;
        }

        qint64 MediaGraph::absoluteCurrentTime() const
        {
            qint64 ret = -1;
            if (m_mediaSeeking) {
                HRESULT hr = m_mediaSeeking->GetCurrentPosition(&ret);
                if (FAILED(hr)) {
                    return ret;
                }
                ret /= 10000; //convert to milliseconds
            }
            return ret;
        }

        Phonon::MediaSource MediaGraph::mediaSource() const
        {
            return m_mediaSource;
        }

        void MediaGraph::play()
        {
            removeFilter(m_fakeSource);
            ensureSourceConnectedTo();
            removeUnusedFilters();
            m_mediaControl->Run();
            m_thread->addStateRequest();
        }

        void MediaGraph::pause()
        {
            ensureSourceConnectedTo();
            removeUnusedFilters();
            m_mediaControl->Pause();
            m_thread->addStateRequest();
        }

        void MediaGraph::stop()
        {
            m_mediaControl->Stop();
            restoreUnusedFilters();
            m_thread->addStateRequest();
        }

        bool MediaGraph::isLoading() const
        {
            return m_isLoading;
        }

        HRESULT MediaGraph::absoluteSeek(qint64 time)
        {
            qint64 newtime = time * 10000;
            HRESULT hr = m_mediaSeeking->SetPositions(&newtime, AM_SEEKING_AbsolutePositioning,
                0, AM_SEEKING_NoPositioning);
            return hr;
        }

        HRESULT MediaGraph::removeFilter(const Filter& filter)
        {
            FILTER_INFO info;
            filter->QueryFilterInfo(&info);
#ifdef GRAPH_DEBUG
            qDebug() << "removeFilter" << QString::fromWCharArray(info.achName);
#endif
            if (info.pGraph) {
                info.pGraph->Release();
                return m_graph->RemoveFilter(filter);
            }

            //already removed
            return S_OK;
        }

        HRESULT MediaGraph::cleanup()
        {
            stop();

            ensureSourceDisconnected();

            QSet<Filter> list = m_decoders;
            if (m_demux) {
                list << m_demux;
            }
            if (m_realSource) {
                list << m_realSource;
            }

            foreach(const Filter &decoder, m_decoders) {
                list += getFilterChain(m_demux, decoder);
            }

            foreach(const Filter &filter, list) {
                HRESULT hr = removeFilter(filter);
                if(FAILED(hr)) {
                    return hr;
                }
            }

            //Let's reinitialize the internal lists
            m_decoders.clear();
            m_demux = Filter();
            m_realSource = Filter();
            m_mediaSource = Phonon::MediaSource();

            //in case we're already loading a source, we abort that
            m_graph->Abort();
            return S_OK;
        }


        bool MediaGraph::disconnectNodes(BackendNode *source, BackendNode *sink)
        {
            const Filter sinkFilter = sink->filter(m_index);
            const QList<InputPin> inputs = BackendNode::pins(sinkFilter, PINDIR_INPUT);

            QList<OutputPin> outputs;
            if (source == m_mediaObject) {
                outputs = BackendNode::pins(m_fakeSource, PINDIR_OUTPUT);
                foreach(const Filter dec, m_decoders) {
                    outputs += BackendNode::pins(dec, PINDIR_OUTPUT);
                }
            } else {
                outputs = BackendNode::pins(source->filter(m_index), PINDIR_OUTPUT);
            }


            foreach(InputPin inPin, inputs) {
                foreach(OutputPin outPin, outputs) {
                    tryDisconnect(outPin, inPin);
                }
            }

            if (m_sinkConnections.remove(sink)) {
                m_connectionsDirty = true;
            }
            return true;
        }

        bool MediaGraph::tryDisconnect(const OutputPin &out, const InputPin &in)
        {
            bool ret = false;

            OutputPin output;
            if (SUCCEEDED(in->ConnectedTo(&output))) {

                if (output == out) {
                    //we need a simple disconnection
                    ret = SUCCEEDED(out->Disconnect()) && SUCCEEDED(in->Disconnect());
                } else {
                    InputPin in2;
                    if (SUCCEEDED(out->ConnectedTo(&in2))) {
                        PIN_INFO info;
                        in2->QueryPinInfo(&info);
                        Filter tee(info.pFilter);
                        CLSID clsid;
                        tee->GetClassID(&clsid);
                        if (clsid == CLSID_InfTee) {
                            //we have to remove all intermediate filters between the tee and the sink
                            PIN_INFO info;
                            in->QueryPinInfo(&info);
                            Filter sink(info.pFilter);
                            QSet<Filter> list = getFilterChain(tee, sink);
                            list -= sink;
                            list -= tee;
                            out->QueryPinInfo(&info);
                            Filter source(info.pFilter);

                            if (list.isEmpty()) {
                                output->QueryPinInfo(&info);
                                if (Filter(info.pFilter) == tee) {
                                    ret = SUCCEEDED(output->Disconnect()) && SUCCEEDED(in->Disconnect());
                                }
                            } else {
                                ret = true;
                                foreach(Filter f, list) {
                                    ret = ret && SUCCEEDED(removeFilter(f));
                                }
                            }

                            //Let's try to see if the Tee filter is still useful
                            if (ret) {
                                int connections = 0;
                                foreach (OutputPin out, BackendNode::pins(tee, PINDIR_OUTPUT)) {
                                    InputPin p;
                                    if ( SUCCEEDED(out->ConnectedTo(&p))) {
                                        connections++;
                                    }
                                }
                                if (connections == 0) {
                                    //this avoids a crash if the filter is destroyed
                                    //by the subsequent call to removeFilter
                                    output = OutputPin();
                                    removeFilter(tee); //there is no more output for the tee, we remove it
                                }
                            }
                        }
                    }
                }
            }
            return ret;
        }

        bool MediaGraph::tryConnect(const OutputPin &out, const InputPin &newIn)
        {
            ///The management of the creation of the Tees is done here (this is the only place where we call IPin::Connect
            InputPin inPin;
            if (SUCCEEDED(out->ConnectedTo(&inPin))) {

                //the fake source has another mechanism for the connection
                if (BackendNode::pins(m_fakeSource, PINDIR_OUTPUT).contains(out)) {
                    return false;
                }

                //the output pin is already connected
                PIN_INFO info;
                inPin->QueryPinInfo(&info);
                Filter filter(info.pFilter); //this will ensure the interface is "Release"d
                CLSID clsid;
                filter->GetClassID(&clsid);
                if (clsid == CLSID_InfTee) {
                    //there is already a Tee (namely 'filter') in use
                    foreach(OutputPin pin, BackendNode::pins(filter, PINDIR_OUTPUT)) {
                        if (VFW_E_NOT_CONNECTED == pin->ConnectedTo(&inPin)) {
                            return SUCCEEDED(pin->Connect(newIn, 0));
                        }
                    }

                    //we shoud never go here
                    return false;
                } else {
                    AM_MEDIA_TYPE type;
                    out->ConnectionMediaType(&type);

                    //first we disconnect the current connection (and we save the current media type)
                    if (!tryDisconnect(out, inPin)) {
                        return false;
                    }

                    //..then we try to connect the new node
                    if (SUCCEEDED(out->Connect(newIn, 0))) {

                        //we have to insert the Tee
                        if (!tryDisconnect(out, newIn)) {
                            return false;
                        }

                        Filter filter(CLSID_InfTee, IID_IBaseFilter);
                        if (!filter) {
                            //rollback
                            m_mediaObject->catchComError(m_graph->Connect(out, inPin));
                            return false;
                        }

                        if (m_mediaObject->catchComError(m_graph->AddFilter(filter, 0))) {
                            return false;
                        }


                        InputPin teeIn = BackendNode::pins(filter, PINDIR_INPUT).first(); //a Tee has always one input
                        HRESULT hr = out->Connect(teeIn, &type);
                        if (FAILED(hr)) {
                            hr = m_graph->Connect(out, teeIn);
                        }
                        if (m_mediaObject->catchComError(hr)) {
                            m_mediaObject->catchComError(m_graph->Connect(out, inPin));
                            return false;
                        }

                        OutputPin teeOut = BackendNode::pins(filter, PINDIR_OUTPUT).last(); //the last is always the one that's not connected

                        hr = m_graph->Connect(teeOut, inPin);
                        if (m_mediaObject->catchComError(hr)) {
                            m_mediaObject->catchComError(m_graph->Connect(out, inPin));
                            return false;
                        }

                        teeOut = BackendNode::pins(filter, PINDIR_OUTPUT).last(); //the last is always the one that's not connected
                        if (m_mediaObject->catchComError(m_graph->Connect(teeOut, newIn))) {
                            m_mediaObject->catchComError(m_graph->Connect(out, inPin));
                            return false;
                        }

                        return true;
                    } else {
                        //we simply reconnect the pins as they
                        m_mediaObject->catchComError(m_graph->Connect(out, inPin));
                        return false;
                    }
                }

            } else {
                return SUCCEEDED(m_graph->Connect(out, newIn));
            }
        }

        bool MediaGraph::connectNodes(BackendNode *source, BackendNode *sink)
        {
            bool ret = false;
            const QList<InputPin> inputs = BackendNode::pins(sink->filter(m_index), PINDIR_INPUT);
            const QList<OutputPin> outputs = BackendNode::pins(source == m_mediaObject ? m_fakeSource : source->filter(m_index), PINDIR_OUTPUT);

#ifdef GRAPH_DEBUG
            qDebug() << Q_FUNC_INFO << source << sink << this;
#endif

            foreach(OutputPin outPin, outputs) {

                InputPin p;
                foreach(InputPin inPin, inputs) {
                    if (tryConnect(outPin, inPin)) {
                        //tell the sink node that it just got a new input
                        sink->connected(source, inPin);
                        ret = true;
                        if (source == m_mediaObject) {
                            m_connectionsDirty = true;
                            m_sinkConnections += sink;
#ifdef GRAPH_DEBUG
                            qDebug() << "found a sink connection" << sink << m_sinkConnections.count();
#endif
                        }
                        break;
                    }
                }
            }

            return ret;
        }


        HRESULT MediaGraph::loadSource(const Phonon::MediaSource &source)
        {
            m_isLoading = true;
            m_hasVideo = false;
            m_hasAudio = false;

            //we try to reset the clock here
            absoluteSeek(0);

            //cleanup of the previous filters
            HRESULT hr = cleanup();
            if (FAILED(hr)) {
                return hr;
            }

            //setting the current source
            m_mediaSource = source;

            const QSet<Filter> previous = getAllFilters();
            switch (source.type())
            {
            case Phonon::MediaSource::Disc:
                if (source.discType() == Phonon::Dvd) {
                    //TODO : add the navigation system

                    m_realSource = Filter(CLSID_DVDNavigator, IID_IBaseFilter);
                    if (m_realSource) {
                        return REGDB_E_CLASSNOTREG;
                    }

                    hr = m_graph->AddFilter(m_realSource, L"DVD Navigator");
                    if (FAILED(hr)) {
                        return hr;
                    }


                } else if (source.discType() == Phonon::Cd) {
                    m_realSource = Filter(new QAudioCDPlayer);
                    hr = m_graph->AddFilter(m_realSource, L"Audio CD Reader");
                    if (FAILED(hr)) {
                        return hr;
                    }

                }
                m_renderId = m_thread->addFilterToRender(m_realSource);
                return hr;
            case Phonon::MediaSource::Invalid:
                return hr;
            case Phonon::MediaSource::Url:
            case Phonon::MediaSource::LocalFile:
                {
                    QString url;
                    if (source.url().isValid()) {
                        url = source.url().toString();
                    } else if (!source.fileName().isEmpty()) {
                        url = source.fileName();
                    }
                    m_renderId = m_thread->addUrlToRender(url);
                }
                break;
            case Phonon::MediaSource::Stream:
                {
                    m_realSource = Filter(new IODeviceReader(source));
                    HRESULT hr = m_graph->AddFilter(m_realSource, L"Phonon Stream Reader");

                    if (FAILED(hr)) {
                        return hr;
                    }

                    m_renderId = m_thread->addFilterToRender(m_realSource);
                }
                break;
            }

            return hr;
        }

        void MediaGraph::stateReady(Phonon::State state)
        {
            emit stateReady(this, state);
        }

        void MediaGraph::finishLoading(quint16 workId, HRESULT hr, QSet<Filter> newlyCreated)
        {
            if (m_renderId == workId) {
                m_isLoading = false;
                emit loadingFinished(this, reallyFinishLoading(hr, newlyCreated));
            }
        }


        HRESULT MediaGraph::reallyFinishLoading(HRESULT hr, const QSet<Filter> &newlyCreated)
        {
            if (FAILED(hr)) {
                return hr;
            }

            //we keep the source and all the way down to the decoders
            QSet<Filter> keptFilters;
            foreach(const Filter &filter, newlyCreated) {
                if (isSourceFilter(filter)) {
                    m_realSource = filter; //save the source filter
                    if (!m_demux ) {
                        m_demux = filter; //in the WMV case, the demuxer is the source filter itself
                    }
                    keptFilters << filter;
                } else if (isDemuxerFilter(filter)) {
                    m_demux = filter;
                    keptFilters << filter;
                } else if (isDecoderFilter(filter)) {
                    m_decoders += filter;
                    keptFilters << filter;
                } 
            }

            //hack to make the unencoded wav file play
            ///we need to do something smarter that checks the output pins of the demuxer for unencoded streams.
            if (m_decoders.isEmpty() && m_demux &&
                BackendNode::pins(m_demux, PINDIR_OUTPUT).count() == 1) {
                    m_decoders += m_demux;
            }

            foreach(const Filter &decoder, m_decoders) {
                keptFilters += getFilterChain(m_demux, decoder);
            }

            //now we need to decide what are the filters we want to keep
            //usually these are the demuxer and the decoders

            foreach(const Filter &filter, newlyCreated - keptFilters) {
                hr = removeFilter(filter);
                if (FAILED(hr)) {
                    return hr;
                }
            }

            if(!m_realSource) {
                //there should we a source
                return E_FAIL;
            }

            ensureSourceConnectedTo(true);
            return hr;
        }

        void MediaGraph::restoreUnusedFilters()
        {
#ifdef GRAPH_DEBUG
            qDebug() << Q_FUNC_INFO << m_unusedFilters;
#endif
            ComPointer<IGraphConfig> graphConfig(m_graph, IID_IGraphConfig);
            foreach(const Filter filter, m_unusedFilters) {
                m_graph->AddFilter(filter, 0);
            }
            m_unusedFilters.clear();
        }

        void MediaGraph::removeUnusedFilters()
        {
            //called from play and pause to 'clean' the graph
            QSet<Filter> usedFilters;
            const Filter root = m_realSource ? m_realSource : m_fakeSource;
            usedFilters += root;
            usedFilters += connectedFilters( root);

            m_unusedFilters += getAllFilters() - usedFilters;
            ComPointer<IGraphConfig> graphConfig(m_graph, IID_IGraphConfig);
            foreach(const Filter filter, m_unusedFilters) {
                FILTER_INFO info;
                filter->QueryFilterInfo(&info);
                if (info.pGraph) {
                    info.pGraph->Release();
                    graphConfig->RemoveFilterEx(filter, REMFILTERF_LEAVECONNECTED);
                }
            }
            removeUselessDecoders();
#ifdef GRAPH_DEBUG
            qDebug() << Q_FUNC_INFO << m_unusedFilters;
#endif
        }

        QSet<Filter> MediaGraph::connectedFilters(const Filter &filter)
        {
            QSet<Filter> ret;
            foreach(OutputPin out, BackendNode::pins(filter, PINDIR_OUTPUT)) {
                InputPin in;
                if (SUCCEEDED(out->ConnectedTo(&in))) {
                    PIN_INFO info;
                    in->QueryPinInfo(&info);
                    const Filter current(info.pFilter);
                    if (current) {
                        ret += current;
                        ret += connectedFilters(current);
                    }
                }
            }
            return ret;
        }

        HRESULT MediaGraph::removeUselessDecoders()
        {
            foreach(const Filter &decoder, m_decoders) {
                QList<OutputPin> list = BackendNode::pins(decoder, PINDIR_OUTPUT);
                if (!list.isEmpty()) {
                    OutputPin pin = list.first();
                    InputPin input;
                    if (pin->ConnectedTo(&input) == VFW_E_NOT_CONNECTED) {
                        //"we should remove this filter from the graph
                        HRESULT hr = removeFilter(decoder);
                        if (FAILED(hr)) {
                            return hr;
                        }
                    }
                }
            }
            return S_OK;
        }

        //utility functions
        QSet<Filter> MediaGraph::getFilterChain(const Filter &source, const Filter &sink)
        {
            QSet<Filter> ret;
            Filter current = sink;
            while (current && BackendNode::pins(current, PINDIR_INPUT).count() == 1 && current != source) {
                ret += current;
                InputPin pin = BackendNode::pins(current, PINDIR_INPUT).first();
                current = Filter();
                OutputPin output;
                if (pin->ConnectedTo(&output) == S_OK) {
                    PIN_INFO info;
                    if (SUCCEEDED(output->QueryPinInfo(&info)) && info.pFilter) {
                        current = Filter(info.pFilter); //this will take care of releasing the interface pFilter
                    }
                }
            }
            if (current != source) {
                //the soruce and sink don't seem to be connected
                ret.clear();
            }
            return ret;
        }

        bool MediaGraph::isDecoderFilter(const Filter &filter)
        {
            if (filter == 0) {
                return false;
            }
#ifdef GRAPH_DEBUG
            {
                FILTER_INFO info;
                filter->QueryFilterInfo(&info);
                qDebug() << Q_FUNC_INFO << QString::fromWCharArray(info.achName);
                if (info.pGraph) {
                    info.pGraph->Release();
                }
            }
#endif


            QList<InputPin> inputs = BackendNode::pins(filter, PINDIR_INPUT);
            QList<OutputPin> outputs = BackendNode::pins(filter, PINDIR_OUTPUT);

            //TODO: find a better way to detect if a node is a decoder
            if (inputs.count() == 0 || outputs.count() ==0) {
                return false;
            }

            //the input pin must be encoded data
            AM_MEDIA_TYPE type;
            HRESULT hr = inputs.first()->ConnectionMediaType(&type);
            if (FAILED(hr)) {
                return false;
            }

            if (type.majortype != MEDIATYPE_Video &&
                type.majortype != MEDIATYPE_Audio) {
                    return false;
            }

            //...and the output must be decoded
            AM_MEDIA_TYPE type2;
            hr = outputs.first()->ConnectionMediaType(&type2);
            if (FAILED(hr)) {
                return false;
            }

            if (type.majortype != type2.majortype) {
                return false;
            }

            if (type.majortype == MEDIATYPE_Video) {
                m_hasVideo = true;
            } else {
                m_hasAudio = true;
            }

#ifdef GRAPH_DEBUG
            {
                FILTER_INFO info;
                filter->QueryFilterInfo(&info);
                qDebug() << "found a decoder filter" << QString::fromWCharArray(info.achName);
                if (info.pGraph) {
                    info.pGraph->Release();
                }
            }
#endif

            return true;
        }

        bool MediaGraph::isSourceFilter(const Filter &filter) const
        {
            //a source filter is one that has no input
            return BackendNode::pins(filter, PINDIR_INPUT).isEmpty();
        }

        bool MediaGraph::isDemuxerFilter(const Filter &filter) const
        {
            QList<InputPin> inputs = BackendNode::pins(filter, PINDIR_INPUT);
            QList<OutputPin> outputs = BackendNode::pins(filter, PINDIR_OUTPUT);

#ifdef GRAPH_DEBUG
            {
                FILTER_INFO info;
                filter->QueryFilterInfo(&info);
                qDebug() << Q_FUNC_INFO << QString::fromWCharArray(info.achName);
                if (info.pGraph) {
                    info.pGraph->Release();
                }
            }
#endif

            if (inputs.count() != 1 || outputs.count() == 0) {
                return false; //a demuxer has only one input
            }

            AM_MEDIA_TYPE type;
            HRESULT hr = inputs.first()->ConnectionMediaType(&type);
            if (FAILED(hr)) {
                return false;
            }

            if (type.majortype != MEDIATYPE_Stream) {
                return false;
            }

            foreach(const OutputPin &outpin, outputs) {
                AM_MEDIA_TYPE type;
                //for now we support only video and audio
                hr = outpin->ConnectionMediaType(&type);
                if (SUCCEEDED(hr) && 
                    type.majortype != MEDIATYPE_Video && type.majortype != MEDIATYPE_Audio) {
                        return false;
                }
            }
#ifdef GRAPH_DEBUG
            {
                FILTER_INFO info;
                filter->QueryFilterInfo(&info);
                qDebug() << "found a demuxer filter" << QString::fromWCharArray(info.achName);
                if (info.pGraph) {
                    info.pGraph->Release();
                }
            }
#endif
            return true;
        }

        QMultiMap<QString, QString> MediaGraph::metadata() const
        {
            QMultiMap<QString, QString> ret;
            ComPointer<IAMMediaContent> mediaContent(m_demux, IID_IAMMediaContent);
            if (mediaContent) {
                //let's get the meta data
                BSTR str;
                HRESULT hr = mediaContent->get_AuthorName(&str);
                if (SUCCEEDED(hr)) {
                    ret.insert(QLatin1String("ARTIST"), QString::fromWCharArray(str));
                    SysFreeString(str);
                }
                hr = mediaContent->get_Title(&str);
                if (SUCCEEDED(hr)) {
                    ret.insert(QLatin1String("TITLE"), QString::fromWCharArray(str));
                    SysFreeString(str);
                }
                hr = mediaContent->get_Description(&str);
                if (SUCCEEDED(hr)) {
                    ret.insert(QLatin1String("DESCRIPTION"), QString::fromWCharArray(str));
                    SysFreeString(str);
                }
                hr = mediaContent->get_Copyright(&str);
                if (SUCCEEDED(hr)) {
                    ret.insert(QLatin1String("COPYRIGHT"), QString::fromWCharArray(str));
                    SysFreeString(str);
                }
                hr = mediaContent->get_MoreInfoText(&str);
                if (SUCCEEDED(hr)) {
                    ret.insert(QLatin1String("MOREINFO"), QString::fromWCharArray(str));
                    SysFreeString(str);
                }
            }
            return ret;
        }

        void MediaGraph::handleEvents()
        {
            long eventCode;
            LONG_PTR param1, param2;
            while (m_mediaEvent && SUCCEEDED(m_mediaEvent->GetEvent(&eventCode, &param1, &param2, 0))) {
                m_mediaObject->handleEvents(this, eventCode, param1);
                m_mediaEvent->FreeEventParams(eventCode, param1, param2);
            }
        }

        Filter MediaGraph::realSource() const
        {
            return m_realSource;
        }

        QList<qint64> MediaGraph::titles() const
        {
            //for now we only manage that for the audio cd
            ComPointer<ITitleInterface> titleIFace(m_realSource, IID_ITitleInterface);
            if (titleIFace) {
                return titleIFace->titles();
            } else {
                // the default value: only one title that starts at position 0
                return QList<qint64>() << 0;
            }
        }



        HRESULT MediaGraph::saveToFile(const QString &filepath) const
        {
            const WCHAR wszStreamName[] = L"ActiveMovieGraph";
            HRESULT hr;
            ComPointer<IStorage> storage;

            // First, create a document file that will hold the GRF file
            hr = StgCreateDocfile(reinterpret_cast<const wchar_t *>(filepath.utf16()),
                STGM_CREATE | STGM_TRANSACTED | STGM_READWRITE |
                STGM_SHARE_EXCLUSIVE,
                0, &storage);

            if (FAILED(hr)) {
                return hr;
            }

            // Next, create a stream to store.
            ComPointer<IStream> stream;
            hr = storage->CreateStream(wszStreamName,
                STGM_WRITE | STGM_CREATE | STGM_SHARE_EXCLUSIVE,
                0, 0, &stream);

            if (FAILED(hr)) {
                return hr;
            }

            // The IpersistStream::Save method converts a stream into a persistent object.
            ComPointer<IPersistStream> persist(m_graph, IID_IPersistStream);
            hr = persist->Save(stream, TRUE);
            if (SUCCEEDED(hr)) {
                hr = storage->Commit(STGC_DEFAULT);
            }

            return hr;
        }

    }
}

QT_END_NAMESPACE

#include "moc_mediagraph.cpp"
#include "mediagraph.moc"
