/* super.c
 * David Lutz
 * Error handling by Jason Hunter
 *
 * Routines for reading and manipulating the ext2 superblock
 */

#include <stdio.h>
#include <time.h>
#define _SUPER_C_
#include "ext2.h"
#include "proto.h"

/*
 * Global variables calculated from the super block info
 */
super_block sb;
int BLOCK_SIZE;
int FRAG_SIZE;

/* load_super
 *
 * returns 0 for failure
 *        -1 for bad magic
 *         1 for OK
 */
int load_super( void )
{
  if( readdisk( (byte *)&sb, 1024, sizeof(super_block) ) !=
     sizeof(super_block) )
  {
    fprintf( stderr, "Drive unreadable.\n" );
    return( -1 );		/* read failed */
  }

  /* Now set a few of our (shhhh!) Global Variables */
  BLOCK_SIZE = EXT2_MIN_BLOCK << sb.s_log_block_size;
  FRAG_SIZE = EXT2_MIN_FRAG << sb.s_log_frag_size;

  if( sb.s_magic != EXT2_SUPER_MAGIC )
  {
    fprintf( stderr, "Bad magic number in superblock.  Not Linux.\n" );
    return( -1 );		/* bad magic number */
  }

  return( 1 );			/* everything OK */
}

/* print_super
 *
 * print out a super block (good for debugging)
 */
void print_super( void )
{
  printf( "Inode Count: %lu\n", sb.s_inodes_count );
  printf( "Block Count: %lu\n", sb.s_blocks_count );
  printf( "Reserved Block Count: %lu\n", sb.s_r_blocks_count );
  printf( "Free Blocks: %lu\n", sb.s_free_blocks_count );
  printf( "Free Inodes: %lu\n", sb.s_free_inodes_count );
  printf( "First Data Block: %lu\n", sb.s_first_data_block );
  printf( "Log Block Size: %lu\n", sb.s_log_block_size );
  printf( "Log Frag Size: %ld\n", sb.s_log_frag_size );
  printf( "Blocks per Group: %lu\n", sb.s_blocks_per_group );
  printf( "Fragments per Group: %lu\n", sb.s_frags_per_group );
  printf( "Inodes per Group: %lu\n", sb.s_inodes_per_group );
  printf( "Mount Time: %s", ctime( &(sb.s_mtime) ) );
  printf( "Write Time: %s", ctime( &(sb.s_wtime) ) );
  printf( "Mount Count: %u\n", sb.s_mnt_count );
  printf( "Max Mount Count: %d\n", sb.s_max_mnt_count );
  printf( "Magic Number: %X  (%s)\n", sb.s_magic,
	 sb.s_magic == EXT2_SUPER_MAGIC ? "OK" : "BAD" );
  printf( "File System State: %X\n", sb.s_state );
  printf( "Error Behaviour: %X\n", sb.s_errors );
  printf( "Pad Value: %u\n", sb.s_pad );
  printf( "Last Check: %s", ctime( &(sb.s_lastcheck) ) );
  printf( "Check Interval: %lu\n", sb.s_checkinterval );
  printf( "Creator OS: %lu\n", sb.s_creator_os );
  printf( "Revision Level: %lu\n", sb.s_rev_level );
  printf( "Reserved Block Default UID: %u\n", sb.s_def_resuid );
  printf( "Reserved Block Default GID: %u\n", sb.s_def_resgid );
  printf( "An' a bunch of padding we ain't gonna show ya!\n" );
}
