#define EMULATE_SETFILEPOINTEREX
/*
 * win32.c: a stdio-like disk I/O implementation for low-level disk access on Win32
 *          can access an NTFS-volume while mounted!
 *
 *
 * when compiling with -mno-cygwin, the program uses MSVCRT.DLL,
 * which used "%I64d" instead of "%lld" for "long long int"
 */

#include <windows.h>
#include <winioctl.h>

#include <stdio.h>
#include <ctype.h>

#include "win32.h"
//#define DEBUG
#define FORCE_ALIGNED_READ
#define SUPPORT_FLOPPY
#define SUPPORT_NT_NAMES
//#define SUPPORT_XP_NAMES
#define SUPPORT_LINUX_NAMES
#define SUPPORT_GRUB_NAMES
#define CACHED_READ

int win32_quiet=0;

#ifdef EMULATE_SETFILEPOINTEREX
  BOOL WINAPI SetFilePointerEx(
  HANDLE hFile,
  LARGE_INTEGER liDistanceToMove,
  PLARGE_INTEGER lpNewFilePointer,
  DWORD dwMoveMethod
  )
{
  liDistanceToMove.LowPart = SetFilePointer(hFile, liDistanceToMove.LowPart, &liDistanceToMove.HighPart, dwMoveMethod);
  if (liDistanceToMove.LowPart == INVALID_SET_FILE_POINTER && GetLastError() != NO_ERROR) {
    lpNewFilePointer->QuadPart = -1;
    return FALSE;
  } else {
    lpNewFilePointer->QuadPart = liDistanceToMove.QuadPart;
    return TRUE;
  }
}
#endif

typedef struct {
  HANDLE handle;
  DWORD signature;
  int drivenumber;
  int partnumber;
  LARGE_INTEGER part_start;
  LARGE_INTEGER part_end;
  LARGE_INTEGER current_pos;
  char device[10];
} win32_fd_t;

win32_fd_t win32_fds[64] = {
  {(HANDLE)42}, // fd==0 breaks some code in attr.c, so skip it
  {0}};

#ifdef DEBUG
static __inline__ void Dprintf(const char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  va_end(ap);
}
#else
static void Dprintf(const char *fmt, ...) {}
#endif

#define perror(msg) win32_perror(__FILE__,__LINE__,__FUNCTION__,msg)

int win32_perror(char* file, int line, char* func, char* msg)
{
  char buffer[1024] = "";
  DWORD err = GetLastError();
  if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, err, 0, buffer, sizeof(buffer), NULL) <= 0) {
    sprintf(buffer, "HRESULT 0x%lx", err);
  }
  if (buffer[strlen(buffer)-1]=='\r') {buffer[strlen(buffer)-1]=='\0';}
  if (buffer[strlen(buffer)-1]=='\n') {buffer[strlen(buffer)-1]=='\0';}
  fprintf(stderr, "%s(%d):%s\t%s %s\n", file, line, func, msg, buffer);
  return 0;
}

/*
 * open a file
 * if name is in format "(hd[0-9],[0-9])" then open a partition
 * if name is in format "(hd[0-9])" then open a volume
 * if name is in format "/dev/hd[a-z][0-9]" then open a volume
 * if name is in format "/dev/hd[a-z]" then open a partition
 * if name is in format "[C-Z]:" then open a partition
 * else open a file
 */

int win32_open(const char* name, int mode)
{
  int drive = 0;
  int part = 0;
  int fd = 1;
  Dprintf("win32_open(%s,%d)\n", name, mode);
  
  char szBuf[256];
  sprintf(szBuf, "[Bazooka] win32_open(%s, %d)\n", name, mode);
  OutputDebugString(szBuf);

  // grab a free fd
  while (win32_fds[fd].handle != 0) {
    fd++;
    if (fd>sizeof(win32_fds)/sizeof(win32_fds[0])) {
      perror("too many open files");
      OutputDebugString("[Bazooka] too many open files");
      return -1;
    }
  }
  memset(&win32_fds[fd],0,sizeof(win32_fds[0]));

  sprintf(win32_fds[fd].device, name);
  char filename[1024];

#ifdef SUPPORT_NT_NAMES
OutputDebugString("[Bazooka] #ifdef SUPPORT_NT_NAMES\n");
  if ((name != NULL)
      && (toupper(name[0])>='C') && (toupper(name[0]) <='Z')
      && (name[1] == ':')) {
    // open the device "\\.\C:"
    sprintf(filename,"\\\\.\\%s", name);
    HANDLE handle = CreateFile(filename,
			       GENERIC_READ,
			       FILE_SHARE_READ | FILE_SHARE_WRITE,
			       NULL,
			       OPEN_EXISTING,
			       FILE_ATTRIBUTE_SYSTEM,
			       NULL);
    if (handle == INVALID_HANDLE_VALUE) {
      char msg[1024];
      sprintf(msg,"CreateFile(%s) failed", filename);
      perror(msg);
      sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
      OutputDebugString(szBuf);
      return -1;
    }
    // open a partition on the device
    char buffer[10240];
    DWORD numread;
    BOOL rvl;

    // find out the partition info
    rvl = DeviceIoControl(handle,
			  IOCTL_DISK_GET_PARTITION_INFO,
			  NULL, 0,
			  &buffer, sizeof(buffer),
			  &numread,
			  NULL);

    if (!rvl) {
      perror("IOCTL_DISK_GET_PARTITION_INFORMATION failed");
      OutputDebugString("[Bazooka] IOCTL_DISK_GET_PARTITION_INFORMATION failed\n");
      return -1;
    }

    PARTITION_INFORMATION* part_info = (PARTITION_INFORMATION*)buffer;
    part = part_info->PartitionNumber-1;
    win32_fds[fd].partnumber = part_info->PartitionNumber-1;
    win32_fds[fd].part_start = part_info->StartingOffset;
    win32_fds[fd].part_end.QuadPart =
      part_info->StartingOffset.QuadPart +
      part_info->PartitionLength.QuadPart;


    // find out the partition layout, to obtain the drive signature
    rvl = DeviceIoControl(handle,
			  IOCTL_DISK_GET_DRIVE_LAYOUT,
			  NULL, 0,
			  &buffer, sizeof(buffer),
			  &numread,
			  NULL);
    if (!rvl) {
      perror("IOCTL_DISK_GET_DRIVE_LAYOUT failed");
      OutputDebugString("[Bazooka] IOCTL_DISK_GET_DRIVE_LAYOUT 1 failed\n");
      return -1;
    }

    DRIVE_LAYOUT_INFORMATION* drive_layout = (DRIVE_LAYOUT_INFORMATION*)buffer;
    win32_fds[fd].signature = drive_layout->Signature;

    rvl = CloseHandle(handle);
    if (!rvl) {
      perror("CloseHandle failed");
      OutputDebugString("[Bazooka] CloseHandle failed 1 \n");
      return -1;
    }

    for(drive=0;;drive++) {
      sprintf(filename, "\\\\.\\PhysicalDrive%d", drive);
      handle = CreateFile(filename,
			  GENERIC_READ,
			  FILE_SHARE_READ,
			  NULL,
			  OPEN_EXISTING,
			  FILE_ATTRIBUTE_SYSTEM,
			  NULL);
      if (handle == INVALID_HANDLE_VALUE) {
	printf("CreateFile(%s) failed", filename);
	sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
    OutputDebugString(szBuf);
	if (!win32_quiet) {
	  char msg[1024];
	  sprintf(msg,"CreateFile(%s) failed", filename);
   	  sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
      OutputDebugString(szBuf);
	  perror(msg);
	}
	if (drive>16) {
      OutputDebugString("[Bazooka] drive > 16 \n");
	  return -1;
	}
      }
      // find out the partition layout, to obtain the drive signature
      rvl = DeviceIoControl(handle,
			    IOCTL_DISK_GET_DRIVE_LAYOUT,
			    NULL, 0,
			    &buffer, sizeof(buffer),
			    &numread,
			    NULL);
      if (!rvl) {
	perror("IOCTL_DISK_GET_DRIVE_LAYOUT failed");
    OutputDebugString("[Bazooka] IOCTL_DISK_GET_DRIVE_LAYOUT 2 failed \n");
	return -1;
      }

      if (win32_fds[fd].signature == drive_layout->Signature) {
	break;
      }
      rvl = CloseHandle(handle);
      if (!rvl) {
	perror("CloseHandle failed");
    OutputDebugString("[Bazooka] CloseHandle failed 2 \n");

	return -1;
      }
    }

    sprintf(win32_fds[fd].device, "(hd%d,%d)", drive, part);
    win32_fds[fd].drivenumber = drive;
    win32_fds[fd].handle = handle;
    win32_fds[fd].current_pos.QuadPart = win32_fds[fd].part_start.QuadPart;
    OutputDebugString("[Bazooka] win32_open normally returned ! \n");
    return fd;
  }
#endif
#ifdef SUPPORT_XP_NAMES
OutputDebugString("[Bazooka] #ifdef SUPPORT_XP_NAMES\n");
  // on windows XP only!
  OSVERSIONINFO osvi;
  osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx (&osvi);
  if (osvi.dwMajorVersion >= 5 && osvi.dwMinorVersion >= 1) {
    if ((name != NULL)
	&& (toupper(name[0])>='C') && (toupper(name[0]) <='Z')
	&& (name[1] == ':')) {
      typedef struct _DISK_EXTENT {
	DWORD           DiskNumber;
	LARGE_INTEGER   StartingOffset;
	LARGE_INTEGER   ExtentLength;
      } DISK_EXTENT, *PDISK_EXTENT;
      typedef struct _VOLUME_DISK_EXTENTS {
	DWORD       NumberOfDiskExtents;
	DISK_EXTENT Extents[1];
      } VOLUME_DISK_EXTENTS, *PVOLUME_DISK_EXTENTS;
#define IOCTL_VOLUME_BASE   ((DWORD) 'V')
#define IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS    CTL_CODE(IOCTL_VOLUME_BASE, 0, METHOD_BUFFERED, FILE_ANY_ACCESS)

      // open the device "\\.\C:"
      sprintf(filename,"\\\\.\\%s", name);
      HANDLE handle = CreateFile(filename,
				 0,
				 FILE_SHARE_READ,
				 NULL,
				 OPEN_EXISTING,
				 FILE_ATTRIBUTE_SYSTEM,
				 NULL);
      if (handle == INVALID_HANDLE_VALUE) {
	char msg[1024];
	sprintf(msg,"CreateFile(%s) failed", filename);
 	sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
    OutputDebugString(szBuf);
	perror(msg);
	return -1;
      }
      DWORD numbytes;
      // find out which disk this partition is on
      if (DeviceIoControl(
			  handle,
			  // this only works in WinXP or better...
			  IOCTL_VOLUME_GET_VOLUME_DISK_EXTENTS,
			  NULL,
			  0,
			  buffer,
			  sizeof(buffer),
			  &numbytes,
			  NULL
			  )) {
	// don't need the device no more
	CloseHandle(handle);
	VOLUME_DISK_EXTENTS* ext = (VOLUME_DISK_EXTENTS*)buffer;
#ifdef DEBUG
	int i;
	for (i =0; i<ext->NumberOfDiskExtents; i++) {
	  Dprintf("ext[%ld]=(%x,%I64x,%I64x)\n",
		  i,
		  ext->Extents[i].DiskNumber,
		  ext->Extents[i].StartingOffset,
		  ext->Extents[i].ExtentLength);
	}
#endif
	if (ext != NULL && ext->NumberOfDiskExtents == 1) {
	  // open the disk
	  drive = ext->Extents[0].DiskNumber;
	  sprintf(filename, "\\\\.\\PhysicalDrive%lu", drive);
	  HANDLE handle = CreateFile(filename,
				     GENERIC_READ,
				     FILE_SHARE_READ,
				     NULL,
				     OPEN_EXISTING,
				     FILE_ATTRIBUTE_SYSTEM,
				     NULL);
	  if (handle == INVALID_HANDLE_VALUE) {
	    if (!win32_quiet) {
	      char msg[1024];
	      sprintf(msg,"CreateFile(%s) failed", filename);
    	  sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
          OutputDebugString(szBuf);
	      perror(msg);
	    }
	    return -1;
	  }
	  sprintf(win32_fds[fd].device, "(hd%d,?)", drive);
	  // we know all we need to know now
	  win32_fds[fd].handle = handle;
	  win32_fds[fd].drivenumber = drive;
	  win32_fds[fd].partnumber = part;
	  win32_fds[fd].part_start = ext->Extents[0].StartingOffset;
	  win32_fds[fd].part_end = ext->Extents[0].ExtentLength;
	  win32_fds[fd].current_pos.QuadPart = 0;
      OutputDebugString("[Bazooka]normally goto done !\n");
	  goto done;
	}
      }
    }
  }
#endif
  // parse name
  int numparams = 0;
#ifdef SUPPORT_FLOPPY
OutputDebugString("[Bazooka] #ifdef SUPPORT_FLOPPY\n");
  if (numparams == 0) {
    numparams = sscanf(name,"(fd%d)",&drive);
    if (numparams > 0) {
      sprintf(filename, "\\\\.\\%c:", 'A'+drive);

      HANDLE handle = CreateFile(filename,
				 GENERIC_READ | GENERIC_WRITE,
				 FILE_SHARE_READ | FILE_SHARE_WRITE,
				 NULL,
				 OPEN_EXISTING,
				 0,
				 NULL);
      if (handle == INVALID_HANDLE_VALUE) {
	char msg[1024];
	sprintf(msg,"CreateFile(%s) failed", filename);
 	sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
    OutputDebugString(szBuf);
	perror(msg);
	return -1;
      }
      win32_fds[fd].handle = handle;
      win32_fds[fd].part_start.QuadPart = 0;
      win32_fds[fd].part_end.QuadPart = -1;
      win32_fds[fd].current_pos.QuadPart = 0;
      goto done;
    }
  }
#endif
#ifdef SUPPORT_GRUB_NAMES
  if (numparams == 0) {
    numparams = sscanf(name,"(hd%d,%d)",&drive,&part);
  }
#endif
#ifdef SUPPORT_LINUX_NAMES
  if (numparams == 0) {
    char drive_char;
    numparams = sscanf(name,"/dev/hd%c%d",&drive_char,&part);
    drive = toupper(drive_char) - 'A';
    part--;
  }
#endif

  if (numparams >= 1) {
    if (numparams == 2) {
      Dprintf("win32_open(%s) -> drive %d, part %d\n", name, drive, part);
    } else {
      Dprintf("win32_open(%s) -> drive %d\n", name, drive);
    }

    // open the disk
    sprintf(filename, "\\\\.\\PhysicalDrive%d", drive);

    HANDLE handle = CreateFile(filename,
			       GENERIC_READ,
			       FILE_SHARE_READ,
			       NULL,
			       OPEN_EXISTING,
			       FILE_ATTRIBUTE_SYSTEM,
			       NULL);
    if (handle == INVALID_HANDLE_VALUE) {
#ifdef DEBUG
      char msg[1024];
      sprintf(msg,"CreateFile(%s) failed", filename);
      perror(msg);
#endif
      sprintf(szBuf, "[Bazooka] CreateFile(%s) failed\n", filename);
      OutputDebugString(szBuf);

      return -1;
    }


    if (numparams == 1) {
      // open the device itself
      win32_fds[fd].handle = handle;
      win32_fds[fd].part_start.QuadPart = 0;
      win32_fds[fd].part_end.QuadPart = -1;
      win32_fds[fd].current_pos.QuadPart = 0;
      win32_fds[fd].drivenumber = drive;
    } else {
      // open a partition on the device
      char buffer[10240];
      DWORD numread;

      // find out the partition layout
      BOOL rvl = DeviceIoControl(handle,
				 IOCTL_DISK_GET_DRIVE_LAYOUT,
				 NULL, 0,
				 &buffer, sizeof(buffer),
				 &numread,
				 NULL);
      if (!rvl) {
	printf("%s\n", name);
	perror("IOCTL_DISK_GET_DRIVE_LAYOUT failed");
    OutputDebugString("[Bazooka] IOCTL_DISK_GET_DRIVE_LAYOUT 3 failed \n");

	return -1;
      }

      DRIVE_LAYOUT_INFORMATION* drive_layout = (DRIVE_LAYOUT_INFORMATION*)buffer;
      int entry;
      for(entry=0; entry<drive_layout->PartitionCount; entry++) {
	if (drive_layout->PartitionEntry[entry].PartitionNumber-1 == part) {
	  break;
	}
      }

      if (entry > drive_layout->PartitionCount) {
	if (!win32_quiet) {
	  printf("partition %d not found on drive %d\n", part, drive);
	}
	 sprintf(szBuf, "[Bazooka] partition %d not found on drive %d\n", part, drive);
     OutputDebugString(szBuf);

	return -2;
      }

#ifdef DEBUG
      printf("win32: sig %Ix - num %ld - type %x - rec %d\n"
	     ,drive_layout->Signature
	     ,drive_layout->PartitionEntry[entry].PartitionNumber
	     ,drive_layout->PartitionEntry[entry].PartitionType
	     ,drive_layout->PartitionEntry[entry].RecognizedPartition
	     );
#endif
      win32_fds[fd].handle = handle;
      win32_fds[fd].part_start = drive_layout->PartitionEntry[entry].StartingOffset;
      win32_fds[fd].part_end.QuadPart =
	drive_layout->PartitionEntry[0].StartingOffset.QuadPart +
	drive_layout->PartitionEntry[0].PartitionLength.QuadPart;
      win32_fds[fd].current_pos.QuadPart = 0;
      win32_fds[fd].signature = drive_layout->Signature;
      win32_fds[fd].drivenumber = drive;
      win32_fds[fd].partnumber = drive_layout->PartitionEntry[entry].PartitionNumber-1;
    }
  } else {
    Dprintf("win32_open(%s) -> file\n", name);
    HANDLE handle = CreateFile(name, GENERIC_READ | GENERIC_WRITE, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);
    BY_HANDLE_FILE_INFORMATION info;
    BOOL rvl = GetFileInformationByHandle(handle, &info);
    if (!rvl) {
      perror("ioctl failed");
      OutputDebugString("[Bazooka] ioctl failed \n");
      return -1;
    }
    win32_fds[fd].handle = handle;
    win32_fds[fd].part_start.QuadPart = 0;
    win32_fds[fd].part_end.QuadPart = (((s64)info.nFileSizeHigh)<<32)+((s64)info.nFileSizeLow);
    win32_fds[fd].current_pos.QuadPart = 0;
    win32_fds[fd].signature = 0;
    win32_fds[fd].partnumber = 0;
  }
 done:
  Dprintf("win32_open(%s)=%d -> offset 0x%I64x\n", name, fd, win32_fds[fd].part_start);
  sprintf(szBuf, "[Bazooka] win32_open(%s)=%d -> offset 0x%I64x\n", name, fd, win32_fds[fd].part_start);
  OutputDebugString(szBuf);
  
  win32_lseek(fd, 0, SEEK_SET);
  return fd;
}

s64 win32_lseek(const int fd, s64 pos, int mode) {
  Dprintf("win32_seek(%I64u=0x%I64x,%d)\n", pos, pos, mode);
  LARGE_INTEGER offset;

  int disp;
  if (mode==SEEK_SET) {
    disp=FILE_BEGIN;
	offset.QuadPart = win32_fds[fd].part_start.QuadPart + pos;
  } else if (mode==SEEK_CUR) {
	disp=FILE_CURRENT;
	offset.QuadPart = pos;
  } else if (mode==SEEK_END) {
	// end of partition != end of disk
	disp=FILE_BEGIN;
	offset.QuadPart = win32_fds[fd].part_end.QuadPart + pos;
  } else {
	printf("win32_seek() wrong mode %d\n", mode);
	return -1;
  }

  BOOL rvl = SetFilePointerEx(win32_fds[fd].handle, offset,
							  &win32_fds[fd].current_pos, disp);
  Dprintf("win32_seek SetFilePointer(%I64u)=%I64u\n", offset.QuadPart, win32_fds[fd].current_pos.QuadPart);


  if (!rvl) {
	char msg[1024];
	sprintf(msg,"SetFilePointer(%I64u) failed", offset.QuadPart);
	perror(msg);
	return -1;
  }
  return pos;
}

s64 win32_read(const int fd, const void *b, s64 count) {
  LARGE_INTEGER base, offset, numtoread;
  offset.QuadPart = win32_fds[fd].current_pos.QuadPart & 0x1FF;
  base.QuadPart = win32_fds[fd].current_pos.QuadPart - offset.QuadPart;
  numtoread.QuadPart = ((count+offset.QuadPart-1) | 0x1FF) + 1;

  Dprintf("win32_read(fd=%d,b=%p,count=0x%I64x)->(%I64x+%I64x:%I64x)\n",
		  fd, b, count, base, offset,numtoread);

  DWORD numread = 0;
  BOOL rvl = -1;

#ifndef FORCE_ALIGNED_READ
  if (
	  ((((long)b) & ((long)0x1FF)) == 0)
		&& ((count & ((s64)0x1FF))==0)
	  && ((win32_fds[fd].current_pos.QuadPart & 0x1FF) == 0)
	  ) {
	Dprintf("normal read\n");
	rvl = ReadFile(win32_fds[fd].handle, (LPVOID)b,
				   count, &numread, (LPOVERLAPPED)NULL);
  } else {
	Dprintf("aligned read\n");
#endif

	BYTE* alignedbuffer = (BYTE*)VirtualAlloc(NULL, count, MEM_COMMIT, PAGE_READWRITE);
	Dprintf("win32_read set SetFilePointerEx(0x%I64x)\n", base.QuadPart);
	rvl = SetFilePointerEx(win32_fds[fd].handle, base, &win32_fds[fd].current_pos, FILE_BEGIN);

	if (!rvl) {
	  printf("SetFilePointerEx failed\n");
	}
	rvl = ReadFile(win32_fds[fd].handle, (LPVOID)alignedbuffer,
				   numtoread.QuadPart, &numread, (LPOVERLAPPED)NULL);
	if (!rvl) {
	  printf("ReadFile failed\n");
	}

#ifndef FORCE_ALIGNED_READ
	LARGE_INTEGER new_pos;
	new_pos.QuadPart = win32_fds[fd].current_pos.QuadPart + count;
	Dprintf("reset SetFilePointerEx(0x%I64x)\n", new_pos.QuadPart);
	rvl = SetFilePointerEx(win32_fds[fd].handle, new_pos,
						   &win32_fds[fd].current_pos, FILE_BEGIN);

	if (!rvl) {
	  printf("reset SetFilePointerEx failed\n");
	}
#endif

	memcpy((void*)b,alignedbuffer+offset.QuadPart,count);
	VirtualFree(alignedbuffer, 0, MEM_RELEASE);
#ifndef FORCE_ALIGNED_READ
  }
#endif
  if (!rvl) {
	perror("ReadFile failed");
	return -1;
  }

#if 0
{
int i;
for(i=0; i<count; i++) {
  unsigned char c = ((unsigned char*)b)[i];
  fprintf(stderr,"%02x ",c);
  if (i%16==15) { printf("\n"); }
}
fprintf(stderr,"\n");
for(i=0; i<count; i++) {
  unsigned char c = ((unsigned char*)b)[i];
  fprintf(stderr,"%c",isprint(c)?c:'.');
  if (i%16==15) { printf("\n"); }
}
fprintf(stderr,"\n");
}
#endif

  if (numread > count) {
	return count;
  } else {
	return numread;
  }
}

/*
 * Beware: this can severely damage the FileSystem!
 * I have only tested this on floppy disks with no filesystem
 */
s64 win32_write(const int fd, const void *b, s64 count) {
  Dprintf("win32_write(%d,%p,%I64d)\n", fd, b, count);
  BOOL rvl;
  DWORD numwritten = 0;
  rvl = WriteFile(win32_fds[fd].handle, (LPVOID)b,
				  count, &numwritten, (LPOVERLAPPED)NULL);
  if (!rvl) {
	perror("WriteFile() failed");
  }
  return rvl;
}

int win32_close(int fd) {

  Dprintf("win32_close(%d)\n", fd);

  BOOL rvl = CloseHandle(win32_fds[fd].handle);
  win32_fds[fd].handle = 0;

  if (!rvl) {
	perror("CloseHandle failed");
	return -1;
  }

  return 0;
}

s64 win32_bias(int fd) {
  return win32_fds[fd].part_start.QuadPart;
}

long win32_signature(int fd) {
  return win32_fds[fd].signature;
}

int win32_drivenumber(int fd) {
  return win32_fds[fd].drivenumber;
}

int win32_partnumber(int fd) {
  return win32_fds[fd].partnumber;
}

char* win32_device(int fd) {
  return win32_fds[fd].device;
}

s64 win32_filepos(int fd) {
  return win32_fds[fd].current_pos.QuadPart - win32_fds[fd].part_start.QuadPart;
}


