#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "stage1.h"
#include "win32.h"

#define SECTOR_SIZE 512
#define STAGE2_ADDRESS 0x8000
#define STAGE2_VER_STR_OFFS 0x12

typedef unsigned char BYTE;
typedef unsigned short WORD;
typedef unsigned long LONG;

BYTE stage1_buffer[SECTOR_SIZE];
BYTE stage2_buffer[SECTOR_SIZE];
BYTE stage2_buffer2[SECTOR_SIZE];
LONG stage2_address = STAGE2_ADDRESS;
int mkbootdisk=0;
char* bootdevice="grub.img";//"(fd0)";

struct {
  BYTE* major;
  BYTE* minor;
  BYTE* boot_drive;
  BYTE* force_lba;
  WORD* stage2_address;
  LONG* stage2_sector;
  WORD* stage2_segment;
} bootparams = {
  stage1_buffer + STAGE1_VER_MAJ_OFFS,
  stage1_buffer + STAGE1_VER_MAJ_OFFS+1,
  stage1_buffer + STAGE1_BOOT_DRIVE,
  stage1_buffer + STAGE1_FORCE_LBA,
  (WORD*)(stage1_buffer + STAGE1_STAGE2_ADDRESS),
  (LONG*)(stage1_buffer + STAGE1_STAGE2_SECTOR),
  (WORD*)(stage1_buffer + STAGE1_STAGE2_SEGMENT),
};

int 
stage1(char* filename, LONG sector, BYTE lba, BYTE drive) 
{
  int rvl = 0;

  printf("stage1(%ld)\n", sector);

  // open stage 1 image
  FILE* fp=fopen(filename,"r+b");
  if (fp == NULL) {
	char* p;
	for(p=filename; *p != '\0'; p++) {
	  if (*p == '/') { *p = '\\'; }
	}

	printf("Cannot open %s: ", filename);
	perror("");
	rvl--;
	return rvl;
  }
  // read stage 1
  fseek(fp,0,SEEK_SET);
  fread(stage1_buffer,SECTOR_SIZE,1,fp);

  // insert the parameters from the command line
  if (sector != 0) {
	*bootparams.stage2_sector = sector;
	*bootparams.stage2_address = stage2_address;
	*bootparams.force_lba = lba;
	*bootparams.boot_drive = drive;

	// write the modified image back to file
	if (!mkbootdisk){
	  fseek(fp,0,SEEK_SET);
	  if(fwrite(stage1_buffer,SECTOR_SIZE,1,fp) < 0) {
		rvl--;
	  }
	}
  }

  int r = fclose(fp);
  if (r != 0) {
	perror("fclose failed\n");
  }

  // print results
  //dump(stage1_buffer,32,16);
  printf(" version=%d.%d\n", *bootparams.major, *bootparams.minor);
  printf(" boot_drive=0x%x\n", *bootparams.boot_drive);
  printf(" force_lba=%d\n", *bootparams.force_lba);
  printf(" stage2_address=0x%x\n", *bootparams.stage2_address);
  printf(" stage2_sector=%lu\n", *bootparams.stage2_sector);
  printf(" stage2_segment=0x%x\n", *bootparams.stage2_segment);

  // write stage1 to boot disk
  if (mkbootdisk) {
	printf("copying stage1 to boot sector\n");
	int fd = win32_open(bootdevice,1);
	win32_lseek(fd, 0, SEEK_SET);
	win32_write(fd,stage1_buffer,SECTOR_SIZE);
	win32_close(fd);
  }

  return rvl;
}

#define GET(x) {ptr-=sizeof(x);memcpy(&x,ptr,sizeof(x));}
#define PUT(x) {ptr-=sizeof(x);memcpy(ptr,&x,sizeof(x));}
#define BACK(x) {ptr+=sizeof(x);}

int 
stage2(char* filename, char* blocklist, char* menu) {
  int rvl = 0;

  printf("stage2(%s)\n", blocklist);

  // open stage2 image
  FILE* fp=fopen(filename,"r+b");
  if (fp == NULL) {
	char* p;
	for(p=filename; *p != '\0'; p++) {
	  if (*p == '/') { *p = '\\'; }
	}
	printf("Cannot open %s: ", filename);
	perror("");
	rvl--;
	return rvl;
  }

  // read first sector of stage2
  fseek(fp, 0, SEEK_SET);
  fread(stage2_buffer,SECTOR_SIZE,1,fp);

  BYTE* ptr;
  WORD count;
  LONG offset;
  WORD addr;

  // insert blocklist into stage2
  if (blocklist) {
	ptr = &(stage2_buffer[SECTOR_SIZE]);

	addr = (stage2_address + SECTOR_SIZE) >> 4;

	if (*blocklist=='(') {
	  blocklist = strchr(blocklist,')');
	  if (blocklist) {
		blocklist++;
	  }
	}
	int first = 1;
	while(blocklist && *blocklist) {
	  if (sscanf(blocklist, "%ld+%hd", &offset, &count)==2) {
		if (first) {
		  first = 0;
		  // first sector was already loaded
		  offset++;
		  count--;
		}
		//printf(" offset=%d, count=%d, addr=0x%x\n", offset, count, addr);

		if (((*(WORD*)(ptr-4)) & 0x8000) != 0) {
		  printf("IMAGE TOO LARGE! p=0x%x\n",*(WORD*)(ptr-4));
		  break;
		}

		if (count != 0) {
		  PUT(addr);
		  PUT(count);
		  PUT(offset);
		}
		addr += (count*SECTOR_SIZE)>>4;

	  } else {
		break;
	  }
	  blocklist = strchr(blocklist,',');
	  if (blocklist) {
		blocklist++;
	  }
	}

	  count = 0; offset = 0; addr = 0;
	  PUT(addr);
	  PUT(count);
	  PUT(offset);
	
	fseek(fp, 0, SEEK_SET);
	if (!mkbootdisk) {
	  // write updated block to stage2 image
	  if (! fwrite(stage2_buffer,SECTOR_SIZE,1,fp)) {
		rvl--;
	  }
	}
  }

  // read second block of stage2
  fseek(fp, SECTOR_SIZE, SEEK_SET);
  fread(stage2_buffer2,SECTOR_SIZE,1,fp);
  ptr = &(stage2_buffer2[STAGE2_VER_STR_OFFS]);
  // find end of version string
  while (*ptr) ptr++;
  ptr++;

  // insert path to menu after version string
  if (menu) {
	memcpy(ptr,menu,strlen(menu));
	ptr+=strlen(menu);
  }
  *ptr='\0';

  // write updated block to stage2 image
  if (!mkbootdisk) {
	fseek(fp, SECTOR_SIZE, SEEK_SET);
	fwrite(stage2_buffer2,SECTOR_SIZE,1,fp);
  }

  // print results
  ptr = &(stage2_buffer[SECTOR_SIZE]);
  do {
	GET(addr);
	GET(count);
	GET(offset);
	if ((count & 0x8000) != 0) {
	  break;
	}
	if (addr == 0) {
	  break;
	}
	printf(" %d sectors to load from sector %ld at 0x%x\n", count, offset, addr);
  } while (addr != 0);

  if (menu) {
	printf(" menu '%s'\n", menu);
  }
  //dump(stage2_buffer, 32, 16);

  // copy stage2 to bootdisk
  if (mkbootdisk) {
	printf("copying stage2 to boot disk\n");
	int fd = win32_open(bootdevice,1);
	win32_lseek(fd, SECTOR_SIZE, SEEK_SET);
	// write first sector
	rvl = win32_write(fd, stage2_buffer, SECTOR_SIZE);
	if (!rvl) { goto done; }
	// write second sector
	rvl = win32_write(fd, stage2_buffer2, SECTOR_SIZE);
	if (!rvl) { goto done; }

	// copy the rest of the stage2 image file
	int numread;
	char buffer[SECTOR_SIZE];
	fseek(fp, SECTOR_SIZE*2, SEEK_SET);
	for(;;) {
	  numread = fread(buffer,SECTOR_SIZE,1,fp);
	  if (numread <= 0) {
		break;
	  }
	  if (!win32_write(fd, buffer, numread*SECTOR_SIZE)) {
		break;
	  }
	}

 done:
	win32_close(fd);
  }
  fclose(fp);
  return rvl;
}

