/* -*-Asm-*- */
/*
 *  GRUB  --  GRand Unified Bootloader
 *  Copyright (C) 1999,2000,2001   Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *  Copyright (C) 2004	CyberLink Corp.
 *
 *  Add Windows XP-Compatible Loader
 *
 */

#include <stage1.h>
	
/*
 *  defines for the code go here
 */

	/* Absolute addresses
	   This makes the assembler generate the address without support
	   from the linker. (ELF can't relocate 16-bit addresses!) */
#define ABS(x) (x-_start+0x7e00)

	/* Print message string */
#define MSG(x)	movw $ABS(x), %si

	/* XXX:	binutils-2.9.1.0.x doesn't produce a short opcode for this. */
#define	MOV_MEM_TO_AL(x)	.byte 0xa0;  .word x
	
	.file	"stage1.S"

	.text

	/* Tell GAS to generate 16-bit instructions so that this code works
	   in real mode. */
	.code16

.globl _start; _start:
	/*
	 * _start is loaded at 0x7c00 and is jumped to with CS:IP 0:0x7c00
	 */

	/*
	 * Beginning of the sector is compatible with the FAT/HPFS BIOS
	 * parameter block.
	 */

	jmp	after_BPB
	nop	/* do I care about this ??? */

	/*
	 * This space is for the BIOS parameter block!!!!  Don't change
	 * the first jump, nor start the code anywhere but right after
	 * this area.
	 */

	. = _start + 3

	/* scratch space */

	mode 			= .			// mode:	  -1
	disk_address_packet 	= mode + 1		// [field]    [offset]
	sectors 		= disk_address_packet	// sectors:	  0
	heads 			= sectors + 4		// heads:	  4
	cylinders 		= heads + 4		// cylinders:	  8
	sector_start 		= cylinders + 2		// sector_start:  10 
	head_start 		= sector_start + 1	// head_start:	  11
	cylinder_start 		= head_start + 1	// cylinder_start:12

	// . = mode + 17

after_BPB:
	/* general setup */

	cli				/* we're not safe here! */
	xorw	%ax, %ax
	movw	%ax, %ss
	movw	$STAGE1_STACKSEG, %sp	/* set up the REAL stack */
	sti				/* we're safe again */

	/* copy this sector to 0000:7E00 */

	movw	%ax, %ds
	movw	%ax, %es
	movw	$0x7c00, %si
	movw	$0x7e00, %di
	movw    $0x100, %cx
	rep	movsw

	/*
	 * ljmp to the next instruction because some bogus BIOSes
	 * jump to 07C0:0000 instead of 0000:7C00.
	 */
	ljmp	$0, $ABS(real_start)

geometry_error_string:	.string "Geom"
read_error_string:	.string "Read"
hd_probe_error_string:	.string "Hard Disk"
general_error_string:	.string " Error"

/*
debug:	// destroys %ax
	pop	%ax
	push	%ax
	pusha
	pushf
	call	print_2digit
	popf
	popa
	ret

print_2digit:
	pushw	%ax
	shrb	$4, %al
	call	print_digit
	popw	%ax
	andb	$0x0f, %al
print_digit:
// would love to have this, but no space...
//	cmpb	$0x0a, %al
//	sbbb	$0x69, %al
//	das
	addb	$0x30, %al
	movw	$0x0001, %bx
	movb	$0xe, %ah
	int	$0x10
	ret
*/
	. = _start + STAGE1_BPBEND

	/*
	 * End of BIOS parameter block.
	 */

stage1_version:	
	.byte	COMPAT_VERSION_MAJOR, COMPAT_VERSION_MINOR
boot_drive:	
	.byte	GRUB_NO_DRIVE	/* the disk to load stage2 from */
				/* 0xff means use the boot drive */
force_lba:
	.byte	0
stage2_address:
	.word	0x8000
stage2_sector:
	.long	1
stage2_segment:
	.word	0x800

/*
	Read a sector using LBA

	Input:	drive in DL
		2 bytes memory segment address in stage2_segment
		4 bytes disk sector address in %ebp
	Output: 
	Destroy: %ax, %bx, %cx, %dx, %si, %di
	Set carry on error
*/
read_lba:
	pushw	%dx

	/* check if LBA is supported */
	movb	$0x41, %ah
	movw	$0x55aa, %bx
	int	$0x13

	/* 
	 *  %dl may have been clobbered by INT 13, AH=41H.
	 *  This happens, for example, with AST BIOS 1.04.
	 */
	popw	%dx

	jc	read_lba_fail
	cmpw	$0xaa55, %bx
	jne	read_lba_fail

	shr	$1, %cl
	jnc	read_lba_fail

	/* set %si to the disk address packet */
	movw	$ABS(disk_address_packet), %si

	/* clear packet */
	movw	%si, %di
	movw	$16, %cx
	movb	$0, %al
	rep 	stosb

	/* set the mode to non-zero */
	/* the size and the reserved byte */
	/* the blocks */
	movl	$0x01001001, -1(%si)	// mode=0x01, size=0x10, blocks=0x01
	
	/* the absolute address (low 32 bits) */
	movl	%ebp, 8(%si)

	/* the segment of buffer address */
	movw	ABS(stage2_segment), %ax	
	movw	%ax, 6(%si)

/*
 * BIOS call "INT 0x13 Function 0x42" to read sectors from disk into memory
 *	Call with	%ah = 0x42
 *			%dl = drive number
 *			%ds:%si = segment:offset of disk address packet
 *	Return:
 *			%al = 0x0 on success; err code on failure
 */

	movb	$0x42, %ah
	int	$0x13
	ret

read_lba_fail:
	stc
	ret


real_start:
	/* check if ctrl or alt is pressed */
	mov	$2, %ah
	int	$0x16
	andb	$0xf, %al
	jz	failsafe

	/* Try LBA first */
	movb	ABS(boot_drive), %dl
	/* save drive reference first thing! */
	pushw	%dx		
	movl	ABS(stage2_sector), %ebp
	call	read_lba
	jnc 	run_stage2

	/* Try CHS then */

	/*
	 *  Determine the hard disk geometry from the BIOS!
	 *  We do this first, so that LS-120 IDE floppies work correctly.
	 */

	call	get_disk_chs
	movb	$0, -1(%si)	/* set the mode to zero */
	jc	hd_probe_error

setup_sectors:
	/* load logical sector start (bottom half) */
	movl	ABS(stage2_sector), %eax

	/* zero %edx */
	xorl	%edx, %edx

	/* divide by number of sectors */
	divl	(%si)

	/* save sector start */
	movb	%dl, 10(%si)

	xorl	%edx, %edx	/* zero %edx */
	divl	4(%si)		/* divide by number of heads */

	/* save head start */
	movb	%dl, 11(%si)

	/* save cylinder start */
	movw	%ax, 12(%si)

	/* do we need too many cylinders? */
	cmpw	8(%si), %ax
	jge	geometry_error

/*
 *  This is the loop for taking care of BIOS geometry translation (ugh!)
 */

	/* get high bits of cylinder */
	movb	13(%si), %dl

	shlb	$6, %dl		/* shift left by 6 bits */
	movb	10(%si), %cl	/* get sector */

	incb	%cl		/* normalize sector (sectors go
					from 1-N, not 0-(N-1) ) */
	orb	%dl, %cl	/* composite together */
	movb	12(%si), %ch	/* sector+hcyl in cl, cylinder in ch */

	/* restore %dx */
	popw	%dx
	
	/* head number */
	movb	11(%si), %dh

	call	read_chs
	jc	read_error

	/* boot stage2 */

run_stage2:
	jmp	*ABS(stage2_address)

/* END OF MAIN LOOP */

/*
 * message: write the string pointed to by %si
 *
 *   WARNING: trashes %si, %ax, and %bx
 */

	/*
	 * Use BIOS "int 10H Function 0Eh" to write character in teletype mode
	 *	%ah = 0xe	%al = character
	 *	%bh = page	%bl = foreground color (graphics modes)
	 */
1:
	movw	$0x0001, %bx
	movb	$0xe, %ah
	int	$0x10		/* display a byte */
message:
	lodsb
	cmpb	$0, %al
	jne	1b	/* if not end of string, jmp to display */
	ret

/*
 * BIOS Geometry translation error (past the end of the disk geometry!).
 */
geometry_error:
	MSG(geometry_error_string)
	jmp general_error

/*
 * Disk probe failure.
 */
hd_probe_error:
	MSG(hd_probe_error_string)
	jmp	general_error

/*
 * Read error on the disk.
 */
read_error:
	MSG(read_error_string)

general_error:
	call message
	MSG(general_error_string)
	call message

/* go here when you need to stop the machine hard after an error condition */
stop:	jmp	stop

/*
	Get the drive parameters.

	Input:	DL = Drive
	Output: %si = $ABS(sectors)
		Cylinders -- 8(%si), 2 bytes
		Heads	  -- 4(%si), 4 bytes
		Sectors	  -- 0(%si), 4 bytes
	Destroy: %eax, %dh, %cx
	Set carry on error
*/
get_disk_chs:

	movw	$ABS(sectors), %si

	movb	$8, %ah
	int	$0x13
	jc	1f

	/*
	  CH = low eight bits of maximum cylinder number
	  CL = maximum sector number (bits 5-0)
	  high two bits of maximum cylinder number (bits 7-6)
	  DH = maximum head number
	*/

	/* save number of heads */
	movzx   %dh, %eax
	incw	%ax
	movl	%eax, 4(%si)

	movb	%cl, %ah
	shrb	$6, %ah
	movb	%ch, %al

	/* save number of cylinders */
	incw	%ax
	movw	%ax, 8(%si)

	movzx	%cl, %ax
	andb	$0x3f, %al	// clears carry

	/* save number of sectors */
	movl	%eax, (%si)
1:
	ret

/*
 * BIOS call "INT 0x13 Function 0x2" to read sectors from disk into memory
 *	Call with	%ah = 0x2
 *			%al = number of sectors
 *			%ch = cylinder
 *			%cl = sector (bits 6-7 are high bits of "cylinder")
 *			%dh = head
 *			%dl = drive (0x80 for hard disk, 0x0 for floppy disk)
 *			%es:%bx = segment:offset of buffer
 *	Return:
 *			%al = 0x0 on success; err code on failure
 */

/*
	%cx, %dx: parameter for int 0x13/ah=2
	Set carry on error
*/
read_chs:
	pushw	ABS(stage2_segment)
	popw	%es		/* load %es segment with disk buffer */

	xorw	%bx, %bx	/* %bx = 0, put it at 0 in the segment */
	movw	$0x0201, %ax	/* function 2 */
	int	$0x13
	ret

/*
 *  This is a fail-safe loader
 */

failsafe:
	movw	$0x7c0, ABS(stage2_segment)

	/* Try to find an active partition */

	movw	$ABS(part_start), %di
	movw	$4, %cx
next:
	cmpb	%ch, (%di)
	pusha
	jnl	continue
	call	read_one_sector
	jnc	run_it
	cmpb	$0xb, 4(%di)
	jz	adjust
	cmpb	$0xc, 4(%di)
	jnz	continue
adjust:
	addb	$6, 2(%di)
	addw	$6, 8(%di)
	adcw	$0, 10(%di)
	call	read_one_sector
run_it:
	jnc	_start-0x200	// back to 0x7c00

continue:
	popa
	addw	$0x10, %di
	loop	next
	int 	$0x18

read_one_sector:
	/* %di points to a partition table entry */

	movb	(%di), %dl
	call	get_disk_chs	
	jc	use_chs

	/* check if the lba is in disk chs range */

	movw	(%si), %ax 
	mulw	4(%si)
	mulw	8(%si)

	cmpw	%dx, 10(%di)
	ja	use_lba
	jb	use_chs
	cmpw	%ax, 8(%di)
	jnb	use_lba

use_chs:
	movw	(%di), %dx	/* load drive and CHS address */
	movw	2(%di), %cx
	jmp	read_chs

use_lba:
	movb	(%di), %dl	/* load drive and LBA address */
	movl	8(%di), %ebp
	jmp	read_lba
/*
geometry_error_string:	.string "G"
read_error_string:	.string "R"
hd_probe_error_string:	.string "H"
general_error_string:	.string "E"
*/
	/*
	 *  Windows NT breaks compatibility by embedding a magic
	 *  number here.
	 */

	. = _start + STAGE1_WINDOWS_NT_MAGIC
nt_magic:	
	.long 0
	.word 0

part_start:
	. = _start + STAGE1_PARTEND

/* the last 2 bytes in the sector 0 contain the signature */
	.word	STAGE1_SIGNATURE
