#!/usr/bin/perl
# --
# bin/otrs.PostMasterMailbox.pl - the global eMail handle for email2db
# Copyright (C) 2001-2016 OTRS AG, http://otrs.com/
# --
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU AFFERO General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# or see http://www.gnu.org/licenses/agpl.txt.
# --

use strict;
use warnings;

use File::Basename;
use FindBin qw($RealBin);
use lib dirname($RealBin);
use lib dirname($RealBin) . '/Kernel/cpan-lib';
use lib dirname($RealBin) . '/Custom';

use Getopt::Std;

use Kernel::System::MailAccount;
use Kernel::System::ObjectManager;

# get options
my %Opts;
getopt( 'upsdtb', \%Opts );
if ( $Opts{h} ) {
    print "PostMasterMailbox.pl - Fetch mail accounts for OTRS\n";
    print "Copyright (C) 2001-2016 OTRS AG, http://otrs.com/\n";
    print "usage: PostMasterMailbox.pl -t <TYPE> (POP3|POP3S|IMAP|IMAPS) -s <SERVER> -u <USER> ";
    print "-p <PASSWORD> [-d 1-2] [-b <BACKGROUND_INTERVAL_IN_MIN>] [-f force]\n";
    exit 1;
}

# set debug
if ( !$Opts{d} ) {
    $Opts{d} = 0;
}

# check -b option
if ( $Opts{b} && $Opts{b} !~ /^\d+$/ ) {
    print STDERR "ERROR: Need -b <BACKGROUND_INTERVAL_IN_MIN>, e. g. -b 5 for fetching emails ";
    print STDERR "every 5 minutes.\n";
    exit 1;
}

# create object manager
local $Kernel::OM = Kernel::System::ObjectManager->new(
    'Kernel::System::Log' => {
        LogPrefix => 'OTRS-otrs.PostMasterMailbox.pl',
    },
);

# create pid lock
if (
    !$Opts{f}
    && !$Kernel::OM->Get('Kernel::System::PID')->PIDCreate( Name => 'PostMasterMailbox' )
    )
{
    print "NOTICE: PostMasterMailbox.pl is already running (use '-f 1' if you want to start it ";
    print "forced)!\n";
    exit 1;
}
elsif (
    $Opts{f}
    && !$Kernel::OM->Get('Kernel::System::PID')->PIDCreate( Name => 'PostMasterMailbox' )
    )
{
    print "NOTICE: PostMasterMailbox.pl is already running but is starting again!\n";
}

# fetch mails -b is not used
if ( !$Opts{b} ) {
    Fetch();
}

# while to run several times if -b is used
else {
    while (1) {

        # set new PID
        $Kernel::OM->Get('Kernel::System::PID')->PIDCreate(
            Name  => 'PostMasterMailbox',
            Force => 1,
        );

        # fetch mails
        Fetch();

        # sleep till next interval
        print "NOTICE: Waiting for next interval ($Opts{b} min)...\n";
        sleep 60 * $Opts{b};
    }
}

# delete pid lock
$Kernel::OM->Get('Kernel::System::PID')->PIDDelete( Name => 'PostMasterMailbox' );
exit(0);

sub Fetch {

    my $MailAccount = Kernel::System::MailAccount->new();

    # debug info

    if ( $Opts{d} > 1 ) {
        $Kernel::OM->Get('Kernel::System::Log')->Log(
            Priority => 'debug',
            Message  => 'Global OTRS email handle (PostMasterMailbox.pl) started...',
        );
    }

    if ( $Opts{s} || $Opts{u} || $Opts{p} || $Opts{t} ) {
        if ( !$Opts{t} ) {

            # delete pid lock
            $Kernel::OM->Get('Kernel::System::PID')->PIDDelete( Name => 'PostMasterMailbox' );
            print STDERR "ERROR: Need -t <TYPE> (POP3|IMAP)\n";
            exit 1;
        }
        if ( !$Opts{s} ) {

            # delete pid lock
            $Kernel::OM->Get('Kernel::System::PID')->PIDDelete( Name => 'PostMasterMailbox' );
            print STDERR "ERROR: Need -s <SERVER>\n";
            exit 1;
        }
        if ( !$Opts{u} ) {

            # delete pid lock
            $Kernel::OM->Get('Kernel::System::PID')->PIDDelete( Name => 'PostMasterMailbox' );
            print STDERR "ERROR: Need -u <USER>\n";
            exit 1;
        }
        if ( !$Opts{p} ) {

            # delete pid lock
            $Kernel::OM->Get('Kernel::System::PID')->PIDDelete( Name => 'PostMasterMailbox' );
            print STDERR "ERROR: Need -p <PASSWORD>\n";
            exit 1;
        }
        $MailAccount->MailAccountFetch(
            Login         => $Opts{u},
            Password      => $Opts{p},
            Host          => $Opts{s},
            Type          => $Opts{t},
            Trusted       => 0,
            DispatchingBy => '',
            QueueID       => 0,
            Debug         => $Opts{d},
            CMD           => 1,
            UserID        => 1,
        );
    }
    else {
        my %List = $MailAccount->MailAccountList( Valid => 1 );
        for my $Key ( sort keys %List ) {
            my %Data = $MailAccount->MailAccountGet( ID => $Key );
            $MailAccount->MailAccountFetch(
                %Data,
                Debug  => $Opts{d},
                CMD    => 1,
                UserID => 1,
            );
        }
    }

    # debug info
    if ( $Opts{d} > 1 ) {
        $Kernel::OM->Get('Kernel::System::Log')->Log(
            Priority => 'debug',
            Message  => 'Global OTRS email handle (PostMasterMailbox.pl) stopped.',
        );
    }
}
