/****************************************************************************
**
** Implementation of SQLite driver classes
**
** Created : 001103
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the sql module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition licenses may use this
** file in accordance with the Qt Commercial License Agreement provided
** with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "qsql_sqlite.h"

#include "../../qsqlrecord.h"
#include <qlist.h>

#include <sqlite.h>

#define QSQLite_DRIVER_NAME "QSQLITE2"

class QSQLiteDriverPrivate
{
public:
    QSQLiteDriverPrivate() : access(0) {}
    sqlite *access;
};

class QSQLiteResultPrivate
{
public:
    QSQLiteResultPrivate() : access(0) {}
    sqlite *access;

    // and we have too keep our own struct for the data (sqlite works via 
    // callback.

    QList<QSqlRecord> results;
    QStringList columnNames;
};

QSQLiteResult::QSQLiteResult( const QSQLiteDriver* db )
: QSqlResult( db )
{
    d = new QSQLiteResultPrivate();
    d->access = db->d->access;
    d->results.setAutoDelete(true);
}

QSQLiteResult::~QSQLiteResult()
{
    cleanup();
    delete d;
}

void QSQLiteResult::cleanup()
{
    d->results.clear();
    d->columnNames.clear();
    setAt( -1 );
    setActive( FALSE );
}

bool QSQLiteResult::fetch( int i )
{
    if (d->results.at(i) != 0) {
	setAt( i );
	return TRUE;
    }
    return FALSE;
}

bool QSQLiteResult::fetchNext()
{
    if (d->results.next() != 0) {
	setAt( at() + 1 );
	return TRUE;
    }
    return FALSE;
}

bool QSQLiteResult::fetchPrev()
{
    if (d->results.prev() != 0) {
	setAt( at() + 1 );
	return TRUE;
    }
    return FALSE;
}

bool QSQLiteResult::fetchLast()
{
    if (d->results.last() != 0) {
	setAt( d->results.count() - 1 );
	return TRUE;
    }
    return FALSE;
}

bool QSQLiteResult::fetchFirst()
{
    if (d->results.first() != 0) {
	setAt( 0 );
	return TRUE;
    }
    return FALSE;
}

Variant QSQLiteResult::data( int field )
{
    if ( !isSelect() )
	return Variant();

    QSqlRecord *rec = d->results.current();
    if (!rec)
	return Variant();

    return rec->value(field);
}

bool QSQLiteResult::isNull( int field )
{
    QSqlRecord *rec = d->results.current();
    if (!rec)
	return TRUE;
    return rec->isNull(field);
}

int QSQLiteResult::populate(void *set, int colNum, char **fvals, char **cnames)
{
    QSQLiteResult *r = (QSQLiteResult *)set;
    return r->populate(colNum, fvals, cnames);
}

int QSQLiteResult::populate(int colNum, char **fvals, char **cnames)
{
    int i;
    setSelect( !( colNum == 0) );
    if (colNum == 0)
	return 0;

    // is this the first?
    if (d->columnNames.count() != (uint)colNum) {
	// must be first call.
	for (i = 0; i < colNum; i++) {
	    d->columnNames.append(cnames[i]);
	}
    }

    if (fvals) {
	QSqlRecord *r = new QSqlRecord();
	// not just a heading, but the data as well.
	for (i = 0; i < colNum; i++) {
	    r->append(QSqlField(cnames[i], Variant::String));
	    if (fvals[i]) {
		r->setValue(i, Variant(fvals[i]));
	    }
	}
	d->results.append(r);
    }
    return 0;
}

/*
   Execute \a query.
*/
bool QSQLiteResult::reset ( const QString& query )
{
    if ( !driver() )
	return FALSE;
    if ( !driver()-> isOpen() || driver()->isOpenError() )
	return FALSE;

    cleanup();

    // Um, ok.  callback based so.... pass private static function for this.
    sqlite_exec(d->access, query.latin1(), populate, this, 0);
    setActive( TRUE );
    return TRUE;
}

int QSQLiteResult::size()
{
    return d->results.count();
}

int QSQLiteResult::numRowsAffected()
{
    return sqlite_changes(d->access);
}

/////////////////////////////////////////////////////////

QSQLiteDriver::QSQLiteDriver( QObject * parent, const char * name )
: QSqlDriver(parent, name ? name : QSQLite_DRIVER_NAME)
{
    d = new QSQLiteDriverPrivate();
    d->access = 0;
}

QSQLiteDriver::~QSQLiteDriver()
{
    delete d;
}

bool QSQLiteDriver::hasFeature( DriverFeature f ) const
{
    switch ( f ) {
    case Transactions:
	return TRUE;
    case QuerySize:
	return TRUE;
    case BLOB:
	return TRUE;
    default:
	return FALSE;
    }
}

/*
   SQLite dbs have no user name, passwords, hosts or ports.
   just file names.
*/
bool QSQLiteDriver::open( const QString & db, const QString &, const QString &, const QString &, int)
{
    if ( isOpen() )
	close();

    d->access = sqlite_open(db.latin1(), 0, 0);

    if (d->access) {
	setOpen( TRUE );
	return TRUE;
    }
    setOpenError( TRUE );
    return FALSE;
}

void QSQLiteDriver::close()
{
    if ( isOpen() ) {
	sqlite_close( d->access );
	d->access = 0;
	setOpen( FALSE );
	setOpenError( FALSE );
    }
}

QSqlQuery QSQLiteDriver::createQuery() const
{
    return QSqlQuery( new QSQLiteResult( this ) );
}
