/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Id: klDisplayList.h,v 1.4 2004/01/14 16:50:18 drgoldie Exp $
 *
\**************************************************************************/


// records and and executes gl commands
// currently the following commands are
// stored:
//
//
// glMatrixMode
// glLoadMatrixf, glLoadMatrixd, glLoadMatrixx,
// glMultMatrixf, glMultMatrixd, glMultMatrixx, 
// glPushMatrix
// glPopMatrix
// glLoadIdentity,
// glTranslatef, glTranslatefv, glTranslatex
// glScalef, glScalex
// glRotatef, glRotatex
// glOrtho, glOrthox
// glFrustum, glFrustumx
//
// glClearColor, glClearColorx
// glClearDepth, glClearDepthx
// glClear
// glColorMask
// glPolygonMode
// glEnable
// glDisable
// glDepthMask
// glCullFace
// glFrontFace
// glScissor
// glFogi, glFogf
// glFogfv, glFogiv
// glBlendFunc
// glDepthFunc
//
// glLightf, glLightx
// glLightfv, glLightxv
// glLightModeli, glLightModelf, glLightModelx
// glLightModelfv,glLightModelxv
// glShadeModel
//
// glMaterialf, glMaterialx
// glMaterialfv, glMaterialiv, glMaterialxv
// glColorMaterial
// glPolygonMode
// 
// glTexEnvf
// glBindTexture
//
// glPushAttrib
// glPopAttrib
//
// glBegin
// glEnd
// glVertex
// glColor
// glNormal
// glTexCoord
//
// glCallLists
// glCallList
//
// glDrawPixels
// glTexImage2D
//
//
// yet missing:
//
//   glDrawElements


#ifndef __KLDISPLAYLIST_HEADERFILE__
#define __KLDISPLAYLIST_HEADERFILE__


#include "Base/klChunkedMemory.h"

class klContext;
class klDisplayList;

//typedef int (klDisplayList::* DL_EXECCOMMAND)(const void* nData);

typedef int (*DPL_EXECCOMMAND)(const void* nData, klContext* nContext);

struct CMD_Dummy {
	DPL_EXECCOMMAND	func;
};


class klDisplayList
{
public:
	enum {
		BLOCK_SIZE = 4096
	};


	klDisplayList(unsigned int nListName, klContext* nContext) : commands(BLOCK_SIZE)
	{
		context = nContext;
		listName = nListName;
	}

	unsigned int getName() const {  return listName;  }

	void reset()
	{
		commands.cleanup();
		for(int i=0; i<storedExecMem.getSize(); i++)
			delete static_cast<unsigned char *>(storedExecMem[i]);
		storedExecMem.removeAll();
	}

	void execute()
	{
		// hopefully the calling function turned of recording
		// and turned on command execution...

		klChunkedMemoryIterator it(commands);

		while(it.hasMoreBlocks())
		{
			const void* data = it.getBlock();
			const CMD_Dummy* cmd = (CMD_Dummy*)data;

			int size = cmd->func(data, context);
			it.next(size);
		}
	}


	// all our store functions
	//
	void store_MatrixMode(GLenum nMode);
	void store_LoadMatrix(const klMatrix& nMatrix);
	void store_MultMatrix(const klMatrix& nMatrix);
	void store_PushMatrix();
	void store_PopMatrix();
	void store_LoadIdentity();
	void store_Translate(const klFloat& nX, const klFloat& nY, const klFloat& nZ);
	void store_Scale(const klFloat& nX, const klFloat& nY, const klFloat& nZ);
	void store_Rotate(const klFloat& nA, const klFloat& nX, const klFloat& nY, const klFloat& nZ);
	void store_Ortho(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval);
	void store_Frustum(const klFloat& nLeft, const klFloat& nRight, const klFloat& nBottom, const klFloat& nTop, const klFloat& nNearval, const klFloat& nFarval);
	void store_ClearColor(const klVec4& nColor);
	void store_ClearDepth(const klFloat& nDepth);
	void store_Clear(int nFlags);
	void store_ColorMask(GLboolean nRed, GLboolean nGreen, GLboolean nBlue, GLboolean nAlpha);
	void store_Enable(GLenum nSelect);
	void store_Disable(GLenum nSelect);
	void store_DepthMask(GLboolean nSet);
	void store_CullFace(GLenum nCull);
	void store_FrontFace(GLenum nFace);
	void store_Scissor(GLint x, GLint y, GLsizei width, GLsizei height);
	void store_Fog(GLenum pname, const klFloat& param);
	void store_FogColor(const klColor4& nColor);
	void store_BlendFunc(GLenum sfactor, GLenum dfactor);
	void store_DepthFunc(GLenum nFunc);
	void store_Lightv(GLenum nLight, GLenum nSelect, const klVec4& nValues);
	void store_LightModel(GLenum nParam, const klFloat& nValue);
	void store_LightModelv(GLenum nParam, const klVec4& nValues);
	void store_ShadeModel(GLenum nModel);
	void store_Material(GLenum nSide, GLenum nSelect, const klFloat& nValue);
	void store_Materialv(GLenum nSide, GLenum nSelect, const klVec4& nValues);
	void store_ColorMaterial(GLenum nSide, GLenum nMode);
	void store_PolygonMode(GLenum nSide, GLenum nPoly);
	void store_TexEnv(GLint nParam);
	void store_PushAttrib(GLbitfield nMask);
	void store_PopAttrib();
	void store_BindTexture(GLuint nTexture);
	void store_Begin(GLenum nType);
	void store_End();
	void store_Color(const klColor4& nColor);
	void store_Normal(const klVec3& nNormal);
	void store_TexCoord(const klVec4& nTexCoord);
	void store_Vertex(const klVec4& nVertex);
	void store_TexImage2D(GLint level, GLint internalformat, GLsizei width, GLsizei height,
						  GLint border, GLenum format, GLenum type, const GLvoid *pixels);
	void store_DrawPixels(GLsizei nWidth, GLsizei nHeight, GLenum nFormat, GLenum nType, const void* nPixels);

protected:
	unsigned int		listName;
	klChunkedMemory		commands;
	klContext*			context;
	klPtrList<void>		storedExecMem;			// here we store memory blocks needed for execution (such as texture)
												// the main purpose is to be able to delete them when cleaning up
};


#endif //__KLDISPLAYLIST_HEADERFILE__
