/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * Copies the video backbuffer onto the device's display
 * (used under windows ce only). Uses macro blocks to good
 * cache usage (this is multiple times faster than rotating
 * the whole image at once...)
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klBlitter.cpp,v 1.2 2004/01/13 14:40:27 tuhtah Exp $
 *
\**************************************************************************/


#include "klBlitter.h"
#include <memory.h>


namespace klBlitter
{


void
rotateBlockCW(unsigned short* nDst, unsigned int nDstPitch,
			  const unsigned short* nSrc, unsigned int nSrcPitch, unsigned int nW, unsigned int nH)
{
	int x,y=nH;
	nSrc += (nH-1)*nSrcPitch;

	while(y--)
	{
		x = nW;
		unsigned short* dstT = nDst;
		const unsigned short* srcT = nSrc;

		while(x--)
		{
			*dstT = *srcT;
			srcT++;
			dstT+=nDstPitch;
		}

		nDst ++;
		nSrc -= nSrcPitch;
	}
}


void
rotateBlockCCW(unsigned short* nDst, unsigned int nDstPitch,
			   const unsigned short* nSrc, unsigned int nSrcPitch, unsigned int nW, unsigned int nH)
{
	int x,y=nH;
	nDst += (nH-1)*nDstPitch;

	while(y--)
	{
		x = nW;
		unsigned short* dstT = nDst;
		const unsigned short* srcT = nSrc;

		while(x--)
		{
			*dstT = *srcT;
			srcT++;
			dstT-=nDstPitch;
		}

		nDst ++;
		nSrc += nSrcPitch;
	}
}


void
blit(unsigned short* nDst, const unsigned short* nSrc, int nSrcW, int nSrcH,
	 ROTATION nRot,
	 int nDstW, int nDstH,
	 int nBlockW, int nBlockH)
{
	if(nDstW==-1 || nDstH==-1)
	{
		if(nRot==ROTATE_90 || nRot==ROTATE_270)
		{
			nDstW = nSrcH;
			nDstH = nSrcW;
		}
		else
		{
			nDstW = nSrcW;
			nDstH = nSrcH;
		}
	}


	switch(nRot)
	{
	case ROTATE_0:
		{
			if(nDstW==nSrcW && nDstH==nSrcH)			// this is quite faster...
				memcpy(nDst, nSrc, nDstW*nDstH*2);
			else
			{
				unsigned int x,y=nSrcH;

				while(y--)
				{
					x = nSrcW;
					while(x--)
					{
						*nDst = *nSrc;

						nDst++;
						nSrc++;
					}
					nDst += nDstW-nSrcW;
				}
			}
		}
		break;

	case ROTATE_90:
		{
			unsigned int	x=0,y=0,
							numX=nSrcW/nBlockW,numY=nSrcH/nBlockH;

			for(y=0; y<numY; y++)
				for(x=0; x<numX; x++)
					rotateBlockCW((nDst+(numY-y-1)*nBlockW+x*nBlockH*nDstW), nDstW,
								  nSrc+(x*nBlockW+y*nBlockH*nSrcW), nSrcW,
								  nBlockW,nBlockH);
		}
		break;

	case ROTATE_180:
		{
			unsigned int x,y=nSrcH;
			nSrc += nSrcH*nSrcW-1;

			while(y--)
			{
				x = nSrcW;
				while(x--)
				{
					*nDst = *nSrc;

					nDst++;
					nSrc--;
				}
				nDst += nDstW-nSrcW;
			}
		}
		break;

	case ROTATE_270:
		{
			unsigned int	x=0,y=0,
							numX=nSrcW/nBlockW,numY=nSrcH/nBlockH;

			for(y=0; y<numY; y++)
				for(x=0; x<numX; x++)
					rotateBlockCCW((nDst+y*nBlockW+(numX-x-1)*nBlockH*nDstW), nDstW,
								   nSrc+(x*nBlockW+y*nBlockH*nSrcW), nSrcW,
								   nBlockW,nBlockH);
		}
		break;
	}
}


}	// namespace klBlitter
