/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Id: klEGLSurfaceX86Window.cpp,v 1.1 2004/01/13 11:20:10 drgoldie Exp $
 *
\**************************************************************************/


#include "klEGLSurfaceX86Window.h"


klEGLSurfaceX86Window::klEGLSurfaceX86Window()
{
	colorType = COLOR_TYPE_RGB565;
	depthType = DEPTH_TYPE_INT32;

    ZeroMemory( &bitmapInfo, sizeof( BITMAPINFO ) );
    bitmapInfo.bmiHeader.biSize = sizeof( bitmapInfo.bmiHeader );
    bitmapInfo.bmiHeader.biWidth  = 0;		// leave this uninitialized
    bitmapInfo.bmiHeader.biHeight = 0;		// leave this uninitialized
    bitmapInfo.bmiHeader.biPlanes = 1;
    bitmapInfo.bmiHeader.biBitCount = 32;
    bitmapInfo.bmiHeader.biCompression = BI_RGB;
    bitmapInfo.bmiHeader.biSizeImage = 0;
    bitmapInfo.bmiHeader.biXPelsPerMeter = 0;
    bitmapInfo.bmiHeader.biYPelsPerMeter = 0;
    bitmapInfo.bmiHeader.biClrUsed = 0;
    bitmapInfo.bmiHeader.biClrImportant = 0;

	bitmapData = NULL;

	// change this to false in order to render at full
	// resolution. since we target small display (mobile devices)
	// it makes sense to have this as default on the PC...
	halfResSurface = false;

	width = height = 0;
	cBuffer = NULL;
	zBuffer = NULL;

	hwnd = NULL;
	hdc = NULL;
}


klEGLSurfaceX86Window::~klEGLSurfaceX86Window()
{
	destroy();
}


bool
klEGLSurfaceX86Window::setNativeWindow(NativeWindowType nWnd)
{
	hwnd = nWnd;									// note: we keep this HWND and HDC
	HDC tmpHdc = ::GetDC(hwnd);						//       until the surface is destroyed

	return setNativeDisplay(tmpHdc);
}


bool
klEGLSurfaceX86Window::setNativeDisplay(NativeDisplayType nDisp)
{
	RECT rect;

	hdc = nDisp;

	if(GetBoundsRect(hdc, &rect, 0)==0)
	{
		width = height = 0;
		if(hwnd)
			::ReleaseDC(hwnd, hdc);
		return false;
	}

	if(halfResSurface)
		setSize((rect.right-rect.left)/2, (rect.bottom-rect.top)/2);
	else
		setSize((rect.right-rect.left), (rect.bottom-rect.top));
	return true;
}


NativeWindowType
klEGLSurfaceX86Window::getNativeWindow()
{
	return hwnd;
}


NativeDisplayType
klEGLSurfaceX86Window::getNativeDisplay()
{
	return hdc;
}


bool
klEGLSurfaceX86Window::destroy()
{
	bool ret = true;

	if(bitmapData)
		delete bitmapData;

	if(cBuffer)
		delete cBuffer;
	cBuffer = NULL;

	if(zBuffer)
		delete zBuffer;
	zBuffer = NULL;

	if(hdc && hwnd)
		ret = (::ReleaseDC(hwnd, hdc)==1);

	bitmapData = NULL;

	width = height = 0;

	hdc = NULL;
	hwnd = NULL;

	return ret;
}


void
klEGLSurfaceX86Window::setSize(int nWidth, int nHeight)
{
	if(width!=nWidth || height!=nHeight)
	{
		width = nWidth;
		height = nHeight;

		if(cBuffer)
			delete cBuffer;
		cBuffer = new unsigned short[width*height];

		if(zBuffer)
			delete zBuffer;
		zBuffer = new unsigned int[width*height];
	}

	if(bitmapData==NULL ||
	   bitmapInfo.bmiHeader.biWidth!=2*nWidth ||
	   bitmapInfo.bmiHeader.biHeight!=2*nHeight)
		{
			if(bitmapData)
				delete bitmapData;
			bitmapData = new RGBQUAD[4*nWidth*nHeight];			// 4x pixels for 2x zooming...

			bitmapInfo.bmiHeader.biWidth  = 2*nWidth;
			bitmapInfo.bmiHeader.biHeight = 2*nHeight;
		}
}


inline void convertPixel16To32BGRX(unsigned short nPix16, unsigned int& nPix32)
{
	const unsigned short RED_MASK    = 0x1F << 11;
	const unsigned short GREEN_MASK  = 0x3F << 5;
	const unsigned short BLUE_MASK   = 0x1F;

	nPix32 = (((nPix16&RED_MASK) >> 8) << 16) |
			 (((nPix16&GREEN_MASK) >> 3) << 8) |
			 (((nPix16&BLUE_MASK) << 3) << 0);
}


bool
klEGLSurfaceX86Window::flip()
{
	if(halfResSurface)
	{
		int x,y, w2 = 2*width;
		const unsigned short* src = reinterpret_cast<unsigned short*>(cBuffer) + width*(height-1);
		unsigned int* dst = (unsigned int*)bitmapData;
		unsigned int pix32;

		for(y=0; y<height; y++)
		{
			for(x=0; x<width; x++)
			{
				convertPixel16To32BGRX(*src, pix32);

				dst[0] = dst[1] = dst[w2] = dst[w2+1] = pix32;

				dst+=2;
				src++;
			}
			src -= w2;
			dst += w2;
		}

		SetDIBitsToDevice(hdc, 0,0, 2*width,2*height, 0,0, (UINT)0,
						  bitmapInfo.bmiHeader.biHeight, bitmapData, &bitmapInfo, DIB_RGB_COLORS);
	}
	else
	{
		int x,y, w2 = 2*width;
		const unsigned short* src = reinterpret_cast<unsigned short*>(cBuffer) + width*(height-1);
		unsigned int* dst = (unsigned int*)bitmapData;

		for(y=0; y<height; y++)
		{
			for(x=0; x<width; x++)
				convertPixel16To32BGRX(*src++, *dst++);

			src -= w2;
			dst += width;
		}

		SetDIBitsToDevice(hdc, 0,0, width,height, 0,0, (UINT)0,
						  bitmapInfo.bmiHeader.biHeight, bitmapData, &bitmapInfo, DIB_RGB_COLORS);
	}

	return true;
}
