/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565_Triangle.h,v 1.6 2004/02/10 18:19:31 drgoldie Exp $
 *
\**************************************************************************/


//
// class klRSW565 {
//

//
// Triangle drawing methods
// (all modes: shaded, textured, etc...)
//

void
drawTriangle(const DeviceVertex& nV0, const DeviceVertex& nV1, const DeviceVertex& nV2,
			 const MipMapLevel* nMipmapLevel)
{
	DeviceVertex v[3];
	const DeviceVertex *nv[3];
	EDGE_HINT edgeHint;

	klFloat		dx1 = nV1.pos[0]-nV0.pos[0], dy1 = nV1.pos[1]-nV0.pos[1],
				dx2 = nV2.pos[0]-nV0.pos[0], dy2 = nV2.pos[1]-nV0.pos[1];

	// take care we don't overflow fixed point when calculating 'wise'
	//
	dx1 >>= 4;	dy1 >>= 4;	dx2 >>= 4;	dy2 >>= 4;

	klFloat wise = dx1*dy2 - dy1*dx2, w,h;
	if(wise==0)			// area zero?
		return;

	bool isWise = wise<constants._0;
	w.setInt(nMipmapLevel ? nMipmapLevel->width-1 : 1);
	h.setInt(nMipmapLevel ? nMipmapLevel->height-1 : 1);

	nv[0] = &nV0;	nv[1] = &nV1;	nv[2] = &nV2;

	// The texture drawing method can only draw clockwise triangles...
	// so we rotate it if necessary during data setup
	//
	for(int i=0; i<3; i++)
	{
		const DeviceVertex& vs = isWise ? *nv[2-i] : *nv[i];

		v[i].pos = vs.pos;
		v[i].col = vs.col;
		v[i].col.multiplyBy255();

		if(settings.numtexstages>0)
		{
			v[i].texZ = vs.texZ;
			v[i].tex[0] = vs.tex[0] * w;
			v[i].tex[1] = vs.tex[1] * h;
		}

		if(settings.fog==RASTER_FOG_ON)
		{
			v[i].fog = vs.fog;
			v[i].fog.multiplyBy255();
		}
	}


	// check if we can optimize for special cases (e.g. flat shading or texturing without shading)
	//

	if(settings.blendEnabled)
		gouraudFullAlpha = (v[0].col[3]+v[1].col[3]+v[2].col[3]) > constants._764;	// are all alphas at 255?
	else
	{
		v[0].col[3] = v[1].col[3] = v[2].col[3] = constants._255;
		gouraudFullAlpha = true;
	}


	if((v[0].col.getSumInt()+v[1].col.getSumInt()+v[2].col.getSumInt())>3050)	// completely white? (that's 12*255=3060)
	{
		gouraudColor = GOURAUD_WHITE;
		edgeHint = (settings.numtexstages>0) ? HINT_TEXTURE : HINT_FLAT;
	}
	else
	if(v[0].col==v[1].col && v[0].col==v[2].col)								// flat?
	{
		gouraudColor = GOURAUD_FLAT;
		edgeHint = (settings.numtexstages>0) ? HINT_TEXTURE : HINT_FLAT;
	}
	else																		// otherwise a RGB shading
	{
		gouraudColor = GOURAUD_SHADED;
		edgeHint = (settings.numtexstages>0) ? HINT_TEXTURE_GOURAUD : HINT_GOURAUD;
	}
		

	/////////////////////////////////////////////////////////////////////////////////////////////


	int Top, Middle, Bottom, MiddleIsLeft, Height;
	int MiddleCompare, BottomCompare;
	klFloat		Y0 = v[0].pos[1],
				Y1 = v[1].pos[1],
				Y2 = v[2].pos[1];

	scanLineTexPersp = settings.doTex();
	sortVertices(Top, Middle, Bottom, MiddleCompare, BottomCompare, Y0, Y1, Y2);

	if(v[Top].pos[1]==v[Bottom].pos[1])
		return;

	// calculate some raster settings
	//
	edgeOptions = edgeOptionsBase;
	if(settings.numtexstages==0 || (settings.texenv==RASTER_TEXENV_MODULATE && gouraudColor!=GOURAUD_WHITE))
		edgeOptions |= OPT_SHADING_RGB;
	if(!gouraudFullAlpha && settings.blendEnabled)
		edgeOptions |= OPT_SHADING_ALPHA;
	if(scanLineTexPersp && checkTriangleSize(v, Bottom, Top))
	{
		// in case of small triangles we do affine texturing. the difference
		// is hardly noticable but we skip the cpu-costly gradient setup...
		edgeOptions |= OPT_AFFINE_TEX;
		scanLineTexPersp = false;
	}


	// check how texturing/shading is done
	//
	if(settings.numtexstages==0)
		shadeTexMode = MODE_PURE_SHADING;
	else
	if(settings.texenv==RASTER_TEXENV_DECAL || gouraudColor==GOURAUD_WHITE)
		shadeTexMode = MODE_PURE_TEXTURING;
	else
		shadeTexMode = MODE_MIXED_SHADING_TEXTURING;


	TexGradients gradients, *pGradients = (scanLineTexPersp ? &gradients : NULL);
	if(scanLineTexPersp)
		gradients.set(v);

	Edge	TopToBottom(pGradients,v,Top,Bottom, edgeOptions),
			TopToMiddle(pGradients,v,Top,Middle, edgeOptions),
			MiddleToBottom(pGradients,v,Middle,Bottom, edgeOptions),
			*pLeft, *pRight;

	// the triangle is expected clockwise, so
	// if bottom > middle then middle is right
	if(BottomCompare > MiddleCompare)
		{	MiddleIsLeft = 0;	pLeft = &TopToBottom;	pRight = &TopToMiddle;  }
	else
		{	MiddleIsLeft = 1;	pLeft = &TopToMiddle;	pRight = &TopToBottom;	}


	// get an edge funciton to interpolate values along the edges
	//
	STEPFUNC stepFunc = getEdgeFunction(edgeHint, scanLineTexPersp);
	if(!stepFunc)
		return;


	// calculate the initial buffer offset
	//
	curLineOffset = pLeft->Y * getWidth();


	/////////////////////////////////////////////////////////////////////////////7
	//
	//                   Finally we really start drawing...
	//

	// draw top part of the triangle
	//
	Height = TopToMiddle.Height;
	while(Height--) {
		drawScanLine(pLeft,pRight, gradients, nMipmapLevel);

		(TopToMiddle.*stepFunc)();
		(TopToBottom.*stepFunc)();
		curLineOffset += getWidth();
	}


	if(MiddleIsLeft)
		{	pLeft = &MiddleToBottom; pRight = &TopToBottom;  }
	else
		{	pLeft = &TopToBottom; pRight = &MiddleToBottom;	 }


	// draw bottom part of the triangle
	//
	Height = MiddleToBottom.Height;
	while(Height--) {
		drawScanLine(pLeft,pRight, gradients, nMipmapLevel);

		(MiddleToBottom.*stepFunc)();
		(TopToBottom.*stepFunc)();
		curLineOffset += getWidth();
	}
}


bool
checkTriangleSize(DeviceVertex const *pVertices, int nBot, int nTop)
{
	const int maxSize = 32;

	int y0 = pVertices[nTop].pos[1].getInt();
	int y2 = pVertices[nBot].pos[1].getInt();

	if((y2-y0)>maxSize)
		return false;

	int x0 = pVertices[0].pos[0].getInt(),
		x1 = pVertices[1].pos[0].getInt(),
		x2 = pVertices[2].pos[0].getInt(),
		tmp;

	if(x0>x1)
	{  tmp=x0;  x0=x1;  x1=tmp;  }
	if(x1>x2)
	{  tmp=x1;  x1=x2;  x2=tmp;  }
	if(x0>x1)
	{  tmp=x0;  x0=x1;  x1=tmp;  }

	return (x2-x0)<=maxSize;
}


static unsigned int
getShift(int nVal)
{
	switch(nVal)
	{
	case 1:		return 0;
	case 2:		return 1;
	case 4:		return 2;
	case 8:		return 3;
	case 16:	return 4;
	case 32:	return 5;
	case 64:	return 6;
	case 128:	return 7;
	case 256:	return 8;
	case 512:	return 9;
	case 1024:	return 10;
	}

	return 0;
}


void sortVertices(int& Top, int& Middle, int& Bottom, int& MiddleCompare, int& BottomCompare,
				  klFloat Y0, klFloat Y1, klFloat Y2)
{
	// sort vertices in y
	if(Y0 < Y1) {
		if(Y2 < Y0) {
			Top = 2; Middle = 0; Bottom = 1;
			MiddleCompare = 0; BottomCompare = 1;
		} else {
			Top = 0;
			if(Y1 < Y2) {
				Middle = 1; Bottom = 2;
				MiddleCompare = 1; BottomCompare = 2;
			} else {
				Middle = 2; Bottom = 1;
				MiddleCompare = 2; BottomCompare = 1;
			}
		}
	} else {
		if(Y2 < Y1) {
			Top = 2; Middle = 1; Bottom = 0;
			MiddleCompare = 1; BottomCompare = 0;
		} else {
			Top = 1;
			if(Y0 < Y2) {
				Middle = 0; Bottom = 2;
				MiddleCompare = 3; BottomCompare = 2;
			} else {
				Middle = 2; Bottom = 0;
				MiddleCompare = 2; BottomCompare = 3;
			}
		}
	}
}


// this method returns the Span-Func (which rasters scanlines) and
// the stepfunc which updates rasterization settings when stepping through the scanlines
STEPFUNC
getEdgeFunction(EDGE_HINT nEdgeHint, bool nDoPerspTex)
{
	switch(nEdgeHint)
	{
	case HINT_FLAT:
		return &Edge::stepFlat;

	case HINT_GOURAUD:
		return &Edge::stepShade;

	case HINT_TEXTURE:
		return nDoPerspTex ? &Edge::stepTex : &Edge::stepAffineTex;

	case HINT_TEXTURE_GOURAUD:
		return nDoPerspTex ? &Edge::stepTexShade : &Edge::stepAffineTexShade;
	}

	return NULL;
}


PIXELWRITEFUNC getPixelWriteFunc(klRasterizer_Settings& nSet)
{
	// we only have special a func if all color channels shall be written...
	//
	if((nSet.colorMask&COLOR_MASK_RGB)!=COLOR_MASK_RGB)
		return NULL;

	// we have a special func for these three cases
	//
	if(!nSet.blendEnabled || (nSet.blendSrc==GL_ONE && nSet.blendDst==GL_ZERO))
		return &klRSW565::drawScanLine_PixelWrite_One_Zero;							// simple decal
	else
	if(nSet.blendSrc==GL_ONE && nSet.blendDst==GL_ONE)
		return &klRSW565::drawScanLine_PixelWrite_One_One;							// normal add
	else
	if(nSet.blendSrc==GL_SRC_ALPHA && nSet.blendDst==GL_ONE_MINUS_SRC_ALPHA)
		return &klRSW565::drawScanLine_PixelWrite_SrcAlpha_OneMinusSrcAlpha;		// most common alpha blending case

	return NULL;
}


//
// } class klRSW565
//
