/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klVec4.h,v 1.5 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLVEC4T_HEADERFILE__
#define __KLVEC4T_HEADERFILE__


// 4D vector implementation with templated basic data type
//
// FLOAT specifies the basic data type
// see klFloat_fixed for a sample implementation
//
template<class FLOAT>
class klVec4T
{
	enum
	{
		INT1SHIFT = 31 - 16		// 16-bits fixed point precision
	};

public:
	klVec4T()														{}
	klVec4T(const float *nV)										{  setFloat(nV);  }
	klVec4T(float nV0, float nV1, float nV2, float nV3)				{  v[0].setFloat(nV0);	v[1].setFloat(nV1);	v[2].setFloat(nV2);	v[3].setFloat(nV3);  }
	klVec4T(const FLOAT& nV0, const FLOAT& nV1,
				  const FLOAT& nV2, const FLOAT& nV3)		{  v[0] = nV0;  v[1] = nV1;  v[2] = nV2;  v[3] = nV3;  }
	klVec4T(const klVec4T& nOther)							{  v[0] = nOther.v[0];  v[1] = nOther.v[1];  v[2] = nOther.v[2];  v[3] = nOther.v[3];  }

	void setFixed(const int* nV)		{  v[0].setFixed(nV[0]);	v[1].setFixed(nV[1]);		v[2].setFixed(nV[2]);		v[3].setFixed(nV[3]);	}
	void setInt(const int* nV)			{  v[0].setInt(nV[0]);		v[1].setInt(nV[1]);			v[2].setInt(nV[2]);			v[3].setInt(nV[3]);		}
	void setFloat(const float* nV)		{  v[0].setFloat(nV[0]);	v[1].setFloat(nV[1]);		v[2].setFloat(nV[2]);		v[3].setFloat(nV[3]);	}
	void setDouble(const double* nV)	{  v[0].setDouble(nV[0]);	v[1].setDouble(nV[1]);		v[2].setDouble(nV[2]);		v[3].setDouble(nV[3]);	}
	void setNative(const FLOAT* nV)		{  v[0] = nV[0];			v[1] = nV[1];				v[2] = nV[2];				v[3] = nV[3];			}
	void setInt1(const int* nV)			{  v[0].setFixed(nV[0]>>INT1SHIFT);		v[1].setFixed(nV[1]>>INT1SHIFT);
										   v[2].setFixed(nV[2]>>INT1SHIFT);		v[3].setFixed(nV[3]>>INT1SHIFT);  }

	void setFixed(int nIdx, int nV)		{  v[nIdx].setFixed(nV);  }
	void setInt(int nIdx, int nV)		{  v[nIdx].setInt(nV);  }
	void setFloat(int nIdx, float nV)	{  v[nIdx].setFloat(nV);  }
	void setDouble(int nIdx, double nV)	{  v[nIdx].setDouble(nV);  }

	void setFixed(int nV0, int nV1, int nV2, int nV3)				{  v[0].setFixed(nV0);	v[1].setFixed(nV1);		v[2].setFixed(nV2);		v[3].setFixed(nV3);		}
	void setInt(int nV0, int nV1, int nV2, int nV3)					{  v[0].setInt(nV0);	v[1].setInt(nV1);		v[2].setInt(nV2);		v[3].setInt(nV3);		}
	void setFloat(float nV0, float nV1, float nV2, float nV3)		{  v[0].setFloat(nV0);  v[1].setFloat(nV1);		v[2].setFloat(nV2);		v[3].setFloat(nV3);		}
	void setDouble(double nV0, double nV1, double nV2, double nV3)	{  v[0].setDouble(nV0);	v[1].setDouble(nV1);	v[2].setDouble(nV2);	v[3].setDouble(nV3);	}

	void getByte(unsigned char* nV)	{  for(int i=0; i<4; i++)  nV[i] = v[i].getByte();  }
	void getByte(int* nV)			{  for(int i=0; i<4; i++)  nV[i] = v[i].getByte();  }
	void getFixed(int* nV)			{  for(int i=0; i<4; i++)  nV[i] = v[i].getFixed();  }
	void getInt(int* nV)			{  for(int i=0; i<4; i++)  nV[i] = v[i].getInt();  }
	void getInt1(int* nV)			{  for(int i=0; i<4; i++)  nV[i] = v[i].getInt()<<INT1SHIFT;  }
	void getFloat(float* nV)		{  for(int i=0; i<4; i++)  nV[i] = v[i].getFloat();  }
	void getDouble(double* nV)		{  for(int i=0; i<4; i++)  nV[i] = v[i].getDouble();  }

	const FLOAT& operator[](int nIdx) const  {  return v[nIdx];  }
	FLOAT& operator[](int nIdx)  {  return v[nIdx];  }

	FLOAT getSum()  {  return v[0]+v[1]+v[2]+v[3];  }

	int getSumInt()  {  return v[0].getInt()+v[1].getInt()+v[2].getInt()+v[3].getInt();  }

	void normalize()  {  FLOAT invLen = getInvLength();  *this *= invLen;  }

	FLOAT getLength() const  {  return FLOAT(sqrt(v[0]*v[0] + v[1]*v[1] + v[2]*v[2] + v[3]*v[3]));  }

	inline FLOAT getInvLength() const;

	inline void transform(const klMatrixT<FLOAT>& nM, const klVec4T<FLOAT>& nV);


	inline static FLOAT transformZ(const klMatrixT<FLOAT>& nM, const klVec4T<FLOAT>& nV);


	inline void interpolate(const klVec4T<FLOAT>& nLeft, const klVec4T<FLOAT>& nRight, const FLOAT& nF);

	inline void divideByW();

	inline klVec4T& multiplyComonentWise(const klVec4T<FLOAT>& nV0, const klVec4T<FLOAT>& nV1);

	void multiplyBy255()		{  v[0].multiplyBy255();  v[1].multiplyBy255();  v[2].multiplyBy255();  v[3].multiplyBy255();  }

	klVec4T& operator=(const klVec4T& nO)  {  v[0]=nO.v[0];  v[1]=nO.v[1];  v[2]=nO.v[2];  v[3]=nO.v[3];  return *this;  }

	// overloaded operators
	//
	klVec4T& operator+=(const klVec4T& nO)	{  v[0]+=nO.v[0];  v[1]+=nO.v[1];  v[2]+=nO.v[2];  v[3]+=nO.v[3];  return *this;  }
	klVec4T& operator-=(const klVec4T& nO)	{  v[0]-=nO.v[0];  v[1]-=nO.v[1];  v[2]-=nO.v[2];  v[3]-=nO.v[3];  return *this;  }

	inline klVec4T& operator*=(const FLOAT& nV);

	bool operator==(const klVec4T& nO) const  {  return v[0]==nO.v[0] && v[1]==nO.v[1] && v[2]==nO.v[2] && v[3]==nO.v[3];  }

	// unfortunately template declarations are handled quite differentely on various compilers...
	//
#ifdef _IS_WINDOWS_
	friend inline const klVec4T operator+(const klVec4T& left, const klVec4T& right);
	friend inline const klVec4T operator-(const klVec4T& left, const klVec4T& right);
#else
  #ifdef GCC_295X
	friend const klVec4T<FLOAT> operator+ (const klVec4T& left, const klVec4T& right);
	friend const klVec4T<FLOAT> operator- (const klVec4T& left, const klVec4T& right);
  #else
	friend const klVec4T<FLOAT> operator+<> (const klVec4T& left, const klVec4T& right);
	friend const klVec4T<FLOAT> operator-<> (const klVec4T& left, const klVec4T& right);
  #endif // GCC_2.95.X
#endif // _IS_WINDOWS_

protected:
	FLOAT v[4];
};


template<class FLOAT> inline const klVec4T<FLOAT>
operator+(const klVec4T<FLOAT>& left, const klVec4T<FLOAT>& right)
{
	return klVec4T<FLOAT>(left.v[0]+right.v[0], left.v[1]+right.v[1], left.v[2]+right.v[2], left.v[3]+right.v[3]);
}


template<class FLOAT> inline const klVec4T<FLOAT>
operator-(const klVec4T<FLOAT>& left, const klVec4T<FLOAT>& right)
{
	return klVec4T<FLOAT>(left.v[0]-right.v[0], left.v[1]-right.v[1], left.v[2]-right.v[2], left.v[3]-right.v[3]);
}


template<class FLOAT> inline void
klVec4T<FLOAT>::transform(const klMatrixT<FLOAT>& nM, const klVec4T<FLOAT>& nV)
{
	v[0] = nM[0] * nV[0] + nM[4] * nV[1] + nM[8] *  nV[2] + nV[3] * nM[12];
	v[1] = nM[1] * nV[0] + nM[5] * nV[1] + nM[9] *  nV[2] + nV[3] * nM[13];
	v[2] = nM[2] * nV[0] + nM[6] * nV[1] + nM[10] * nV[2] + nV[3] * nM[14];
	v[3] = nM[3] * nV[0] + nM[7] * nV[1] + nM[11] * nV[2] + nV[3] * nM[15];
}


template<class FLOAT> inline FLOAT
klVec4T<FLOAT>::transformZ(const klMatrixT<FLOAT>& nM, const klVec4T<FLOAT>& nV)
{
	return nM[2] * nV[0] + nM[6] * nV[1] + nM[10] * nV[2] + nV[3] * nM[14];
}


template<class FLOAT> inline void
klVec4T<FLOAT>::interpolate(const klVec4T<FLOAT>& nV0, const klVec4T<FLOAT>& nV1, const FLOAT& nF)
{
	FLOAT one;
	one.setInt(1);
	FLOAT	rF = one - nF;

	v[0] = rF*nV0.v[0] + nF*nV1.v[0];
	v[1] = rF*nV0.v[1] + nF*nV1.v[1];
	v[2] = rF*nV0.v[2] + nF*nV1.v[2];
	v[3] = rF*nV0.v[3] + nF*nV1.v[3];
}


template<class FLOAT> inline void
klVec4T<FLOAT>::divideByW()
{
	if(v[3]==1 || v[3]==0)
		return;

	FLOAT f = v[3];
	f.inverse();

	v[0] *= f;
	v[1] *= f;
	v[2] *= f;
	v[3].setInt(1);
}


template<class FLOAT> inline klVec4T<FLOAT>&
klVec4T<FLOAT>::multiplyComonentWise(const klVec4T<FLOAT>& nV0, const klVec4T<FLOAT>& nV1)
{
	v[0] = nV0.v[0] * nV1.v[0];
	v[1] = nV0.v[1] * nV1.v[1];
	v[2] = nV0.v[2] * nV1.v[2];
	v[3] = nV0.v[3] * nV1.v[3];

	return *this;
}


template<class FLOAT> inline klVec4T<FLOAT>&
klVec4T<FLOAT>::operator*=(const FLOAT& nV)
{
	v[0]*=nV;  v[1]*=nV;  v[2]*=nV;  v[3]*=nV;
	return *this;
}


template<class FLOAT> inline FLOAT
klVec4T<FLOAT>::getInvLength() const
{
	FLOAT f = v[0]*v[0] + v[1]*v[1] + v[2]*v[2] + v[3]*v[3];

	f.inverseSqrt();
	return f;
}


//////////////////////////////////////////////
//
//  GPP optimized partitial specializations
//
#ifdef _USE_FIXED_GPP_


template<> inline void
klVec4T<klFloat_fixed<class klFixedBase_gpp> >::transform(const klMatrixT<klFloat_fixed<class klFixedBase_gpp> >& nM, const klVec4T<klFloat_fixed<class klFixedBase_gpp> >& nV)
{
	if(nV[3]==1)
		gppVec3DTransform_16_32s((GPP_VEC3D*)(&nV), (GPP_VEC4D*)this, (int*)&nM);
	else
	{
		v[0] = nM[0] * nV[0] + nM[4] * nV[1] + nM[8] *  nV[2] + nV[3] * nM[12];
		v[1] = nM[1] * nV[0] + nM[5] * nV[1] + nM[9] *  nV[2] + nV[3] * nM[13];
		v[2] = nM[2] * nV[0] + nM[6] * nV[1] + nM[10] * nV[2] + nV[3] * nM[14];
		v[3] = nM[3] * nV[0] + nM[7] * nV[1] + nM[11] * nV[2] + nV[3] * nM[15];
	}
}


template<> inline klVec4T<klFloat_fixed<class klFixedBase_gpp> >&
klVec4T<klFloat_fixed<class klFixedBase_gpp> >::operator*=(const klFloat_fixed<class klFixedBase_gpp>& nV)
{
	gppVec3DScale_16_32s((GPP_VEC3D*)this, nV.getFixed());
	v[3]*=nV;
	return *this;
}


template<> inline void
klVec4T<klFloat_fixed<class klFixedBase_gpp> >::divideByW()
{
	if(v[3]==1 || v[3]==0)
		return;

	FLOAT f = v[3];
	f.inverse();

	gppVec3DScale_16_32s((GPP_VEC3D*)this, f.getFixed());
	v[3].setInt(1);
}


template<> inline klFloat_fixed<class klFixedBase_gpp>
klVec4T<klFloat_fixed<class klFixedBase_gpp> >::getInvLength() const
{
	FLOAT f;
	unsigned int fixed;

	gppVec3DLengthSq_16_32s((GPP_VEC3D*)this, &fixed);
	f.setFixed(fixed);
	f += v[3]*v[3];
	f.inverseSqrt();

	return f;
}


template<> inline void
klVec4T<klFloat_fixed<class klFixedBase_gpp> >::interpolate(const klVec4T<klFloat_fixed<class klFixedBase_gpp> >& nV0, const klVec4T<klFloat_fixed<class klFixedBase_gpp> >& nV1, const klFloat_fixed<class klFixedBase_gpp>& nF)
{
	FLOAT revF;
	revF.setFixed((1<<16)-nF.getFixed());
	int v0[3], v1[3];

	v0[0] = nV0[0].getFixed();		v0[1] = nV0[1].getFixed();		v0[2] = nV0[2].getFixed();
	v1[0] = nV1[0].getFixed();		v1[1] = nV1[1].getFixed();		v1[2] = nV1[2].getFixed();

	gppVec3DScale_16_32s((GPP_VEC3D*)v0, revF.getFixed());
	gppVec3DScale_16_32s((GPP_VEC3D*)v1, nF.getFixed());
	gppVec3DAdd_n_32s((GPP_VEC3D*)v0, (GPP_VEC3D*)v1, (GPP_VEC3D*)this);
	v[3] = revF*nV0.v[3] + nF*nV1.v[3];
}




#endif //_USE_FIXED_GPP_


#endif //__KLVEC4T_HEADERFILE__
