/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/Base/klFloat_fixed.h,v 1.4 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLFLOAT_FIXED_HEADERFILE__
#define __KLFLOAT_FIXED_HEADERFILE__

#include <math.h>


// fixed point implementation of klFloat
//
// MATHBASE provides concrete methods for
// internal usage. while klFloat_fixed is platform
// independent, MATHBASE is the correct place
// to implement hardware specific optimizations.
// (see klFixedBase_generic & klFixedBase_gpp)
//


// VC.NET finally implements friend templates correctly
// Unfortunately this means we have to provide two different
// versions for VC6 and VC.NET
//
#ifdef _FIX_TEMPLATE_
#define VCNETFIX <typename MATHBASE>
#else
#define VCNETFIX
#endif //_FIX_TEMPLATE_


// forward declare all friend functions
//
template <typename> class klFloat_fixed;

template <typename MATHBASE> klFloat_fixed<MATHBASE> sqrt (const klFloat_fixed<MATHBASE>& nV);
template <typename MATHBASE> klFloat_fixed<MATHBASE> cos (const klFloat_fixed<MATHBASE>& nV);
template <typename MATHBASE> klFloat_fixed<MATHBASE> sin (const klFloat_fixed<MATHBASE>& nV);
template <typename MATHBASE> klFloat_fixed<MATHBASE> fabs (const klFloat_fixed<MATHBASE>& nV);
template <typename MATHBASE> klFloat_fixed<MATHBASE> ceil (const klFloat_fixed<MATHBASE>& nV);

template <typename MATHBASE> klFloat_fixed<MATHBASE> operator- (const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator- (const klFloat_fixed<MATHBASE>& left, float right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator- (float left, const klFloat_fixed<MATHBASE>& right);

template <typename MATHBASE> klFloat_fixed<MATHBASE> operator+ (const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator+ (const klFloat_fixed<MATHBASE>& left, float right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator+ (float left, const klFloat_fixed<MATHBASE>& right);

template <typename MATHBASE> klFloat_fixed<MATHBASE> operator* (const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator* (const klFloat_fixed<MATHBASE>& left, float right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator* (float left, const klFloat_fixed<MATHBASE>& right);

template <typename MATHBASE> klFloat_fixed<MATHBASE> operator/ (const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator/ (const klFloat_fixed<MATHBASE>& left, float right);
template <typename MATHBASE> klFloat_fixed<MATHBASE> operator/ (float left, const klFloat_fixed<MATHBASE>& right);



template<class MATHBASE>
class klFloat_fixed
{
public:
	klFloat_fixed()								{}
	klFloat_fixed(float nV)						{  setFloat(nV);  }
	klFloat_fixed(double nV)						{  setDouble(nV);  }
	klFloat_fixed(const klFloat_fixed& nOther)	{  v = nOther.v;  }

	// custom interface
	//
	void setFixed(int nV)							{  v = nV;  }
	void setInt(int nV)								{  v = MATHBASE::fixedFromInt(nV);  }
	void setFloat(float nV)							{  v = MATHBASE::fixedFromFloat(nV);  }
	void setDouble(double nV)						{  v = MATHBASE::fixedFromDouble(nV);  }
	void setKlFloat(const klFloat_fixed& nOther)	{  v = nOther.v;  }

	int getByte() const         {  return v>255<<(MATHBASE::PBITS-8) ? 255 : v>>(MATHBASE::PBITS-8);  }
	int getFixed() const        {  return v;  }
	int getInt() const			{  return v>>MATHBASE::PBITS;  }
	float getFloat() const		{  return MATHBASE::floatFromFixed(v);  }
	double getDouble() const	{  return MATHBASE::doubleFromFixed(v);  }

	void inverse(const klFloat_fixed& nOther)		{  v = MATHBASE::inverse(nOther.v);  }
	void inverse()									{  inverse(*this);  }

	void inverseSqrt(const klFloat_fixed& nOther)	{  v = MATHBASE::inverseSqrt(nOther.v);  }
	void inverseSqrt()								{  inverseSqrt(*this);  }

	void multiplyBy255()							{  int vOld = v;   v<<=8;  v-=vOld;  }


	// some standard math.h routines applied to this class
	//
#ifdef _IS_LINUX_
	friend klFloat_fixed<MATHBASE> sqrt<> VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed<MATHBASE> cos<> VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed<MATHBASE> sin<> VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed<MATHBASE> fabs<> VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed<MATHBASE> ceil<> VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
#else
	friend klFloat_fixed sqrt VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed cos VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed sin VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed fabs VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
	friend klFloat_fixed ceil VCNETFIX (const klFloat_fixed<MATHBASE>& nV);
#endif //_IS_LINUX_
	
	// overloaded operators
	//
	klFloat_fixed& operator=(int nV)						{  setInt(nV);  return *this; }
	klFloat_fixed& operator=(float nV)						{  setFloat(nV);  return *this; }
	klFloat_fixed& operator=(double nV)					{  setDouble(nV);  return *this; }
	klFloat_fixed& operator=(const klFloat_fixed& nOther)	{  v = nOther.v;  return *this;  }

	klFloat_fixed& operator+=(int nV)	{  v += MATHBASE::fixedFromInt(nV);  return *this;  }
	klFloat_fixed& operator-=(int nV)	{  v -= MATHBASE::fixedFromInt(nV);  return *this;  }
	klFloat_fixed& operator*=(int nV)	{  v = MATHBASE::multiply(v, MATHBASE::fixedFromInt(nV));  return *this;  }
	klFloat_fixed& operator/=(int nV)	{  v = MATHBASE::divide(v, MATHBASE::fixedFromInt(nV));  return *this;  }

	klFloat_fixed& operator+=(float nV)	{  v += MATHBASE::fixedFromFloat(nV);  return *this;  }
	klFloat_fixed& operator-=(float nV)	{  v -= MATHBASE::fixedFromFloat(nV);  return *this;  }
	klFloat_fixed& operator*=(float nV)	{  v = MATHBASE::multiply(v, MATHBASE::fixedFromFloat(nV));  return *this;  }
	klFloat_fixed& operator/=(float nV)	{  v = MATHBASE::divide(v, MATHBASE::fixedFromFloat(nV));  return *this;  }

	klFloat_fixed& operator+=(double nV)	{  v += MATHBASE::fixedFromDouble(nV);  return *this;  }
	klFloat_fixed& operator-=(double nV)	{  v -= MATHBASE::fixedFromDouble(nV);  return *this;  }
	klFloat_fixed& operator*=(double nV)	{  v = MATHBASE::multiply(v, fixedFromDouble(nV));  return *this;  }
	klFloat_fixed& operator/=(double nV)	{  v = MATHBASE::divide(v, fixedFromDouble(nV));  return *this;  }

	klFloat_fixed& operator+=(const klFloat_fixed& nOther)	{  v+=nOther.v;  return *this;  }
	klFloat_fixed& operator-=(const klFloat_fixed& nOther)	{  v-=nOther.v;  return *this;  }
	klFloat_fixed& operator*=(const klFloat_fixed& nOther)	{  v = MATHBASE::multiply(v, nOther.v);  return *this;  }
	klFloat_fixed& operator/=(const klFloat_fixed& nOther)	{  v = MATHBASE::divide(v, nOther.v);  return *this;  }

	klFloat_fixed& operator>>=(int nBits)	{  v>>=nBits;  return *this;  }
	klFloat_fixed& operator<<=(int nBits)	{  v<<=nBits;  return *this;  }

	bool operator==(const klFloat_fixed& nOther) const		{  return v==nOther.v;  }
	bool operator!=(const klFloat_fixed& nOther) const		{  return v!=nOther.v;  }
	bool operator<=(const klFloat_fixed& nOther) const		{  return v<=nOther.v;  }
	bool operator>=(const klFloat_fixed& nOther) const		{  return v>=nOther.v;  }
	bool operator<(const klFloat_fixed& nOther) const		{  return v<nOther.v;  }
	bool operator>(const klFloat_fixed& nOther) const		{  return v>nOther.v;  }

	bool operator==(int nOther) const		{  return v == MATHBASE::fixedFromInt(nOther);  }
	bool operator!=(int nOther) const		{  return v != MATHBASE::fixedFromInt(nOther);  }
	bool operator<=(int nOther) const		{  return v <= MATHBASE::fixedFromInt(nOther);  }
	bool operator>=(int nOther) const		{  return v >= MATHBASE::fixedFromInt(nOther);  }
	bool operator<(int nOther) const		{  return v <  MATHBASE::fixedFromInt(nOther);  }
	bool operator>(int nOther) const		{  return v >  MATHBASE::fixedFromInt(nOther);  }

	bool operator==(float nOther) const		{  return v == MATHBASE::fixedFromFloat(nOther);  }
	bool operator!=(float nOther) const		{  return v != MATHBASE::fixedFromFloat(nOther);  }
	bool operator<=(float nOther) const		{  return v <= MATHBASE::fixedFromFloat(nOther);  }
	bool operator>=(float nOther) const		{  return v >= MATHBASE::fixedFromFloat(nOther);  }
	bool operator<(float nOther) const		{  return v <  MATHBASE::fixedFromFloat(nOther);  }
	bool operator>(float nOther) const		{  return v >  MATHBASE::fixedFromFloat(nOther);  }

#ifdef _IS_WINDOWS_
	friend klFloat_fixed operator+ VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right);
	friend klFloat_fixed operator+ VCNETFIX (const klFloat_fixed& left, float right);
	friend klFloat_fixed operator+ VCNETFIX (float left, const klFloat_fixed& right);

	friend klFloat_fixed operator- VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right);
	friend klFloat_fixed operator- VCNETFIX (const klFloat_fixed& left, float right);
	friend klFloat_fixed operator- VCNETFIX (float left, const klFloat_fixed& right);

	friend klFloat_fixed operator* VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right);
	friend klFloat_fixed operator* VCNETFIX (const klFloat_fixed& left, float right);
	friend klFloat_fixed operator* VCNETFIX (const klFloat_fixed& left, int right);
	friend klFloat_fixed operator* VCNETFIX (float left, const klFloat_fixed& right);

	friend klFloat_fixed operator/ VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right);
	friend klFloat_fixed operator/ VCNETFIX (const klFloat_fixed& left, float right);
	friend klFloat_fixed operator/ VCNETFIX (float left, const klFloat_fixed& right);
#else
	friend klFloat_fixed<MATHBASE> operator+ VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(left.v+right.v);};
	friend klFloat_fixed<MATHBASE> operator+ VCNETFIX (const klFloat_fixed& left, float right){
	return klFloat_fixed<MATHBASE>(left.v+MATHBASE::fixedFromFloat(right));};
	friend klFloat_fixed<MATHBASE> operator+ VCNETFIX (float left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::fixedFromFloat(left)+right.v);};

	friend klFloat_fixed operator- VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(left.v-right.v);};
	friend klFloat_fixed operator- VCNETFIX (const klFloat_fixed& left, float right){
	return klFloat_fixed<MATHBASE>(left.v-MATHBASE::fixedFromFloat(right));};
	friend klFloat_fixed operator- VCNETFIX (float left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::fixedFromFloat(left)-right.v);};

	friend klFloat_fixed operator* VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(left.v, right.v));};
	friend klFloat_fixed operator* VCNETFIX (const klFloat_fixed& left, float right){
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(left.v, MATHBASE::fixedFromFloat(right)));};
	friend klFloat_fixed operator* VCNETFIX (float left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(MATHBASE::fixedFromFloat(left), right.v));};

	friend klFloat_fixed operator/ VCNETFIX (const klFloat_fixed& left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(left.v, right.v));};
	friend klFloat_fixed operator/ VCNETFIX (const klFloat_fixed& left, float right){
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(left.v, MATHBASE::fixedFromFloat(right)));};
	friend klFloat_fixed operator/ VCNETFIX (float left, const klFloat_fixed& right){
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(MATHBASE::fixedFromFloat(left), right.v));};
#endif // _IS_WINDOWS_
	klFloat_fixed operator-() const  {  klFloat_fixed w;  w.v = -v;  return w;  }

protected:
	klFloat_fixed(int nV)		{  v = nV;  }		// constructor with fixed point parameter

	int v;
};

#ifdef _IS_WINDOWS_
// binary operator +
//
template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator+(const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(left.v+right.v);
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator+(const klFloat_fixed<MATHBASE>& left, float right)
{
	return klFloat_fixed<MATHBASE>(left.v+MATHBASE::fixedFromFloat(right));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator+(float left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::fixedFromFloat(left)+right.v);
}


// binary operator -
//
template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator-(const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(left.v-right.v);
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator-(const klFloat_fixed<MATHBASE>& left, float right)
{
	return klFloat_fixed<MATHBASE>(left.v-MATHBASE::fixedFromFloat(right));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator-(float left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::fixedFromFloat(left)-right.v);
}


// binary operator *
//
template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator*(const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(left.v, right.v));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator*(const klFloat_fixed<MATHBASE>& left, float right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(left.v, MATHBASE::fixedFromFloat(right)));
}


template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator*(const klFloat_fixed<MATHBASE>& left, int right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(left.v, MATHBASE::fixedFromInt(right)));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator*(float left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::multiply(MATHBASE::fixedFromFloat(left), right.v));
}


// binary operator /
//
template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator/(const klFloat_fixed<MATHBASE>& left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(left.v, right.v));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator/(const klFloat_fixed<MATHBASE>& left, float right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(left.v, MATHBASE::fixedFromFloat(right)));
}

template<class MATHBASE> inline klFloat_fixed<MATHBASE>
operator/(float left, const klFloat_fixed<MATHBASE>& right)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::divide(MATHBASE::fixedFromFloat(left), right.v));
}
#endif // _IS_WINDOWS_

// math.h methods
//
template<class MATHBASE> inline klFloat_fixed<MATHBASE>
cos(const klFloat_fixed<MATHBASE>& nV)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::cos(nV.v));
}


template<class MATHBASE> inline klFloat_fixed<MATHBASE>
sin(const klFloat_fixed<MATHBASE>& nV)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::sin(nV.v));
}


template<class MATHBASE> inline klFloat_fixed<MATHBASE>
fabs(const klFloat_fixed<MATHBASE>& nV)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::fabs(nV.v));
}


template<class MATHBASE> inline klFloat_fixed<MATHBASE>
sqrt(const klFloat_fixed<MATHBASE>& nV)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::sqrt(nV.v));
}


template<class MATHBASE> inline klFloat_fixed<MATHBASE>
ceil(const klFloat_fixed<MATHBASE>& nV)
{
	return klFloat_fixed<MATHBASE>(MATHBASE::ceil(nV.v));
}


#endif //__KLFLOAT_FIXED_HEADERFILE__
