/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../../../juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE

#include "juce_AlertWindow.h"
#include "../lookandfeel/juce_LookAndFeel.h"
#include "../buttons/juce_TextButton.h"
#include "../controls/juce_TextEditor.h"
#include "../controls/juce_ProgressBar.h"
#include "../juce_Desktop.h"
#include "../../../../juce_core/text/juce_LocalisedStrings.h"

static const int titleH = 24;
static const int iconWidth = 80;


//==============================================================================
class AlertWindowTextButton  : public TextButton
{
public:
    AlertWindowTextButton (const String& text)
        : TextButton (text, String::empty)
    {
        setWantsKeyboardFocus (true);
        setMouseClickGrabsKeyboardFocus (false);
    }

    int returnValue;

private:
    AlertWindowTextButton (const AlertWindowTextButton&);
    const AlertWindowTextButton& operator= (const AlertWindowTextButton&);
};

//==============================================================================
class AlertWindowTextEditor  : public TextEditor
{
public:
    AlertWindowTextEditor (const String& name,
                           const bool isPasswordBox)
        : TextEditor (name,
                      isPasswordBox ? T('*') : (const tchar) 0)
    {
    }

    ~AlertWindowTextEditor()
    {
    }

    void returnPressed()
    {
        // pass these up the component hierarchy to be trigger the buttons
        Component::keyPressed (KeyPress (KeyPress::returnKey, 0));
    }

    void escapePressed()
    {
        // pass these up the component hierarchy to be trigger the buttons
        Component::keyPressed (KeyPress (KeyPress::escapeKey, 0));
    }

private:
    AlertWindowTextEditor (const AlertWindowTextEditor&);
    const AlertWindowTextEditor& operator= (const AlertWindowTextEditor&);
};


//==============================================================================
AlertWindow::AlertWindow (const String& title,
                          const String& message,
                          AlertIconType iconType)
   : TopLevelWindow (title, true),
     alertIconType (iconType)
{
    backgroundColour = Colour::greyLevel (0.9f);

    if (message.isEmpty())
        text = T(" "); // to force an update if the message is empty

    setMessage (message);
    setWantsKeyboardFocus (false);

    for (int i = Desktop::getInstance().getNumComponents(); --i >= 0;)
    {
        Component* const c = Desktop::getInstance().getComponent (i);

        if (c != 0 && c->isAlwaysOnTop() && c->isShowing())
        {
            setAlwaysOnTop (true);
            break;
        }
    }

    lookAndFeelChanged();

    constrainer.setMinimumOnscreenAmounts (0x10000, 0x10000, 0x10000, 0x10000);
}

AlertWindow::~AlertWindow()
{
    for (int i = customComps.size(); --i >= 0;)
        removeChildComponent ((Component*) customComps[i]);

    deleteAllChildren();
}

//==============================================================================
void AlertWindow::setMessage (const String& message)
{
    const String newMessage (message.substring (0, 2048));

    if (text != newMessage)
    {
        text = newMessage;

        font.setHeight (15.0f);

        Font titleFont (font.getHeight() * 1.1f, Font::bold);
        textLayout.setText (getName() + T("\n\n"), titleFont);

        textLayout.appendText (text, font);

        updateLayout (true);
        repaint();
    }
}

//==============================================================================
void AlertWindow::buttonClicked (Button* button)
{
    for (int i = 0; i < buttons.size(); i++)
    {
        AlertWindowTextButton* const c = (AlertWindowTextButton*) buttons[i];

        if (button->getName() == c->getName())
        {
            if (c->getParentComponent() != 0)
                c->getParentComponent()->exitModalState (c->returnValue);

            break;
        }
    }
}

//==============================================================================
void AlertWindow::addButton (const String& name,
                             const int returnValue,
                             const KeyPress& shortcutKey1,
                             const KeyPress& shortcutKey2)
{
    AlertWindowTextButton* const b = new AlertWindowTextButton (name);

    b->returnValue = returnValue;
    b->addShortcut (shortcutKey1);
    b->addShortcut (shortcutKey2);
    b->addButtonListener (this);
    b->changeWidthToFitText (28);

    addAndMakeVisible (b);
    buttons.add (b);

    updateLayout (false);
}

int AlertWindow::getNumButtons() const
{
    return buttons.size();
}

//==============================================================================
void AlertWindow::addTextEditor (const String& name,
                                 const String& initialContents,
                                 const String& onScreenLabel,
                                 const bool isPasswordBox)
{
    AlertWindowTextEditor* const tc = new AlertWindowTextEditor (name, isPasswordBox);

    tc->setColours (Colours::black, Colours::white,
                    getLookAndFeel().textEditorHighlight, getLookAndFeel().comboBoxOutline,
                    Colours::transparentBlack);

    tc->setFont (font);
    tc->setText (initialContents);
    tc->setCaretPosition (initialContents.length());
    addAndMakeVisible (tc);
    textBoxes.add (tc);
    allComps.add (tc);
    textboxNames.add (onScreenLabel);

    updateLayout (false);
}

const String AlertWindow::getTextEditorContents (const String& nameOfTextEditor) const
{
    for (int i = textBoxes.size(); --i >= 0;)
        if (((TextEditor*)textBoxes[i])->getName() == nameOfTextEditor)
            return ((TextEditor*)textBoxes[i])->getText();

    return String::empty;
}


//==============================================================================
void AlertWindow::addComboBox (const String& name,
                               const StringArray& items,
                               const String& onScreenLabel)
{
    ComboBox* const cb = new ComboBox (name);

    for (int i = 0; i < items.size(); ++i)
        cb->addItem (items[i], i + 1);

    addAndMakeVisible (cb);
    cb->setSelectedItemIndex (0);

    comboBoxes.add (cb);
    allComps.add (cb);

    comboBoxNames.add (onScreenLabel);

    updateLayout (false);
}

ComboBox* AlertWindow::getComboBoxComponent (const String& nameOfList) const
{
    for (int i = comboBoxes.size(); --i >= 0;)
        if (((ComboBox*) comboBoxes[i])->getName() == nameOfList)
            return (ComboBox*) comboBoxes[i];

    return 0;
}

//==============================================================================
class AlertTextComp : public TextEditor
{
    AlertTextComp (const AlertTextComp&);
    const AlertTextComp& operator= (const AlertTextComp&);

public:
    AlertTextComp (const String& message,
                   const Font& font)
    {
        setReadOnly (true);
        setMultiLine (true, true);
        setCaretVisible (false);
        setScrollbarsShown (false);
        lookAndFeelChanged();

        setFont (font);
        setText (message, false);
    }

    ~AlertTextComp()
    {
    }

    void updateLayout (const int width)
    {
        TextLayout text;
        text.appendText (getText(), getFont());
        text.layout (width - 8, Justification::topLeft, true);
        setSize (width, text.getHeight() + 4);
    }

    void lookAndFeelChanged()
    {
        setColours (Colours::black, Colours::transparentBlack,
                    getLookAndFeel().textEditorHighlight,
                    Colours::transparentBlack, Colours::transparentBlack);
    }
};

void AlertWindow::addTextBlock (const String& text)
{
    AlertTextComp* const c = new AlertTextComp (text, font);

    textBlocks.add (c);
    allComps.add (c);

    addAndMakeVisible (c);

    updateLayout (false);
}

//==============================================================================
void AlertWindow::addProgressBarComponent (double& progressValue)
{
    ProgressBar* const pb = new ProgressBar (progressValue);

    progressBars.add (pb);
    allComps.add (pb);

    addAndMakeVisible (pb);

    updateLayout (false);
}

//==============================================================================
void AlertWindow::addCustomComponent (Component* const component)
{
    customComps.add (component);
    allComps.add (component);

    addAndMakeVisible (component);

    updateLayout (false);
}

int AlertWindow::getNumCustomComponents() const
{
    return customComps.size();
}

Component* AlertWindow::getCustomComponent (const int index) const
{
    return (Component*) customComps [index];
}

Component* AlertWindow::removeCustomComponent (const int index)
{
    Component* const c = getCustomComponent (index);

    if (c != 0)
    {
        customComps.removeValue (c);
        allComps.removeValue (c);
        removeChildComponent (c);

        updateLayout (false);
    }

    return c;
}

//==============================================================================
void AlertWindow::paint (Graphics& g)
{
    getLookAndFeel().drawAlertBox (g, *this, textArea, textLayout);

    g.setColour (Colours::black);
    g.setFont (12.0f);

    int i;
    for (i = textBoxes.size(); --i >= 0;)
    {
        if (textboxNames[i].isNotEmpty())
        {
            const TextEditor* const te = (TextEditor*) textBoxes[i];

            g.drawFittedText (textboxNames[i],
                              te->getX(), te->getY() - 14,
                              te->getWidth(), 14,
                              Justification::centredLeft, 1);
        }
    }

    for (i = comboBoxNames.size(); --i >= 0;)
    {
        if (comboBoxNames[i].isNotEmpty())
        {
            const ComboBox* const cb = (ComboBox*) comboBoxes[i];

            g.drawFittedText (comboBoxNames[i],
                              cb->getX(), cb->getY() - 14,
                              cb->getWidth(), 14,
                              Justification::centredLeft, 1);
        }
    }
}

void AlertWindow::updateLayout (const bool onlyIncreaseSize)
{
    const int wid = jmax (font.getStringWidth (text),
                          font.getStringWidth (getName()));

    const int sw = (int) sqrt (font.getHeight() * wid);
    int w = jmin (300 + sw * 2, (int)(getParentWidth() * 0.7f));
    const int edgeGap = 10;
    int iconSpace;

    if (alertIconType == NoIcon)
    {
        textLayout.layout (w, Justification::horizontallyCentred, true);
        iconSpace = 0;
    }
    else
    {
        textLayout.layout (w, Justification::left, true);
        iconSpace = iconWidth;
    }

    w = jmax (350, textLayout.getWidth() + iconSpace + edgeGap * 4);
    const int textLayoutH = textLayout.getHeight();
    const int textBottom = 16 + titleH + textLayoutH;
    int h = textBottom;

    if (buttons.size() > 0)
        h += 70;

    int buttonW = 40;
    int i;
    for (i = 0; i < buttons.size(); ++i)
        buttonW += 16 + ((const AlertWindowTextButton*)buttons[i])->getWidth();

    w = jmax (buttonW, w);

    h += (textBoxes.size() + comboBoxes.size() + progressBars.size()) * 40;

    if (buttons.size() > 0)
        h += ((AlertWindowTextButton*) buttons[0])->getHeight();

    for (i = customComps.size(); --i >= 0;)
    {
        w = jmax (w, ((Component*) customComps[i])->getWidth() + 40);
        h += 10 + ((Component*) customComps[i])->getHeight();
    }

    for (i = textBlocks.size(); --i >= 0;)
    {
        AlertTextComp* ac = (AlertTextComp*) textBlocks[i];
        ac->updateLayout ((int) (w * 0.8f));
        h += ac->getHeight() + 10;
    }

    h = jmin (getParentHeight() - 50, h);

    if (onlyIncreaseSize)
    {
        w = jmax (w, getWidth());
        h = jmax (h, getHeight());
    }

    if (! isVisible())
    {
        centreAroundComponent (0, w, h);
    }
    else
    {
        const int cx = getX() + getWidth() / 2;
        const int cy = getY() + getHeight() / 2;

        setBounds (cx - w / 2,
                   cy - h / 2,
                   w, h);
    }

    textArea.setBounds (edgeGap, edgeGap, getWidth() - (edgeGap * 2), h - edgeGap);

    const int spacer = 16;
    int totalWidth = -spacer;

    for (i = buttons.size(); --i >= 0;)
        totalWidth += ((AlertWindowTextButton*)buttons[i])->getWidth() + spacer;

    int x = (getWidth() - totalWidth) / 2;
    int y = (int)(getHeight() * 0.95f);

    for (i = 0; i < buttons.size(); ++i)
    {
        AlertWindowTextButton* const c = (AlertWindowTextButton*) buttons[i];
        int ny = proportionOfHeight (0.95f) - c->getHeight();
        c->setTopLeftPosition (x, ny);
        if (ny < y)
            y = ny;

        x += c->getWidth() + spacer;
    }

    y = textBottom;

    for (i = 0; i < allComps.size(); ++i)
    {
        Component* const c = (Component*) allComps[i];

        const int h = 22;

        const int comboIndex = comboBoxes.indexOf (c);
        if (comboIndex >= 0 && comboBoxNames [comboIndex].isNotEmpty())
            y += 18;

        const int tbIndex = textBoxes.indexOf (c);
        if (tbIndex >= 0 && textboxNames[tbIndex].isNotEmpty())
            y += 18;

        if (customComps.contains (c) || textBlocks.contains (c))
        {
            c->setTopLeftPosition ((getWidth() - c->getWidth()) / 2, y);
            y += c->getHeight() + 10;
        }
        else
        {
            c->setBounds (proportionOfWidth (0.1f), y, proportionOfWidth (0.8f), h);
            y += h + 10;
        }

        c->setFocusOrder ((unsigned short) (i + 100));
    }
}

bool AlertWindow::containsAnyExtraComponents() const
{
    return textBoxes.size()
            + comboBoxes.size()
            + progressBars.size()
            + customComps.size() > 0;
}

//==============================================================================
void AlertWindow::mouseDown (const MouseEvent&)
{
    dragger.startDraggingComponent (this, &constrainer);
}

void AlertWindow::mouseDrag (const MouseEvent& e)
{
    dragger.dragComponent (this, e);
}

void AlertWindow::keyPressed (const KeyPress& key)
{
    for (int i = buttons.size(); --i >= 0;)
    {
        AlertWindowTextButton* const b = (AlertWindowTextButton*)buttons[i];

        if (b->isRegisteredForShortcut (key))
        {
            b->triggerClick();
            return;
        }
    }

    if (buttons.size() == 0)
    {
        exitModalState (0);
    }
    else if (key.isKeyCode (KeyPress::returnKey)
              && buttons.size() == 1)
    {
        AlertWindowTextButton* const mb = (AlertWindowTextButton*) buttons.getFirst();
        mb->triggerClick();
    }
}

void AlertWindow::lookAndFeelChanged()
{
    addToDesktop (getLookAndFeel().getAlertBoxWindowFlags());
}

//==============================================================================
void AlertWindow::showMessageBox (AlertIconType iconType,
                                  const String& title,
                                  const String& message,
                                  const String& buttonText)
{
    AlertWindow aw (title, message, iconType);

    aw.addButton (buttonText.isEmpty() ? TRANS("ok")
                                       : buttonText,
                  0,
                  KeyPress (KeyPress::escapeKey, 0),
                  KeyPress (KeyPress::returnKey, 0));

    aw.runModalLoop();
}

bool AlertWindow::showOkCancelBox (AlertIconType iconType,
                                   const String& title,
                                   const String& message,
                                   const String& button1Text,
                                   const String& button2Text)
{
    const String bt1 (button1Text.isEmpty() ? TRANS("ok")     : button1Text);
    const String bt2 (button2Text.isEmpty() ? TRANS("cancel") : button2Text);

    const KeyPress okShortCut (String::toLowerCase (bt1[0]), 0);
    KeyPress cancelShortCut (String::toLowerCase (bt2[0]), 0);
    if (okShortCut == cancelShortCut)
        cancelShortCut = KeyPress();

    AlertWindow aw (title, message, iconType);
    aw.addButton (bt1, 1, KeyPress (KeyPress::returnKey, 0), okShortCut);
    aw.addButton (bt2, 0, KeyPress (KeyPress::escapeKey, 0), cancelShortCut);

    return aw.runModalLoop() != 0;
}

int AlertWindow::showYesNoCancelBox (AlertIconType iconType,
                                     const String& title,
                                     const String& message,
                                     const String& button1Text,
                                     const String& button2Text,
                                     const String& button3Text)
{
    const String bt1 (button1Text.isEmpty() ? TRANS("yes")    : button1Text);
    const String bt2 (button2Text.isEmpty() ? TRANS("no")     : button2Text);
    const String bt3 (button3Text.isEmpty() ? TRANS("cancel") : button3Text);

    const KeyPress yesShortCut (String::toLowerCase (bt1[0]), 0);
    KeyPress noShortCut (String::toLowerCase (bt2[0]), 0);
    if (yesShortCut == noShortCut)
        noShortCut = KeyPress();

    AlertWindow aw (title, message, iconType);

    aw.addButton (bt1, 1, yesShortCut);
    aw.addButton (bt2, 2, noShortCut);
    aw.addButton (bt3, 0, KeyPress (KeyPress::escapeKey, 0));

    return aw.runModalLoop();
}

END_JUCE_NAMESPACE
