/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_SCROLLBAR_JUCEHEADER__
#define __JUCE_SCROLLBAR_JUCEHEADER__

#include "../../../events/juce_ChangeBroadcaster.h"
#include "../../../events/juce_Timer.h"
#include "../buttons/juce_Button.h"


//==============================================================================
/**
    A scrollbar component.

    To use a scrollbar, set up its total range using the setRangeLimits() method - this
    sets the range of values it can represent. Then you can use setCurrentRange() to
    change the position and size of the scrollbar's 'thumb'.

    Registering a ChangeListener with the scrollbar will allow you to find out when
    the user moves it, and you can use the getCurrentRangeStart() to find out where
    they moved it to.

    The scrollbar will adjust its own visibility according to whether its thumb size
    allows it to actually be scrolled.

    For most purposes, it's probably easier to use a ViewportContainer or ListBox
    instead of handling a scrollbar directly.
*/
class JUCE_API  ScrollBar  : public Component,
                             public ChangeBroadcaster,
                             private Timer
{
public:
    //==============================================================================
    /** Creates a Scrollbar.

        @param isVertical           whether it should be a vertical or horizontal bar
        @param buttonsAreVisible    whether to show the up/down or left/right buttons
    */
    ScrollBar (const bool isVertical,
               const bool buttonsAreVisible = true);

    /** Destructor. */
    ~ScrollBar();

    //==============================================================================
    /** Returns true if the scrollbar is vertical, false if it's horizontal. */
    bool isVertical() const                             { return vertical; }

    /** Changes the scrollbar's direction.

        You'll also need to resize the bar appropriately - this just changes its internal
        layout.

        @param shouldBeVertical     true makes it vertical; false makes it horizontal.
    */
    void setOrientation (const bool shouldBeVertical);

    /** Shows or hides the scrollbar's buttons. */
    void setButtonVisibility (const bool buttonsAreVisible);

    //==============================================================================
    /** Sets the minimum and maximum values that the bar will move between.

        The bar's thumb will always be constrained so that the top of the thumb
        will be >= minimum, and the bottom of the thumb <= maximum.

        @see setCurrentRange
    */
    void setRangeLimits (const double minimum,
                         const double maximum);

    /** Returns the lower value that the thumb can be set to.

        This is the value set by setRangeLimits().
    */
    double getMinimumRangeLimit() const                 { return minimum; }

    /** Returns the upper value that the thumb can be set to.

        This is the value set by setRangeLimits().
    */
    double getMaximumRangeLimit() const                 { return maximum; }

    //==============================================================================
    /** Changes the position of the scrollbar's 'thumb'.

        This sets both the position and size of the thumb - to just set the position without
        changing the size, you can use setCurrentRangeStart().

        If this method call actually changes position, it will trigger a call to
        all the ChangeListeners that are registered with the scrollbar.

        @param newStart     the top (or left) of the thumb, in the range
                            getMinimumRangeLimit() <= newStart <= getMaximumRangeLimit(). If the
                            value is beyond these limits, it will be clipped.
        @param newSize      the size of the thumb, such that
                            getMinimumRangeLimit() <= newStart + newSize <= getMaximumRangeLimit(). If the
                            size is beyond these limits, it will be clipped.
        @see setCurrentRangeStart, getCurrentRangeStart, getCurrentRangeSize
    */
    void setCurrentRange (double newStart,
                          double newSize);

    /** Moves the bar's thumb position.

        This will move the thumb position without changing the thumb size. Note
        that the maximum thumb start position is (getMaximumRangeLimit() - getCurrentRangeSize()).

        If this method call actually changes position, it will trigger a call to
        all the ChangeListeners that are registered with the scrollbar.

        @see setCurrentRange
    */
    void setCurrentRangeStart (double newStart);

    /** Returns the position of the top of the thumb.

        @see setCurrentRangeStart
    */
    double getCurrentRangeStart() const                 { return rangeStart; }

    /** Returns the current size of the thumb.

        @see setCurrentRange
    */
    double getCurrentRangeSize() const                  { return rangeSize; }

    //==============================================================================
    /** Sets the amount by which the up and down buttons will move the bar.

        The value here is in terms of the total range, and is added or subtracted
        from the thumb position when the user clicks an up/down (or left/right) button.
    */
    void setSingleStepSize (const double newSingleStepSize);

    /** Moves the scrollbar by a number of single-steps.

        This will move the bar by a multiple of its single-step interval (as
        specified using the setSingleStepSize() method).

        A positive value here will move the bar down or to the right, a negative
        value moves it up or to the left.
    */
    void moveScrollbarInSteps (const int howManySteps);

    /** Moves the scroll bar up or down in pages.

        This will move the bar by a multiple of its current thumb size, effectively
        doing a page-up or down.

        A positive value here will move the bar down or to the right, a negative
        value moves it up or to the left.
    */
    void moveScrollbarInPages (const int howManyPages);

    /** Scrolls to the top (or left).

        This is the same as calling setCurrentRangeStart (getMinimumRangeLimit());
    */
    void scrollToTop();

    /** Scrolls to the bottom (or right).

        This is the same as calling setCurrentRangeStart (getMaximumRangeLimit() - getCurrentRangeSize());
    */
    void scrollToBottom();


    //==============================================================================
    /** @internal */
    void keyPressed (const KeyPress& key);
    /** @internal */
    void mouseWheelMove (const MouseEvent& e, float wheelIncrement);

    juce_UseDebuggingNewOperator

private:
    //==============================================================================
    double minimum, maximum;
    double rangeStart, rangeSize;
    double singleStepSize, dragStartRange;
    int thumbAreaStart, thumbAreaSize, thumbStart, thumbSize;
    int dragStartMousePos, lastMousePos;
    bool vertical, isDraggingThumb;
    Button* upButton;
    Button* downButton;

    void paint (Graphics& g);
    void resized();
    void updateThumbPosition();

    void mouseDown (const MouseEvent& e);
    void mouseDrag (const MouseEvent& e);
    void mouseUp   (const MouseEvent& e);
    void timerCallback();

    ScrollBar (const ScrollBar&);
    const ScrollBar& operator= (const ScrollBar&);
};


#endif   // __JUCE_SCROLLBAR_JUCEHEADER__
