/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../../../juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE


#include "juce_FileChooser.h"
#include "juce_WildcardFileFilter.h"
#include "juce_FileChooserDialogBox.h"
#include "../lookandfeel/juce_LookAndFeel.h"


//==============================================================================
FileChooser::FileChooser (const String& chooserBoxTitle,
                          const File& currentFileOrDirectory,
                          const String& fileFilters,
                          const bool useNativeDialogBox_)
    : title (chooserBoxTitle),
      filters (fileFilters),
      file (currentFileOrDirectory),
      useNativeDialogBox (useNativeDialogBox_)
{
#ifdef JUCE_LINUX
    useNativeDialogBox = false;
#endif

    if (fileFilters.trim().isEmpty())
        filters = T("*");
}

FileChooser::~FileChooser()
{
}

bool FileChooser::browseForFileToOpen (FilePreviewComponent* previewComponent)
{
    return showDialog (false, false, previewComponent);
}

bool FileChooser::browseForFileToSave()
{
    return showDialog (false, true, 0);
}

bool FileChooser::browseForDirectory()
{
    return showDialog (true, false, 0);
}

const File FileChooser::getResult() const
{
    return file;
}

bool FileChooser::showDialog (const bool isDirectory,
                              const bool isSave,
                              FilePreviewComponent* const previewComponent)
{
    ComponentDeletionWatcher* currentlyFocusedChecker = 0;
    Component* const currentlyFocused = Component::getCurrentlyFocusedComponent();

    if (currentlyFocused != 0)
        currentlyFocusedChecker = new ComponentDeletionWatcher (currentlyFocused);

    String result;

    // the preview component needs to be the right size before you pass it in here..
    jassert (previewComponent == 0 || (previewComponent->getWidth() > 10
                                        && previewComponent->getHeight() > 10));

#ifdef JUCE_WIN32
    if (useNativeDialogBox)
#else
    if (useNativeDialogBox && (previewComponent == 0))
#endif
    {
        result = showPlatformDialog (title, file, filters,
                                     isDirectory, isSave,
                                     previewComponent);
    }
    else
    {
        WildcardFileFilter wildcard (filters, String::empty);

        FileBrowserComponent browserComponent (isDirectory ? FileBrowserComponent::chooseDirectoryMode
                                                           : (isSave ? FileBrowserComponent::saveFileMode
                                                                     : FileBrowserComponent::loadFileMode),
                                               file, &wildcard, previewComponent);

        FileChooserDialogBox box (title, String::empty,
                                  browserComponent,
                                  browserComponent.getLookAndFeel().alertWindowBackground);

        if (box.show())
            result = browserComponent.getCurrentFile().getFullPathName();
    }

    if (currentlyFocused != 0 && ! currentlyFocusedChecker->hasBeenDeleted())
        currentlyFocused->grabKeyboardFocus();

    delete currentlyFocusedChecker;

    if (result.isEmpty())
    {
        return false;
    }
    else
    {
        file = result;
        return true;
    }
}

//==============================================================================
FilePreviewComponent::FilePreviewComponent()
{
}

FilePreviewComponent::~FilePreviewComponent()
{
}

END_JUCE_NAMESPACE
