/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_TEXTEDITOR_JUCEHEADER__
#define __JUCE_TEXTEDITOR_JUCEHEADER__

#include "../juce_Component.h"
#include "../../../events/juce_Timer.h"
#include "../../../events/juce_ActionListenerList.h"
#include "../../../documents/juce_UndoManager.h"
#include "../layout/juce_Viewport.h"
class TextEditor;
class TextHolderComponent;


//==============================================================================
/**
    Receives callbacks from a TextEditor component when it changes.

    @see TextEditor::addListener
*/
class JUCE_API  TextEditorListener
{
public:
    /** Destructor. */
    virtual ~TextEditorListener()  {}

    /** Called when the user changes the text in some way. */
    virtual void textEditorTextChanged (TextEditor& editor) = 0;

    /** Called when the user presses the return key. */
    virtual void textEditorReturnKeyPressed (TextEditor& editor) = 0;

    /** Called when the user presses the escape key. */
    virtual void textEditorEscapeKeyPressed (TextEditor& editor) = 0;

    /** Called when the text editor loses focus. */
    virtual void textEditorFocusLost (TextEditor& editor) = 0;
};


//==============================================================================
/**
    A component containing text that can be edited.

    A TextEditor can either be in single- or multi-line mode, and supports mixed
    fonts and colours.

    @see TextEditorListener, Label
*/
class JUCE_API  TextEditor  : public Component
{
public:
    //==============================================================================
    /** Creates a new, empty text editor.

        @param componentName        the name to pass to the component for it to use as its name
        @param passwordCharacter    if this is not zero, this character will be used as a replacement
                                    for all characters that are drawn on screen - e.g. to create
                                    a password-style textbox, you could set this to T('*')
    */
    TextEditor (const String& componentName = String::empty,
                const tchar passwordCharacter = 0);

    /** Destructor. */
    virtual ~TextEditor();


    //==============================================================================
    /** Puts the editor into either multi- or single-line mode.

        By default, the editor will be in single-line mode, so use this if you need a multi-line
        editor.

        See also the setReturnKeyStartsNewLine() method, which will also need to be turned
        on if you want a multi-line editor with line-breaks.

        @see isMultiLine, setReturnKeyStartsNewLine
    */
    void setMultiLine (const bool shouldBeMultiLine,
                       const bool shouldWordWrap = true);

    /** Returns true if the editor is in multi-line mode.
    */
    bool isMultiLine() const;

    /** Changes the behaviour of the return key.

        If set to true, the return key will insert a new-line into the text; if false
        it will trigger a call to the TextEditorListener::textEditorReturnKeyPressed()
        method. By default this is set to false, and when true it will only insert
        new-lines when in multi-line mode (see setMultiLine()).
    */
    void setReturnKeyStartsNewLine (const bool shouldStartNewLine);

    /** Returns the value set by setReturnKeyStartsNewLine().

        See setReturnKeyStartsNewLine() for more info.
    */
    bool getReturnKeyStartsNewLine() const;

    /** Changes the editor to read-only mode.

        By default, the text editor is not read-only. If you're making it read-only, you
        might also want to call setCaretVisible (false) to get rid of the caret.

        The text can still be highlighted and copied when in read-only mode.

        @see isReadOnly, setCaretVisible
    */
    void setReadOnly (const bool shouldBeReadOnly);

    /** Returns true if the editor is in read-only mode.
    */
    bool isReadOnly() const;

    /** Makes the caret visible or invisible.

        By default the caret is visible.

        @see setCaretColour, setCaretPosition
    */
    void setCaretVisible (bool shouldBeVisible);

    /** Changes the colour of the caret.

        By default it will be black, so you might need to change it if you're working with
        wacky colours.

        @see setCaretVisible, setCaretPosition
    */
    void setCaretColour (const Colour& colour);

    /** Enables/disables a vertical scrollbar.

        (This only applies when in multi-line mode). When the text gets too long to fit
        in the component, a scrollbar can appear to allow it to be scrolled. Even when
        this is enabled, the scrollbar will be hidden unless it's needed.

        By default the scrollbar is enabled.
    */
    void setScrollbarsShown (bool shouldBeEnabled);

    /** Allows a right-click menu to appear for the editor.

        (This defaults to being enabled).

        If enabled, right-clicking (or command-clicking on the Mac) will pop up a menu
        of options such as cut/copy/paste, undo/redo, etc.
    */
    void setPopupMenuEnabled (const bool menuEnabled);

    /** Sets the colours for the editor.

        @param currentTextColour    the colour that will be used when text is added to the editor. Note
                                    that because the editor can contain multiple colours, calling this
                                    method won't change the colour of existing text - to do that, call
                                    applyFontToAllText() after calling this method.
        @param backgroundColour     the colour to use for the text component's background - this can be
                                    transparent if necessary
        @param highlightColour      the colour with which to highlight sections of the text - this can
                                    be transparent if you don't want to show any highlighting.
        @param outlineColour        if this is non-transparent, it will be used to draw a box around
                                    the edge of the component
        @param dropShadowColour     If this is non-transparent, it'll be used to draw an inner shadow
                                    around the edge of the editor
    */
    void setColours (const Colour& currentTextColour,
                     const Colour& backgroundColour = Colours::white,
                     const Colour& highlightColour = Colour (0x401111ee),
                     const Colour& outlineColour = Colours::transparentBlack,
                     const Colour& dropShadowColour = Colours::black.withAlpha (0.22f));


    /** Returns the text colour, as set by setColours(). */
    const Colour& getTextColour() const throw()                     { return textColour; }

    /** Returns the background colour, as set by setColours(). */
    const Colour& getBackgroundColour() const throw()               { return backgroundColour; }

    /** Returns the highlight colour, as set by setColours(). */
    const Colour& getHighlightColour() const throw()                { return highlightColour; }

    /** Returns the outline colour, as set by setColours(). */
    const Colour& getOutlineColour() const throw()                  { return outlineColour; }

    /** Returns the shadow colour, as set by setColours(). */
    const Colour& getShadowColour() const throw()                   { return shadowColour; }

    //==============================================================================
    /** Sets the font to use for newly added text.

        This will change the font that will be used next time any text is added or entered
        into the editor. It won't change the font of any existing text - to do that, use
        applyFontToAllText() instead.

        @see applyFontToAllText
    */
    void setFont (const Font& newFont);

    /** Applies a font to all the text in the editor.

        This will also set the current font to use for any new text that's added.

        @see setFont
    */
    void applyFontToAllText (const Font& newFont);

    /** Returns the font that's currently being used for new text.

        @see setFont
    */
    const Font getFont() const;

    //==============================================================================
    /** If set to true, focusing on the editor will highlight all its text.

        (Set to false by default).

        This is useful for boxes where you expect the user to re-enter all the
        text when they focus on the component, rather than editing what's already there.
    */
    void setSelectAllWhenFocused (bool b);

    /** Sets limits on the characters that can be entered.

        @param maxTextLength        if this is > 0, it sets a maximum length limit; if 0, no
                                    limit is set
        @param allowedCharacters    if this is non-empty, then only characters that occur in
                                    this string are allowed to be entered into the editor.
    */
    void setInputRestrictions (const int maxTextLength,
                               const String& allowedCharacters = String::empty);

    /** When the text editor is empty, it can be set to display a message.

        This is handy for things like telling the user what to type in the box - the
        string is only displayed, it's not taken to actually be the contents of
        the editor.
    */
    void setTextToShowWhenEmpty (const String& text, const Colour& colourToUse);

    //==============================================================================
    /** Changes the size of the scrollbars that are used.

        Handy if you need smaller scrollbars for a small text box.
    */
    void setScrollBarThickness (const int newThicknessPixels);

    /** Shows or hides the buttons on any scrollbars that are used.

        @see ScrollBar::setButtonVisibility
    */
    void setScrollBarButtonVisibility (const bool buttonsVisible);

    //==============================================================================
    /** Registers a listener to be told when things happen to the text.

        @see removeListener
    */
    void addListener (TextEditorListener* const newListener);

    /** Deregisters a listener.

        @see addListener
    */
    void removeListener (TextEditorListener* const listenerToRemove);

    //==============================================================================
    /** Returns the entire contents of the editor. */
    const String getText() const;

    /** Returns a section of the contents of the editor. */
    const String getTextSubstring (const int startCharacter, const int endCharacter) const;

    /** Returns true if there are no characters in the editor.

        This is more efficient than calling getText().isEmpty().
    */
    bool isEmpty() const;

    /** Sets the entire content of the editor.

        This will clear the editor and insert the given text (using the current colour
        and font).

        @param newText                  the text to add
        @param sendTextChangeMessage    if true, this will cause a change message to
                                        be sent to all the listeners.
        @see insertText
    */
    void setText (const String& newText,
                  const bool sendTextChangeMessage = true);

    /** Inserts some text at the current cursor position.

        If a section of the text is highlighted, it will be replaced by
        this string, otherwise it will be inserted.

        To delete a section of text, you can use setHighlightedRegion() to
        highlight it, and call insertTextAtCursor (String::empty).

        @see setCaretPosition, getCaretPosition, setHighlightedRegion
    */
    void insertTextAtCursor (String textToInsert);

    /** Deletes all the text from the editor. */
    void clear();

    /** Deletes the currently selected region, and puts it on the clipboard.

        @see copy, paste, SystemClipboard
    */
    void cut();

    /** Copies any currently selected region to the clipboard.

        @see cut, paste, SystemClipboard
    */
    void copy();

    /** Pastes the contents of the clipboard into the editor at the cursor position.

        @see cut, copy, SystemClipboard
    */
    void paste();

    //==============================================================================
    /** Moves the caret to be in front of a given character.

        @see getCaretPosition
    */
    void setCaretPosition (const int newIndex);

    /** Returns the current index of the caret.

        @see setCaretPosition
    */
    int getCaretPosition() const;

    /** Selects a section of the text.
    */
    void setHighlightedRegion (int startIndex,
                               int numberOfCharactersToHighlight);

    /** Finds the index of the character at a given position.

        The co-ordinates are relative to the component's top-left.
    */
    int getTextIndexAt (const int x, const int y);

    /** Returns the total width of the text, as it is currently laid-out.

        This may be larger than the size of the TextEditor, and can change when
        the TextEditor is resized or the text changes.
    */
    int getTextWidth() const;

    /** Returns the maximum height of the text, as it is currently laid-out.

        This may be larger than the size of the TextEditor, and can change when
        the TextEditor is resized or the text changes.
    */
    int getTextHeight() const;

    /** Changes the size of the gap at the top and left-edge of the editor.

        By default there's a gap of 4 pixels.
    */
    void setIndents (const int newLeftIndent, const int newTopIndent);

    /** Changes the size of border left around the edge of the component.

        @see getBorder
    */
    void setBorder (const BorderSize& border);

    /** Returns the size of border around the edge of the component.

        @see setBorder
    */
    const BorderSize getBorder() const throw();

    //==============================================================================
    /** @internal */
    void paint (Graphics& g);
    /** @internal */
    void paintOverChildren (Graphics& g);
    /** @internal */
    void mouseDown (const MouseEvent& e);
    /** @internal */
    void mouseUp (const MouseEvent& e);
    /** @internal */
    void mouseDrag (const MouseEvent& e);
    /** @internal */
    void mouseDoubleClick (const MouseEvent& e);
    /** @internal */
    void mouseWheelMove (const MouseEvent& e, float increment);
    /** @internal */
    void keyPressed (const KeyPress& key);
    /** @internal */
    void keyStateChanged();
    /** @internal */
    void focusGained (FocusChangeType cause);
    /** @internal */
    void focusLost (FocusChangeType cause);
    /** @internal */
    void resized();
    /** @internal */
    void enablementChanged();

    juce_UseDebuggingNewOperator

protected:
    void scrollToMakeSureCursorIsVisible();
    void moveCaret (int newCaretPos);
    void moveCursorTo (const int newPosition, const bool isSelecting);
    void textChanged();
    void handleCommandMessage (int commandId);

    virtual void returnPressed();
    virtual void escapePressed();

private:
    Viewport* viewport;
    TextHolderComponent* textHolder;
    BorderSize borderSize;

    bool readOnly                   : 1;
    bool multiline                  : 1;
    bool wordWrap                   : 1;
    bool returnKeyStartsNewLine     : 1;
    bool caretVisible               : 1;
    bool popupMenuEnabled           : 1;
    bool selectAllTextWhenFocused   : 1;
    bool scrollbarVisible           : 1;
    bool menuVisible                : 1;
    bool wasFocused                 : 1;
    bool caretFlashState            : 1;

    UndoManager undoManager;
    float cursorX, cursorY, cursorHeight;
    int maxTextLength;
    int selectionStart, selectionEnd;
    int leftIndent, topIndent;
    unsigned int lastTransactionTime;
    Colour backgroundColour, textColour, highlightColour, caretColour, outlineColour, shadowColour;
    Font currentFont;
    int totalNumChars, caretPosition;
    VoidArray sections;
    String textToShowWhenEmpty;
    Colour colourForTextWhenEmpty;

    const tchar passwordCharacter;

    enum
    {
        notDragging,
        draggingSelectionStart,
        draggingSelectionEnd
    } dragType;

    String allowedCharacters;
    VoidArray listeners;

    friend class TextEditorInsertAction;
    friend class TextEditorRemoveAction;

    void coalesceSimilarSections();
    void splitSection (const int sectionIndex, const int charToSplitAt);

    void clearInternal (UndoManager* const um);

    void insert (const String& text,
                 const int insertIndex,
                 const Font& font,
                 const Colour& colour,
                 UndoManager* const um,
                 const int caretPositionToMoveTo);

    void reinsert (const int insertIndex,
                   const VoidArray sections);

    void remove (int startIndex,
                 int endIndex,
                 UndoManager* const um,
                 const int caretPositionToMoveTo);

    int getTotalNumChars();

    void getCharPosition (const int index,
                          float& x, float& y,
                          float& lineHeight);

    int indexAtPosition (const float x,
                         const float y);

    int findWordBreakAfter (int position) const;
    int findWordBreakBefore (int position) const;

    void newTransaction();
    void doUndoRedo (const bool isRedo);

    friend class TextHolderComponent;
    friend class TextEditorViewport;
    void drawContent (Graphics& g);
    void updateTextHolderSize();
    float getWordWrapWidth() const throw();
    void timerCallbackInt();
    void repaintCaret();
    void repaintText (int textStartIndex, int textEndIndex);

    TextEditor (const TextEditor&);
    const TextEditor& operator= (const TextEditor&);
};

#endif   // __JUCE_TEXTEDITOR_JUCEHEADER__
