/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#ifndef __JUCE_TEXTBUTTON_JUCEHEADER__
#define __JUCE_TEXTBUTTON_JUCEHEADER__

#include "juce_Button.h"


//==============================================================================
/**
    A button that uses the standard lozenge-shaped background with a line of
    text on it.

    @see Button, DrawableButton
*/
class JUCE_API  TextButton  : public Button
{
public:
    //==============================================================================
    /** Creates a TextButton.

        @param buttonName           the text to put in the button (the component's name is also
                                    initially set to this string, but these can be changed later
                                    using the setName() and setButtonText() methods)
        @param toolTip              an optional string to use as a toolip

        @see Button
    */
    TextButton (const String& buttonName,
                const String& toolTip = String::empty);

    /** Destructor. */
    ~TextButton();

    //==============================================================================
    /** Changes the button's background colours.

        The toggledOffColour is the colour to use when the button's toggle state
        is off, and toggledOnColour when it's on.

        Exactly how this is used is up to the look-and-feel class.
    */
    void setBackgroundColours (const Colour& toggledOffColour,
                               const Colour& toggledOnColour);

    /** Returns the button's background colour.

        If whenToggledOn is true, it returns the colour used when the button is in
        a toggled-'on' state; if it's false, it returns the normal, 'off' colour.

        @see setBackgroundColours
    */
    const Colour& getBackgroundColour (const bool whenToggledOn) const throw();

    /** Sets a text colour you'd like to use for this button.

        The default colour is black.
    */
    void setTextColour (const Colour& newTextColour);

    /** Returns the text colour that was set with setTextColour(). */
    const Colour& getTextColour() const throw()                         { return textColour; }

    //==============================================================================
    /** Resizes the button to fit neatly around its current text.

        If newHeight is >= 0, the button's height will be changed to this
        value. If it's less than zero, its height will be unaffected.
    */
    void changeWidthToFitText (const int newHeight = -1);

    /** This can be overridden to use different fonts than the default one.

        Note that you'll need to set the font's size appropriately, too.
    */
    virtual const Font getFont();


    //==============================================================================
    juce_UseDebuggingNewOperator

protected:
    /** @internal */
    void paintButton (Graphics& g, bool isMouseOverButton, bool isButtonDown);

private:
    Colour backgroundOff, backgroundOn, textColour;

    TextButton (const TextButton&);
    const TextButton& operator= (const TextButton&);
};


#endif   // __JUCE_TEXTBUTTON_JUCEHEADER__
