/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../../../juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE

#include "juce_Button.h"
#include "../juce_ComponentDeletionWatcher.h"


//==============================================================================
Button::Button (const String& name)
  : Component (name),
    buttonState (buttonNormal),
    shortcuts (2),
    keySource (0),
    autoRepeatDelay (-1),
    autoRepeatSpeed (0),
    autoRepeatMinimumDelay (-1),
    text (name),
    buttonListeners (2),
    repeatTimer (0),
    buttonPressTime (0),
    lastTimeCallbackTime (0),
    radioGroupId (0),
    keyMappingSetToTrigger (0),
    keyMappingCommandUID (0),
    isOn (false),
    clickTogglesState (false),
    needsToRelease (false),
    needsRepainting (false),
    isKeyDown (false),
    triggerOnMouseDown (false),
    connectedEdgeFlags (0)
{
    setFocusOrder (128);
}

Button::~Button()
{
    if (repeatTimer != 0)
        delete repeatTimer;

    clearShortcuts();
}

//==============================================================================
void Button::setButtonText (const String& newText)
{
    if (text != newText)
    {
        text = newText;
        repaint();
    }
}

void Button::setTooltip (const String& t)
{
    tooltip = t;
    generateTooltip = false;
}

const String Button::getTooltip()
{
    if (generateTooltip && keyMappingSetToTrigger != 0 && keyMappingCommandUID != 0)
    {
        String tt (keyMappingSetToTrigger->getDescriptionOfCommand (keyMappingCommandUID));

        OwnedArray <KeyPress> keyPresses;
        keyMappingSetToTrigger->getKeyPressesAssignedToCommand (keyMappingCommandUID, keyPresses);

        for (int i = 0; i < keyPresses.size(); ++i)
        {
            String key (keyPresses.getUnchecked(i)->getTextDescription());

            if (key.length() == 1)
                tt << T(" [shortcut: '") << key << T("']");
            else
                tt << T(" [") << key << T(']');
        }

        return tt;
    }

    return tooltip;
}

void Button::setConnectedEdges (const int connectedEdgeFlags_)
{
    connectedEdgeFlags = connectedEdgeFlags_;
}

//==============================================================================
void Button::setToggleState (const bool shouldBeOn,
                             const bool sendChangeNotification)
{
    if (shouldBeOn != isOn)
    {
        const ComponentDeletionWatcher deletionWatcher (this);

        isOn = shouldBeOn;
        repaint();

        if (sendChangeNotification)
            sendClickMessage (ModifierKeys());

        if ((! deletionWatcher.hasBeenDeleted()) && isOn)
            turnOffOtherButtonsInGroup (sendChangeNotification);
    }
}

void Button::setClickingTogglesState (const bool shouldToggle)
{
    clickTogglesState = shouldToggle;
}

bool Button::getClickingTogglesState() const throw()
{
    return clickTogglesState;
}

void Button::setRadioGroupId (const int newGroupId)
{
    if (radioGroupId != newGroupId)
    {
        radioGroupId = newGroupId;

        if (isOn)
            turnOffOtherButtonsInGroup (true);
    }
}

void Button::turnOffOtherButtonsInGroup (const bool sendChangeNotification)
{
    Component* const p = getParentComponent();

    if (p != 0 && radioGroupId != 0)
    {
        const ComponentDeletionWatcher deletionWatcher (this);

        for (int i = p->getNumChildComponents(); --i >= 0;)
        {
            Component* const c = p->getChildComponent (i);

            if (c != this)
            {
                Button* const b = dynamic_cast<Button*> (c);

                if (b != 0 && b->getRadioGroupId() == radioGroupId)
                {
                    b->setToggleState (false, sendChangeNotification);

                    if (deletionWatcher.hasBeenDeleted())
                        return;
                }
            }
        }
    }
}

//==============================================================================
void Button::enablementChanged()
{
    setState (workOutState (isMouseOver(),
                            isMouseButtonDown()));

    repaint();
}

Button::ButtonState Button::workOutState (const bool over,
                                          const bool down) throw()
{
    if (isEnabled() && ! isCurrentlyBlockedByAnotherModalComponent())
    {
        if ((down && (over || triggerOnMouseDown)) || isKeyDown)
            return buttonDown;
        else if (over)
            return buttonOver;
    }

    return buttonNormal;
}

void Button::setState (const ButtonState newState)
{
    if (buttonState != newState)
    {
        buttonState = newState;
        repaint();

        if (buttonState == buttonDown)
        {
            buttonPressTime = Time::getApproximateMillisecondCounter();
            lastTimeCallbackTime = buttonPressTime;
        }

        sendStateMessage();
    }
}

bool Button::isDown() const throw()
{
    return buttonState == buttonDown;
}

bool Button::isOver() const throw()
{
    return buttonState != buttonNormal;
}

void Button::buttonStateChanged()
{
}

uint32 Button::getMillisecondsSinceButtonDown() const throw()
{
    const uint32 now = Time::getApproximateMillisecondCounter();
    return now > buttonPressTime ? now - buttonPressTime : 0;
}

void Button::setTriggeredOnMouseDown (const bool isTriggeredOnMouseDown) throw()
{
    triggerOnMouseDown = isTriggeredOnMouseDown;
}

//==============================================================================
void Button::clicked()
{
}

void Button::clicked (const ModifierKeys& /*modifiers*/)
{
    clicked();
}

static const int clickMessageId = 0x2f3f4f99;

void Button::triggerClick()
{
    postCommandMessage (clickMessageId);
}

void Button::internalClickCallback (const ModifierKeys& modifiers)
{
    if (clickTogglesState)
        setToggleState ((radioGroupId != 0) || ! isOn, false);

    sendClickMessage (modifiers);
}

void Button::handleCommandMessage (int commandId)
{
    if (commandId == clickMessageId)
    {
        if (isEnabled())
        {
            needsToRelease = true;
            setState (buttonDown);
            getRepeatTimer().startTimer (100);
            internalClickCallback (ModifierKeys::getCurrentModifiers());
        }
    }
    else
    {
        Component::handleCommandMessage (commandId);
    }
}

//==============================================================================
void Button::addButtonListener (ButtonListener* const newListener)
{
    jassert (newListener != 0);
    jassert (! buttonListeners.contains (newListener)); // trying to add a listener to the list twice!

    if (newListener != 0)
        buttonListeners.add (newListener);
}

void Button::removeButtonListener (ButtonListener* const listener)
{
    jassert (buttonListeners.contains (listener)); // trying to remove a listener that isn't on the list!

    buttonListeners.removeValue (listener);
}

void Button::sendClickMessage (const ModifierKeys& modifiers)
{
    const ComponentDeletionWatcher cdw (this);

    if (keyMappingSetToTrigger != 0 && keyMappingCommandUID != 0)
        keyMappingSetToTrigger->invokeCommand (keyMappingCommandUID, true);

    clicked (modifiers);

    if (! cdw.hasBeenDeleted())
    {
        for (int i = buttonListeners.size(); --i >= 0;)
        {
            ButtonListener* const bl = (ButtonListener*) buttonListeners[i];

            if (bl != 0)
            {
                bl->buttonClicked (this);

                if (cdw.hasBeenDeleted())
                    return;
            }
        }
    }
}

void Button::sendStateMessage()
{
    const ComponentDeletionWatcher cdw (this);

    buttonStateChanged();

    if (cdw.hasBeenDeleted())
        return;

    for (int i = buttonListeners.size(); --i >= 0;)
    {
        ButtonListener* const bl = (ButtonListener*) buttonListeners[i];

        if (bl != 0)
        {
            bl->buttonStateChanged (this);

            if (cdw.hasBeenDeleted())
                return;
        }
    }
}

//==============================================================================
void Button::paint (Graphics& g)
{
    if (needsToRelease && isEnabled())
    {
        needsToRelease = false;
        needsRepainting = true;
    }

    paintButton (g, isOver(), isDown());
}

//==============================================================================
void Button::mouseEnter (const MouseEvent&)
{
    setState (workOutState (true, false));
}

void Button::mouseExit (const MouseEvent&)
{
    setState (workOutState (false, false));
}

void Button::focusGained (FocusChangeType)
{
    setState (workOutState (isMouseOver(), isMouseButtonDown()));
    repaint();
}

void Button::focusLost (FocusChangeType)
{
    setState (workOutState (isMouseOver(), isMouseButtonDown()));
    repaint();
}

void Button::mouseDown (const MouseEvent& e)
{
    if (autoRepeatDelay >= 0)
        getRepeatTimer().startTimer (autoRepeatDelay);

    setState (workOutState (true, true));

    if (isEnabled() && triggerOnMouseDown)
        internalClickCallback (e.mods);
}

void Button::mouseUp (const MouseEvent& e)
{
    setState (workOutState (isMouseOver(), false));

    if (isEnabled() && isMouseOver() && ! triggerOnMouseDown)
        internalClickCallback (e.mods);
}

void Button::mouseDrag (const MouseEvent&)
{
    const ButtonState oldState = buttonState;
    setState (workOutState (isMouseOver(), true));

    if (buttonState != oldState)
    {
        if ((autoRepeatDelay >= 0) && isDown())
            getRepeatTimer().startTimer (autoRepeatSpeed);
    }
}

//==============================================================================
void Button::setVisible (bool shouldBeVisible)
{
    if (shouldBeVisible != isVisible())
    {
        Component::setVisible (shouldBeVisible);

        if (! shouldBeVisible)
        {
            needsToRelease = false;
            setState (workOutState (false, false));
        }
        else
        {
            setState (workOutState (isMouseOver(), isMouseButtonDown()));
        }
    }
    else
    {
        Component::setVisible (shouldBeVisible);
    }
}

void Button::parentHierarchyChanged()
{
    Component* const newKeySource = (shortcuts.size() == 0) ? 0 : getTopLevelComponent();

    if (newKeySource != keySource)
    {
        if (keySource->isValidComponent())
            keySource->removeKeyListener (this);

        keySource = newKeySource;

        if (keySource->isValidComponent())
            keySource->addKeyListener (this);
    }
}

//==============================================================================
void Button::setKeyPressToTrigger (KeyPressMappingSet* const keyMappingSetToTrigger_,
                                   const int keyMappingCommandUID_,
                                   const bool generateTooltip_)
{
    keyMappingSetToTrigger = keyMappingSetToTrigger_;
    keyMappingCommandUID = keyMappingCommandUID_;
    generateTooltip = generateTooltip_;
}

void Button::addShortcut (const KeyPress& key)
{
    if (key.isValid())
    {
        jassert (! isRegisteredForShortcut (key));  // already registered!

        shortcuts.add (new KeyPress (key));
        parentHierarchyChanged();
    }
}

void Button::clearShortcuts()
{
    for (int i = shortcuts.size(); --i >= 0;)
    {
        KeyPress* const k = (KeyPress*) shortcuts.getUnchecked(i);
        delete k;
    }

    shortcuts.clear();

    parentHierarchyChanged();
}

bool Button::isShortcutPressed() const
{
    for (int i = shortcuts.size(); --i >= 0;)
    {
        const KeyPress* const shortcut = (KeyPress*) shortcuts.getUnchecked(i);

        if (shortcut->isCurrentlyDown())
            return true;
    }

    return false;
}

bool Button::isRegisteredForShortcut (const KeyPress& key) const throw()
{
    for (int i = shortcuts.size(); --i >= 0;)
    {
        const KeyPress* const shortcut = (KeyPress*) shortcuts.getUnchecked(i);

        if (key == *shortcut)
            return true;
    }

    return false;
}

void Button::keyStateChanged (Component*)
{
    if (isEnabled())
    {
        const bool wasDown = isKeyDown;
        isKeyDown = isShortcutPressed();

        if (autoRepeatDelay >= 0 && (isKeyDown && !wasDown))
            getRepeatTimer().startTimer (autoRepeatDelay);

        setState (workOutState (isMouseOver(), isMouseButtonDown()));

        if (isEnabled() && wasDown && ! isKeyDown)
            internalClickCallback (ModifierKeys::getCurrentModifiers());
    }
}

void Button::keyPressed (const KeyPress&, Component*)
{
    // (not actually used - the key detection happens in keyStateChanged)
}

void Button::keyPressed (const KeyPress& key)
{
    if (isEnabled() && key.isKeyCode (KeyPress::returnKey))
        triggerClick();
    else
        Component::keyPressed (key);
}

//==============================================================================
void Button::setRepeatSpeed (const int initialDelayMillisecs,
                             const int repeatMillisecs,
                             const int minimumDelayInMillisecs) throw()
{
    autoRepeatDelay = initialDelayMillisecs;
    autoRepeatSpeed = repeatMillisecs;
    autoRepeatMinimumDelay = jmin (autoRepeatSpeed, minimumDelayInMillisecs);
}

void Button::repeatTimerCallback() throw()
{
    if (needsRepainting)
    {
        getRepeatTimer().stopTimer();
        setState (workOutState (isMouseOver(), isMouseButtonDown()));
        needsRepainting = false;
    }
    else if ((isMouseButtonDown() && isMouseOver()) || isKeyDown)
    {
        int repeatSpeed = autoRepeatSpeed;

        if (autoRepeatMinimumDelay >= 0)
        {
            double timeHeldDown = jmin (1.0, getMillisecondsSinceButtonDown() / 4000.0);
            timeHeldDown *= timeHeldDown;

            repeatSpeed = repeatSpeed + (int) (timeHeldDown * (autoRepeatMinimumDelay - repeatSpeed));
        }

        repeatSpeed = jmax (1, repeatSpeed);

        getRepeatTimer().startTimer (repeatSpeed);

        const uint32 now = Time::getApproximateMillisecondCounter();
        const int numTimesToCallback
            = (now > lastTimeCallbackTime) ? jmax (1, (now - lastTimeCallbackTime) / repeatSpeed) : 1;

        lastTimeCallbackTime = now;

        const ComponentDeletionWatcher cdw (this);

        for (int i = numTimesToCallback; --i >= 0;)
        {
            internalClickCallback (ModifierKeys::getCurrentModifiers());

            if (cdw.hasBeenDeleted() || ! isDown())
                return;
        }
    }
    else if (! needsToRelease)
    {
        getRepeatTimer().stopTimer();
    }
}

class InternalButtonRepeatTimer  : public Timer
{
public:
    InternalButtonRepeatTimer (Button& owner_)
        : owner (owner_)
    {
    }

    ~InternalButtonRepeatTimer()
    {
    }

    void timerCallback()
    {
        owner.repeatTimerCallback();
    }

private:
    Button& owner;

    InternalButtonRepeatTimer (const InternalButtonRepeatTimer&);
    const InternalButtonRepeatTimer& operator= (const InternalButtonRepeatTimer&);
};

Timer& Button::getRepeatTimer() throw()
{
    if (repeatTimer == 0)
        repeatTimer = new InternalButtonRepeatTimer (*this);

    return *repeatTimer;
}

END_JUCE_NAMESPACE
