/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE


#include "juce_UndoManager.h"
#include "../application/juce_Application.h"


//==============================================================================
UndoManager::UndoManager (const int maxNumberOfUnitsToKeep,
                          const int minimumTransactions)
   : nextIndex (0),
     newTransaction (true),
     reentrancyCheck (false)
{
    setMaxNumberOfStoredUnits (maxNumberOfUnitsToKeep,
                               minimumTransactions);
}

UndoManager::~UndoManager()
{
    clearUndoHistory();
}

//==============================================================================
void UndoManager::clearUndoHistory()
{
    transactions.clear();
    transactionNames.clear();
    nextIndex = 0;
    sendChangeMessage (this);
}

int UndoManager::getNumberOfUnitsTakenUpByStoredCommands()
{
    int total = 0;

    for (int i = transactions.size(); --i >= 0;)
        for (int j = transactions.getUnchecked(i)->size(); --j >= 0;)
            total += transactions.getUnchecked(i)->getUnchecked(j)->getSizeInUnits();

    return total;
}

void UndoManager::setMaxNumberOfStoredUnits (const int maxNumberOfUnitsToKeep,
                                             const int minimumTransactions)
{
    maxNumUnitsToKeep          = jmax (1, maxNumberOfUnitsToKeep);
    minimumTransactionsToKeep  = jmax (1, minimumTransactions);
}

//==============================================================================
bool UndoManager::perform (UndoableAction* const command)
{
    if (command != 0)
    {
        if (reentrancyCheck)
        {
            jassertfalse    // don't call perform() recursively from the UndoableAction::perform() or
                            // undo() methods, or else these actions won't actually get done.

            return false;
        }
        else
        {
            bool success = false;

            JUCE_TRY
            {
                success = command->perform();
            }
            JUCE_CATCH_EXCEPTION

            jassert (success);
            if (success)
            {
                if (nextIndex > 0 && ! newTransaction)
                {
                    OwnedArray<UndoableAction>* commandSet = transactions [nextIndex - 1];

                    jassert (commandSet != 0);
                    if (commandSet == 0)
                        return false;

                    commandSet->add (command);
                }
                else
                {
                    OwnedArray<UndoableAction>* commandSet = new OwnedArray<UndoableAction>();
                    commandSet->add (command);
                    transactions.insert (nextIndex, commandSet);
                    transactionNames.insert (nextIndex, currentTransactionName);
                    ++nextIndex;
                }

                newTransaction = false;
            }

            while (nextIndex < transactions.size())
            {
                transactions.removeLast();
                transactionNames.remove (transactionNames.size() - 1);
            }

            while (nextIndex > 0
                   && getNumberOfUnitsTakenUpByStoredCommands() > maxNumUnitsToKeep
                   && transactions.size() > minimumTransactionsToKeep)
            {
                transactions.remove (0);
                transactionNames.remove (0);
                --nextIndex;
            }

            sendChangeMessage (this);

            return success;
        }
    }

    return false;
}

void UndoManager::beginNewTransaction (const String& actionName)
{
    newTransaction = true;
    currentTransactionName = actionName;
}

//==============================================================================
bool UndoManager::canUndo() const
{
    return nextIndex > 0;
}

bool UndoManager::canRedo() const
{
    return nextIndex < transactions.size();
}

const String UndoManager::getUndoDescription() const
{
    return transactionNames [nextIndex - 1];
}

const String UndoManager::getRedoDescription() const
{
    return transactionNames [nextIndex];
}

bool UndoManager::undo()
{
    const OwnedArray<UndoableAction>* const commandSet = transactions [nextIndex - 1];

    if (commandSet == 0)
        return false;

    reentrancyCheck = true;
    bool failed = false;

    for (int i = commandSet->size(); --i >= 0;)
    {
        if (! commandSet->getUnchecked(i)->undo())
        {
            jassertfalse
            failed = true;
            break;
        }
    }

    reentrancyCheck = false;

    if (failed)
    {
        clearUndoHistory();
    }
    else
    {
        --nextIndex;
    }

    beginNewTransaction();

    sendChangeMessage (this);
    return true;
}

bool UndoManager::redo()
{
    const OwnedArray<UndoableAction>* const commandSet = transactions [nextIndex];

    if (commandSet == 0)
        return false;

    reentrancyCheck = true;
    bool failed = false;

    for (int i = 0; i < commandSet->size(); ++i)
    {
        if (! commandSet->getUnchecked(i)->perform())
        {
            jassertfalse
            failed = true;
            break;
        }
    }

    reentrancyCheck = false;

    if (failed)
    {
        clearUndoHistory();
    }
    else
    {
        ++nextIndex;
    }

    beginNewTransaction();

    sendChangeMessage (this);
    return true;
}

bool UndoManager::undoCurrentTransactionOnly()
{
    return newTransaction ? false
                          : undo();
}

END_JUCE_NAMESPACE
