/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-6 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#include "../../../juce_core/basics/juce_StandardHeader.h"

BEGIN_JUCE_NAMESPACE

#include "juce_AudioFormatManager.h"
#include "juce_AiffAudioFormat.h"
#include "juce_WavAudioFormat.h"
#include "juce_FlacAudioFormat.h"
#include "juce_OggVorbisAudioFormat.h"
#include "../../../juce_core/io/files/juce_FileInputStream.h"


//==============================================================================
AudioFormatManager::AudioFormatManager()
    : knownFormats (4),
      defaultFormatIndex (0)
{
}

AudioFormatManager::~AudioFormatManager()
{
    clearFormats();
    clearSingletonInstance();
}

juce_ImplementSingleton (AudioFormatManager);


//==============================================================================
void AudioFormatManager::registerFormat (AudioFormat* newFormat,
                                         const bool makeThisTheDefaultFormat)
{
    jassert (newFormat != 0);

    if (newFormat != 0)
    {
#ifdef JUCE_DEBUG
        for (int i = getNumKnownFormats(); --i >= 0;)
        {
            if (getKnownFormat (i)->getFormatName() == newFormat->getFormatName())
            {
                jassertfalse // trying to add the same format twice!
            }
        }
#endif

        if (makeThisTheDefaultFormat)
            defaultFormatIndex = knownFormats.size();

        knownFormats.add (newFormat);
    }
}

void AudioFormatManager::registerBasicFormats()
{
#ifdef JUCE_MAC
    registerFormat (new AiffAudioFormat(), true);
    registerFormat (new WavAudioFormat(), false);
#else
    registerFormat (new WavAudioFormat(), true);
    registerFormat (new AiffAudioFormat(), false);
#endif

#if JUCE_USE_FLAC
    registerFormat (new FlacAudioFormat(), false);
#endif

#if JUCE_USE_OGGVORBIS
    registerFormat (new OggVorbisAudioFormat(), false);
#endif
}

void AudioFormatManager::clearFormats()
{
    for (int i = getNumKnownFormats(); --i >= 0;)
    {
        AudioFormat* const af = getKnownFormat(i);
        delete af;
    }

    knownFormats.clear();
    defaultFormatIndex = 0;
}

int AudioFormatManager::getNumKnownFormats() const
{
    return knownFormats.size();
}

AudioFormat* AudioFormatManager::getKnownFormat (const int index) const
{
    return (AudioFormat*) knownFormats [index];
}

AudioFormat* AudioFormatManager::getDefaultFormat() const
{
    return getKnownFormat (defaultFormatIndex);
}

const String AudioFormatManager::getWildcardForAllFormats() const
{
    StringArray allExtensions;

    int i;
    for (i = 0; i < getNumKnownFormats(); ++i)
        allExtensions.addArray (getKnownFormat (i)->getFileExtensions());

    allExtensions.trim();
    allExtensions.removeEmptyStrings();

    String s;
    for (i = 0; i < allExtensions.size(); ++i)
    {
        s << T('*');

        if (! allExtensions[i].startsWithChar (T('.')))
            s << T('.');

        s << allExtensions[i];

        if (i < allExtensions.size() - 1)
            s << T(';');
    }

    return s;
}

//==============================================================================
AudioFormatReader* AudioFormatManager::createReaderFor (const File& file)
{
    // you need to actually register some formats before the manager can
    // use them to open a file!
    jassert (knownFormats.size() > 0);

    for (int i = 0; i < getNumKnownFormats(); ++i)
    {
        AudioFormat* const af = getKnownFormat(i);

        if (af->canHandleFile (file))
        {
            InputStream* const in = file.createInputStream();

            if (in != 0)
            {
                AudioFormatReader* const r = af->createReaderFor (in);

                if (r != 0)
                    return r;
            }
        }
    }

    return 0;
}


END_JUCE_NAMESPACE
