#include "zeerecipeviewwindow.h"

#include "zeebeamdialog.h"
#include "zeehtmlexporter.h"
#include "zeemealmasterexporter.h"
#include "zeerecipeeditwindow.h"

#include "sql/qsqlquery.h"

#include <qpe/ir.h>
#include <qpe/resource.h>

#include <qfile.h>
#include <qframe.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qtextview.h>
#include <qtoolbutton.h>

#include <sys/stat.h>
#include <unistd.h>

ZeeRecipeViewWindow::ZeeRecipeViewWindow(int recipeId, QWidget* parent, const char* name)
    : ZeeModalWindow(parent, name)
{
    m_recipeId = recipeId;

    setCaption(tr("Zee Cookbook - View Recipe"));

    m_pEditRecipeButton = new QToolButton(this);
    m_pEditRecipeButton->setAutoRaise(true);
    m_pEditRecipeButton->setIconSet(Resource::loadIconSet("zeecookbook/edit"));
    connect(m_pEditRecipeButton, SIGNAL(clicked()), this, SLOT(editRecipe()));

    m_pRemoveRecipeButton = new QToolButton(this);
    m_pRemoveRecipeButton->setAutoRaise(true);
    m_pRemoveRecipeButton->setIconSet(Resource::loadIconSet("zeecookbook/remove"));
    connect(m_pRemoveRecipeButton, SIGNAL(clicked()), this, SLOT(removeRecipe()));

    QFrame* pButtonSeparator = new QFrame(this);
    pButtonSeparator->setFrameStyle(QFrame::VLine | QFrame::Sunken);

    m_pBeamRecipeButton = new QToolButton(this);
    m_pBeamRecipeButton->setAutoRaise(true);
    m_pBeamRecipeButton->setIconSet(Resource::loadIconSet("zeecookbook/beam"));
    connect(m_pBeamRecipeButton, SIGNAL(clicked()), this, SLOT(beamRecipe()));

    m_pTextView = new QTextView(this);

    QVBoxLayout* mainLayout = new QVBoxLayout(this, 4, 4);
    QHBoxLayout* buttonLayout = new QHBoxLayout(0);

    buttonLayout->addWidget(m_pEditRecipeButton);
    buttonLayout->addWidget(m_pRemoveRecipeButton);
    buttonLayout->addWidget(pButtonSeparator);
    buttonLayout->addWidget(m_pBeamRecipeButton);
    buttonLayout->addStretch();

    mainLayout->addLayout(buttonLayout);
    mainLayout->addWidget(m_pTextView);

    updateView();
}

ZeeRecipeViewWindow::~ZeeRecipeViewWindow()
{
}

void ZeeRecipeViewWindow::updateView()
{
    QString text;

    QString queryString = "SELECT name, yield_quantity, yield_id, direction";
    queryString += " FROM recipe WHERE id = %1";
    QSqlQuery query(queryString.arg(m_recipeId));

    QString name = "";
    QString yieldQuantity = "";
    int yieldId = -1;
    QString direction = "";

    if (query.first()) {
        name = query.value(0).toString();
        yieldQuantity = query.value(1).toString();
        yieldId = query.value(2).toInt();
        direction = query.value(3).toString();
    }

    QString yieldUnit = "";

    if (yieldId != -1) {
        queryString = "SELECT name FROM yield WHERE id = %1";
        query.exec(queryString.arg(yieldId));

        if (query.first()) {
            yieldUnit = query.value(0).toString();
        }
    }

    queryString = "SELECT name FROM recipe_category, category";
    queryString += " WHERE recipe_id = %1 AND category_id = category.id ORDER BY position";
    query.exec(queryString.arg(m_recipeId));

    QStringList categoryList;

    while (query.next()) {
        QString name = query.value(0).toString();
        categoryList.append(name);
    }

    text += "<center><h2>" + name + "</h2></center>";
    text += "<h3>" + tr("Categories:") + "</h3> " + categoryList.join(", ") + "<br>";
    text += "<h3>" + tr("Yield:") + "</h3> " + yieldQuantity + " " + yieldUnit + "<br>";
    text += "<h3>" + tr("Ingredients:") + "</h3><ul>";

    queryString = "SELECT quantity, unit.name, ingredient.name, is_group";
    queryString += " FROM recipe_ingredient LEFT JOIN unit ON unit_id = unit.id, ingredient";
    queryString += " WHERE recipe_id = %1 AND ingredient_id = ingredient.id ORDER BY position";
    query.exec(queryString.arg(m_recipeId));

    while (query.next()) {
        QString quantity = query.value(0).toString();
        QString unit = query.value(1).toString();
        QString name = query.value(2).toString();
        bool isGroup = query.value(3).toInt();

        if (isGroup) {
            text += "<b>" + quantity + " " + unit + " " + name + "</b>";
        } else {
            text += "<li>" + quantity + " " + unit + " " + name + "</li>";
        }
    }

    text += "</ul><h3>" + tr("Directions:") + "</h3>" + direction.replace(QRegExp("\n"), "<br>");

    m_pTextView->setText(text);
}

void ZeeRecipeViewWindow::editRecipe()
{
    ZeeRecipeEditWindow recipeEditWindow(m_recipeId, this);

    recipeEditWindow.showMaximized();
    recipeEditWindow.exec();

    updateView();
}

void ZeeRecipeViewWindow::removeRecipe()
{
    QMessageBox messageBox(tr("Remove Recipe"), tr("Do you want to remove\nthe selected recipe?"),
                        QMessageBox::Information, QMessageBox::Yes,
                        QMessageBox::No | QMessageBox::Escape | QMessageBox::Default,
                        QMessageBox::NoButton, this);

    messageBox.setIconPixmap(Resource::loadPixmap("zeecookbook/information"));
    messageBox.setButtonText(QMessageBox::Yes, tr("Yes"));
    messageBox.setButtonText(QMessageBox::No, tr("No"));

    if (messageBox.exec() == QMessageBox::Yes) {
        QString queryString = "DELETE FROM recipe WHERE id = %1";
        QSqlQuery query(queryString.arg(m_recipeId));

        queryString = "DELETE FROM recipe_category WHERE recipe_id = %1";
        query.exec(queryString.arg(m_recipeId));

        queryString = "DELETE FROM recipe_ingredient WHERE recipe_id = %1";
        query.exec(queryString.arg(m_recipeId));

        close();
    }
}

static const char* beamFileName = "/tmp/obex/zeecookbook.dat";

void ZeeRecipeViewWindow::beamRecipe()
{
    ZeeBeamDialog beamDialog(this);

    if (beamDialog.exec() == QDialog::Accepted) {
        QString format = beamDialog.selectedFormat();

        ZeeExporter* exporter = 0;

        if (format == tr("HTML")) {
            exporter = new ZeeHtmlExporter();
        } else if (format == tr("Meal-Master")) {
            exporter = new ZeeMealMasterExporter();
        }

        QString name = "";

        QString queryString = "SELECT name FROM recipe WHERE id = %1";
        QSqlQuery query(queryString.arg(m_recipeId));

        if (query.first()) {
            name = query.value(0).toString();
        }

        QString mimeType = exporter->mimeType();

        unlink(beamFileName);
        mkdir("/tmp/obex/", 0755);

        QFile file(beamFileName);

        file.open(IO_WriteOnly);
        QTextStream textStream(&file);

        exporter->exportRecipe(m_recipeId, textStream);

        file.close();

        delete exporter;

        Ir* ir = new Ir(this);
        connect(ir, SIGNAL(done(Ir*)), this, SLOT(beamRecipeDone(Ir*)));
        ir->send(beamFileName, name, mimeType);
    }
}

void ZeeRecipeViewWindow::beamRecipeDone(Ir* ir)
{
    delete ir;
    unlink(beamFileName);
}
