#include "zeemealmasterimporter.h"

#include "zeeutil.h"

#include "sql/qsqlquery.h"

#include <qapplication.h>

ZeeMealMasterImporter::ZeeMealMasterImporter()
    : ZeeImporter()
{
    emptyRegExp = QRegExp("^\\s*$");
    headerRegExp1 = QRegExp("^-----.*Meal-Master", false);
    headerRegExp2 = QRegExp("^MMMMM.*Meal-Master", false);
    titleRegExp = QRegExp("^\\s*Title:\\s", false);
    categoryRegExp = QRegExp("^\\s*Categories:\\s", false);
    yieldRegExp1 = QRegExp("^\\s*Yield:\\s", false);
    yieldRegExp2 = QRegExp("^\\s*Servings:\\s", false);
    yieldQuantityRegExp = QRegExp("[0-9]+");
    ingredientRegExp1 = QRegExp("^.......\\s..\\s.+");
    ingredientRegExp2 = QRegExp("^.......\\s..\\s.............................\\s.......\\s..\\s.+");
    ingredientGroupRegExp1 = QRegExp("^------+.+-+$");
    ingredientGroupRegExp2 = QRegExp("^MMMMM-+.+-+$");
    trailerRegExp1 = QRegExp("^-----\\s*$");
    trailerRegExp2 = QRegExp("^MMMMM\\s*$");
}

ZeeMealMasterImporter::~ZeeMealMasterImporter()
{
}

QString ZeeMealMasterImporter::name() const
{
    return tr("Meal-Master");
}

QString ZeeMealMasterImporter::mimeType() const
{
    return "text/plain";
}

int ZeeMealMasterImporter::totalSteps(QTextStream& textStream) const
{
    int totalSteps = 0;

    while (!textStream.atEnd()) {
        totalSteps++;
        textStream.readLine();
    }

    textStream.device()->reset();

    return totalSteps;
}

void ZeeMealMasterImporter::importRecipeList(QTextStream& textStream)
{
    progress = 0;
    state = Unknown;
    recipeId = -1;
    directionList.clear();

    while (!textStream.atEnd()) {
        progress++;
        line = textStream.readLine();

        switch (state) {
            case Unknown:
                readHeaderLine();
                break;
            case InRecipeTitle:
                readTitleLine();
                break;
            case InRecipeCategory:
                readCategoryLine();
                break;
            case InRecipeYield:
                readYieldLine();
                break;
            case InRecipeIngredient:
                readIngredientLine();
                break;
            case InRecipeDirection:
                readDirectionLine();
                break;
            default:
                readTrailerLine();
                break;
        }
    }
}

void ZeeMealMasterImporter::readHeaderLine()
{
    if ((headerRegExp1.match(line) != -1) || (headerRegExp2.match(line) != -1)) {
        emit progressed(progress);
        qApp->processEvents();

        state = InRecipeTitle;
    }
}

void ZeeMealMasterImporter::readTitleLine()
{
    int index, len;

    if ((index = titleRegExp.match(line, 0, &len)) != -1) {
        QString title = zeeFormatString(line.mid(index + len).stripWhiteSpace());

        QString queryString = "INSERT INTO recipe VALUES(NULL, %1, '', '-1', '')";
        QSqlQuery query(queryString.arg(title));

        queryString = "SELECT last_insert_rowid()";
        query.exec(queryString);

        if (query.first()) {
            recipeId = query.value(0).toInt();
        }

        state = InRecipeCategory;
    } else if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        state = Unknown;
    }
}

void ZeeMealMasterImporter::readCategoryLine()
{
    int index, len;

    if ((index = categoryRegExp.match(line, 0, &len)) != -1) {
        QStringList categoryList = QStringList::split(",", line.mid(index + len));
        QStringList::Iterator it;

        for (it = categoryList.begin(); it != categoryList.end(); ++it) {
            QString categoryName = zeeFormatString((*it).stripWhiteSpace());

            if (categoryName !=  "''") {
                int categoryId = -1;

                QString queryString = "SELECT id FROM category WHERE name = %1";
                QSqlQuery query(queryString.arg(categoryName));

                if (query.first()) {
                    categoryId = query.value(0).toInt();
                } else {
                    queryString = "INSERT INTO category VALUES(NULL, %1)";
                    query.exec(queryString.arg(categoryName));

                    queryString = "SELECT last_insert_rowid()";
                    query.exec(queryString);

                    if (query.first()) {
                        categoryId = query.value(0).toInt();
                    }
                }

                queryString = "SELECT COUNT(*) FROM recipe_category WHERE recipe_id = %1";
                query.exec(queryString.arg(recipeId));

                int count = 0;

                if (query.first()) {
                    count = query.value(0).toInt();
                }

                queryString = "INSERT INTO recipe_category VALUES(NULL, %1, %2, %3)";
                query.exec(queryString.arg(recipeId).arg(count + 1).arg(categoryId));
            }
        }

        state = InRecipeYield;
    } else if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        state = Unknown;
    }
}

void ZeeMealMasterImporter::readYieldLine()
{
    int index, len;

    if (((index = yieldRegExp1.match(line, 0, &len)) != -1) || ((index = yieldRegExp2.match(line, 0, &len)) != -1)) {
        QString yield = line.mid(index + len);

        QString yieldQuantity;
        QString yieldUnit;
        int yieldId = -1;

        if ((index = yieldQuantityRegExp.match(yield, 0, &len)) != -1) {
            yieldQuantity = zeeFormatString(yield.left(index + len).stripWhiteSpace());
            yieldUnit = zeeFormatString(yield.mid(index + len).stripWhiteSpace());
        } else {
            yieldQuantity = "''";
            yieldUnit = zeeFormatString(yield.stripWhiteSpace());;
        }

        if (yieldUnit !=  "''") {
            QString queryString = "SELECT id FROM yield WHERE name = %1";
            QSqlQuery query(queryString.arg(yieldUnit));

            if (query.first()) {
                yieldId = query.value(0).toInt();
            } else {
                queryString = "INSERT INTO yield VALUES(NULL, %1)";
                query.exec(queryString.arg(yieldUnit));

                queryString = "SELECT last_insert_rowid()";
                query.exec(queryString);

                if (query.first()) {
                    yieldId = query.value(0).toInt();
                }
            }
        }

        QString queryString = "UPDATE recipe SET yield_quantity = %1, yield_id = %2 WHERE id = %3";
        QSqlQuery query(queryString.arg(yieldQuantity).arg(yieldId).arg(recipeId));

        state = InRecipeIngredient;
    } else if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        state = Unknown;
    }
}

void ZeeMealMasterImporter::readIngredientLine()
{
    line = zeeStripRightWhiteSpace(line);

    if ((line.length() <= 40) && (ingredientRegExp1.match(line) != -1)) {
        QString quantity = zeeFormatString(line.left(7).stripWhiteSpace());
        QString unitName = zeeFormatString(line.mid(8, 2).stripWhiteSpace());
        QString ingredientName = line.mid(11).stripWhiteSpace();

        addIngredient(quantity, unitName, ingredientName);
    } else if (ingredientRegExp2.match(line) != -1) {
        QString quantity1 = zeeFormatString(line.left(7).stripWhiteSpace());
        QString unitName1 = zeeFormatString(line.mid(8, 2).stripWhiteSpace());
        QString ingredientName1 = line.mid(11, 29).stripWhiteSpace();

        QString quantity2 = zeeFormatString(line.mid(41, 7).stripWhiteSpace());
        QString unitName2 = zeeFormatString(line.mid(49, 2).stripWhiteSpace());
        QString ingredientName2 = line.mid(52).stripWhiteSpace();

        addIngredient(quantity1, unitName1, ingredientName1);
        addIngredient(quantity2, unitName2, ingredientName2);
    } else if ((ingredientGroupRegExp1.match(line) != -1) || (ingredientGroupRegExp2.match(line) != -1)) {
        QString ingredientName = line.replace(QRegExp("^M+"), "");
        ingredientName = ingredientName.replace(QRegExp("^-+"), "");
        ingredientName = ingredientName.replace(QRegExp("-+$"), "");
        ingredientName = zeeFormatString(ingredientName);

        if (ingredientName !=  "''") {
            int ingredientId = -1;

            QString queryString = "SELECT id FROM ingredient WHERE name = %1";
            QSqlQuery query(queryString.arg(ingredientName));

            if (query.first()) {
                ingredientId = query.value(0).toInt();
            } else {
                queryString = "INSERT INTO ingredient VALUES(NULL, %1)";
                query.exec(queryString.arg(ingredientName));

                queryString = "SELECT last_insert_rowid()";
                query.exec(queryString);

                if (query.first()) {
                    ingredientId = query.value(0).toInt();
                }
            }

            queryString = "SELECT COUNT(*) FROM recipe_ingredient WHERE recipe_id = %1";
            query.exec(queryString.arg(recipeId));

            int count = 0;

            if (query.first()) {
                count = query.value(0).toInt();
            }

            queryString = "INSERT INTO recipe_ingredient VALUES(NULL, %1, %2, '', -1, %3, 1)";
            query.exec(queryString.arg(recipeId).arg(count + 1).arg(ingredientId));
        }
    } else if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        state = Unknown;
    } else if (emptyRegExp.match(line) == -1) {
        directionList.append(line.stripWhiteSpace());

        state = InRecipeDirection;
    }
}

void ZeeMealMasterImporter::addIngredient(QString quantity, QString unitName, QString ingredientName)
{
    if ((ingredientName !=  "")) {
        QString queryString = "SELECT recipe_ingredient.id, ingredient_id, name";
        queryString += " FROM recipe_ingredient, ingredient";
        queryString += " WHERE recipe_id = %1 AND ingredient_id = ingredient.id AND is_group = 0";
        queryString += " AND position = (SELECT MAX(position) FROM recipe_ingredient WHERE recipe_id = %2)";
        QSqlQuery query(queryString.arg(recipeId).arg(recipeId));

        if (!query.first() || (quantity !=  "''") || (unitName !=  "''") || !ingredientName.startsWith("-")) {
            ingredientName = zeeFormatString(ingredientName);
            int unitId = -1;
            int ingredientId = -1;

            if (unitName !=  "''") {
                QString queryString = "SELECT id FROM unit WHERE name = %1";
                QSqlQuery query(queryString.arg(unitName));

                if (query.first()) {
                    unitId = query.value(0).toInt();
                } else {
                    queryString = "INSERT INTO unit VALUES(NULL, %1)";
                    query.exec(queryString.arg(unitName));

                    queryString = "SELECT last_insert_rowid()";
                    query.exec(queryString);

                    if (query.first()) {
                        unitId = query.value(0).toInt();
                    }
                }
            }

            QString queryString = "SELECT id FROM ingredient WHERE name = %1";
            QSqlQuery query(queryString.arg(ingredientName));

            if (query.first()) {
                ingredientId = query.value(0).toInt();
            } else {
                queryString = "INSERT INTO ingredient VALUES(NULL, %1)";
                query.exec(queryString.arg(ingredientName));

                queryString = "SELECT last_insert_rowid()";
                query.exec(queryString);

                if (query.first()) {
                    ingredientId = query.value(0).toInt();
                }
            }

            queryString = "SELECT COUNT(*) FROM recipe_ingredient WHERE recipe_id = %1";
            query.exec(queryString.arg(recipeId));

            int count = 0;

            if (query.first()) {
                count = query.value(0).toInt();
            }

            queryString = "INSERT INTO recipe_ingredient VALUES(NULL, %1, %2, %3, %4, %5, 0)";
            query.exec(queryString.arg(recipeId).arg(count + 1).arg(quantity).arg(unitId).arg(ingredientId));
        } else {
            int recipeIngredientId = query.value(0).toInt();
            int ingredientId = query.value(1).toInt();
            ingredientName = zeeFormatString(query.value(2).toString() + ingredientName.replace(0, 1, " "));

            queryString = "SELECT COUNT(*) FROM ingredient WHERE id = %1";
            query.exec(queryString.arg(ingredientId));

            int count = 0;

            if (query.first()) {
                count = query.value(0).toInt();
            }

            if (count > 1) {
                queryString = "SELECT id FROM ingredient WHERE name = %1";
                query.exec(queryString.arg(ingredientName));

                if (query.first()) {
                    ingredientId = query.value(0).toInt();
                } else {
                    queryString = "INSERT INTO ingredient VALUES(NULL, %1)";
                    query.exec(queryString.arg(ingredientName));

                    queryString = "SELECT last_insert_rowid()";
                    query.exec(queryString);

                    if (query.first()) {
                        ingredientId = query.value(0).toInt();
                    }
                }

                queryString = "UPDATE recipe_ingredient SET ingredient_id = %1 WHERE id = %2";
                query.exec(queryString.arg(ingredientId).arg(recipeIngredientId));
            } else {
                queryString = "UPDATE ingredient SET name = %1 WHERE id = %2";
                query.exec(queryString.arg(ingredientName).arg(ingredientId));
            }
        }
    }
}

void ZeeMealMasterImporter::readDirectionLine()
{
    if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        QString directions;
        QStringList::Iterator it;

        for (it = directionList.begin(); it != directionList.end(); ++it) {
            directions += (*it);
        }

        directions = zeeFormatString(directions);

        QString queryString = "UPDATE recipe SET direction = %1 WHERE id = %2";
        QSqlQuery query(queryString.arg(directions).arg(recipeId));

        directionList.clear();
        state = Unknown;
    } else {
        QString direction = line.stripWhiteSpace();

        if ((direction.isEmpty()) && (directionList.last() != "\n")) {
            directionList.append("\n");
        } else {
            if (directionList.last() == "\n") {
                directionList.append("\n" + direction);
            } else {
                directionList.append(" " + direction);
            }
        }
    }
}

void ZeeMealMasterImporter::readTrailerLine()
{
    if ((trailerRegExp1.match(line) != -1) || (trailerRegExp2.match(line) != -1)) {
        state = Unknown;
    }
}
