#include "zeemainwindow.h"

#include "zeebeamdialog.h"
#include "zeeexportdialog.h"
#include "zeehtmlexporter.h"
#include "zeeimportdialog.h"
#include "zeemealmasterexporter.h"
#include "zeemealmasterimporter.h"
#include "zeenewdialog.h"
#include "zeeopendialog.h"
#include "zeeprogressdialog.h"
#include "zeerecipeeditwindow.h"
#include "zeerecipeviewwindow.h"
#include "zeerecipelistview.h"
#include "zeesearchdialog.h"
#include "zeestartupdialog.h"

#include "sql/qsqldatabase.h"
#include "sql/qsqlquery.h"

#include <qpe/applnk.h>
#include <qpe/config.h>
#include <qpe/filemanager.h>
#include <qpe/ir.h>
#include <qpe/qpeapplication.h>
#include <qpe/resource.h>

#include <qfile.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qtoolbutton.h>

#include <sys/stat.h>
#include <unistd.h>

ZeeMainWindow::ZeeMainWindow(QWidget* parent, const char* name)
    : QWidget(parent, name)
{
    m_pDatabase = QSqlDatabase::addDatabase("QSQLITE2");

    setCaption(tr("Zee Cookbook"));
    
    m_pNewCookbookButton = new QToolButton(this);
    m_pNewCookbookButton->setAutoRaise(true);
    m_pNewCookbookButton->setIconSet(Resource::loadIconSet("zeecookbook/new"));
    connect(m_pNewCookbookButton, SIGNAL(clicked()), this, SLOT(newCookbook()));

    m_pOpenCookbookButton = new QToolButton(this);
    m_pOpenCookbookButton->setAutoRaise(true);
    m_pOpenCookbookButton->setIconSet(Resource::loadIconSet("zeecookbook/open"));
    connect(m_pOpenCookbookButton, SIGNAL(clicked()), this, SLOT(openCookbook()));

    QFrame* pButtonSeparator1 = new QFrame(this);
    pButtonSeparator1->setFrameStyle(QFrame::VLine | QFrame::Sunken);
        
    m_pAddRecipeButton = new QToolButton(this);
    m_pAddRecipeButton->setAutoRaise(true);
    m_pAddRecipeButton->setIconSet(Resource::loadIconSet("zeecookbook/add"));
    connect(m_pAddRecipeButton, SIGNAL(clicked()), this, SLOT(addRecipe()));
    
    m_pSearchToolButton = new QToolButton(this);
    m_pSearchToolButton->setAutoRaise(true);
    m_pSearchToolButton->setIconSet(Resource::loadIconSet("zeecookbook/search"));
    connect(m_pSearchToolButton, SIGNAL(clicked()), this, SLOT(searchTool()));

    m_pShowAllToolButton = new QToolButton(this);
    m_pShowAllToolButton->setAutoRaise(true);
    m_pShowAllToolButton->setIconSet(Resource::loadIconSet("zeecookbook/showall"));
    connect(m_pShowAllToolButton, SIGNAL(clicked()), this, SLOT(showAllTool()));
    
    QFrame* pButtonSeparator2 = new QFrame(this);
    pButtonSeparator2->setFrameStyle(QFrame::VLine | QFrame::Sunken);
    
    m_pImportToolButton = new QToolButton(this);
    m_pImportToolButton->setAutoRaise(true);
    m_pImportToolButton->setIconSet(Resource::loadIconSet("zeecookbook/import"));
    connect(m_pImportToolButton, SIGNAL(clicked()), this, SLOT(importTool()));
    
    m_pExportToolButton = new QToolButton(this);
    m_pExportToolButton->setAutoRaise(true);
    m_pExportToolButton->setIconSet(Resource::loadIconSet("zeecookbook/export"));
    connect(m_pExportToolButton, SIGNAL(clicked()), this, SLOT(exportTool()));
        
    m_pRecipeListView = new ZeeRecipeListView(this);
    QPEApplication::setStylusOperation(m_pRecipeListView->viewport(), QPEApplication::RightOnHold);
    connect(m_pRecipeListView, SIGNAL(clicked(QListViewItem*)), this, SLOT(viewRecipe()));
    connect(m_pRecipeListView, SIGNAL(mouseButtonPressed(int, QListViewItem*, const QPoint&, int)), this, SLOT(pressRecipe(int)));
    connect(m_pRecipeListView, SIGNAL(returnPressed(QListViewItem*)), this, SLOT(viewRecipe()));
    
    m_pStatusLabel = new QLabel(this);

    QVBoxLayout* mainLayout = new QVBoxLayout(this, 4, 4);
    QHBoxLayout* buttonLayout = new QHBoxLayout(0);

    buttonLayout->addWidget(m_pNewCookbookButton);
    buttonLayout->addWidget(m_pOpenCookbookButton);
    buttonLayout->addWidget(pButtonSeparator1);
    buttonLayout->addWidget(m_pAddRecipeButton);
    buttonLayout->addStretch();
    buttonLayout->addWidget(m_pSearchToolButton);
    buttonLayout->addWidget(m_pShowAllToolButton);
    buttonLayout->addWidget(pButtonSeparator2);
    buttonLayout->addWidget(m_pImportToolButton);
    buttonLayout->addWidget(m_pExportToolButton);
        
    mainLayout->addLayout(buttonLayout);
    mainLayout->addWidget(m_pRecipeListView);
    mainLayout->addWidget(m_pStatusLabel);

    updateView();

    QTimer::singleShot(0, this, SLOT(startup()));
}

ZeeMainWindow::~ZeeMainWindow()
{
}

void ZeeMainWindow::updateView()
{
    blockSignals(true);

    bool hasDatabase = (m_pDatabase->isOpen());
    bool hasRecipe = false;

    if (hasDatabase) {
        QString queryString = "SELECT COUNT(*) FROM recipe";
        QSqlQuery query(queryString);

        if (query.first()) {
            int count = query.value(0).toInt();
            hasRecipe = (count > 0);
        }
    }
    
    bool hasFilter = (!m_pRecipeListView->filter().isNull());
    int displayCount = m_pRecipeListView->childCount();

    m_pAddRecipeButton->setEnabled(hasDatabase);
    m_pSearchToolButton->setEnabled(hasDatabase && hasRecipe);
    m_pShowAllToolButton->setEnabled(hasDatabase && hasFilter);
    m_pImportToolButton->setEnabled(hasDatabase);
    m_pExportToolButton->setEnabled(hasDatabase && hasRecipe);

    if (hasDatabase) {
        m_pStatusLabel->setText(QString::number(displayCount) + " " + tr("recipe(s)"));
    } else {
        m_pStatusLabel->setText(tr("No cookbook opened"));
    }

    blockSignals(false);
}

void ZeeMainWindow::setDocument(const QString& fileName)
{
    DocLnk docLnk(fileName, true);

    if (m_pDatabase->isOpen()) {
        m_pDatabase->close();
    }

    m_pDatabase->setDatabaseName(docLnk.file());

    if (m_pDatabase->open()) {
        Config config("zeecookbook");
        config.setGroup("General");
        config.writeEntry("LastCookbook", docLnk.file());
    }

    m_pRecipeListView->updateView();
    updateView();
}

void ZeeMainWindow::startup()
{
    ZeeStartupDialog startupDialog(this);

    if (startupDialog.exec() == QDialog::Accepted) {
        switch (startupDialog.selected()) {
            case ZeeStartupDialog::Open:
                openCookbook();
                break;
            case ZeeStartupDialog::Last:
                openLastCookbook();
                break;
            default:
                newCookbook();
                break;
        }
    }
}

void ZeeMainWindow::newCookbook()
{
    ZeeNewDialog newDialog(this);

    if (newDialog.exec() == QDialog::Accepted) {
        QString name = newDialog.selectedName();
        QString location = newDialog.selectedLocation();

        DocLnk newDocLnk;
        newDocLnk.setType("application/zeecookbook");
        newDocLnk.setName(name);

        QString fileName = location +  "/Documents/" + newDocLnk.type() + "/" + name.replace(QRegExp("/"), "_");

        if (QFile::exists(fileName) || QFile::exists(fileName + ".desktop")) {
            int i = 1;
            QString newFileName = fileName + "_" + QString::number(i);

            while (QFile::exists(newFileName) || QFile::exists(newFileName + ".desktop")) {
                i++;
                newFileName = fileName + "_" + QString::number(i);
            }

            fileName = newFileName;
        }

        newDocLnk.setFile(fileName);
        newDocLnk.setLinkFile(fileName + ".desktop");
        newDocLnk.writeLink();
        
        if (m_pDatabase->isOpen()) {
           m_pDatabase->close();
        }

        m_pDatabase->setDatabaseName(newDocLnk.file());

        if (m_pDatabase->open()) {
            QString queryString = "CREATE TABLE recipe (id INTEGER PRIMARY KEY, name, yield_quantity, yield_id, direction)";
            QSqlQuery query(queryString);

            queryString = "CREATE TABLE yield (id INTEGER PRIMARY KEY, name)";
            query.exec(queryString);

            queryString = "CREATE TABLE category (id INTEGER PRIMARY KEY, name)";
            query.exec(queryString);
            
            queryString = "CREATE TABLE unit (id INTEGER PRIMARY KEY, name)";
            query.exec(queryString);
            
            queryString = "CREATE TABLE ingredient (id INTEGER PRIMARY KEY, name)";
            query.exec(queryString);
            
            queryString = "CREATE TABLE recipe_category (id INTEGER PRIMARY KEY, recipe_id, position, category_id)";
            query.exec(queryString);

            queryString = "CREATE TABLE recipe_ingredient (id INTEGER PRIMARY KEY, recipe_id, position, quantity, unit_id, ingredient_id, is_group)";
            query.exec(queryString);

            Config config("zeecookbook");
            config.setGroup("General");
            config.writeEntry("LastCookbook", newDocLnk.file());
        }

        m_pRecipeListView->updateView();
        updateView();
    }
}

void ZeeMainWindow::openLastCookbook()
{
    Config config("zeecookbook");
    config.setGroup("General");
    QString fileName = config.readEntry("LastCookbook");

    if (m_pDatabase->isOpen()) {
        m_pDatabase->close();
    }

    m_pDatabase->setDatabaseName(fileName);
    m_pDatabase->open();

    m_pRecipeListView->updateView();
    updateView();
}

void ZeeMainWindow::openCookbook()
{
    ZeeOpenDialog openDialog(this);
    openDialog.showMaximized();

    if (openDialog.exec() == QDialog::Accepted) {
        DocLnk* docLnk = openDialog.selected();

        if (docLnk) {
            if (m_pDatabase->isOpen()) {
                m_pDatabase->close();
            }

            m_pDatabase->setDatabaseName(docLnk->file());

            if (m_pDatabase->open()) {
                Config config("zeecookbook");
                config.setGroup("General");
                config.writeEntry("LastCookbook", docLnk->file());
            }

            m_pRecipeListView->updateView();
            updateView();
        }
    }
}

void ZeeMainWindow::addRecipe()
{
    QString queryString = "INSERT INTO recipe VALUES(NULL, '', '', '-1', '')";
    QSqlQuery query(queryString);

    queryString = "SELECT last_insert_rowid()";
    query.exec(queryString);

    if (query.first()) {
        int newRecipeId = query.value(0).toInt();

        ZeeRecipeEditWindow recipeEditWindow(newRecipeId, this);

        recipeEditWindow.showMaximized();
        recipeEditWindow.exec();

        m_pRecipeListView->updateView();
        m_pRecipeListView->select(newRecipeId);

        updateView();
    }
}

void ZeeMainWindow::pressRecipe(int button)
{
    if (button == Qt::RightButton) {
        showRecipePopupMenu();
    }
}

void ZeeMainWindow::showRecipePopupMenu()
{
    QPopupMenu* recipePopupMenu = new QPopupMenu(m_pRecipeListView);
    
    recipePopupMenu->insertItem(Resource::loadIconSet("zeecookbook/view"), tr("View"), this, SLOT(viewRecipe()));
    recipePopupMenu->insertSeparator();
    recipePopupMenu->insertItem(Resource::loadIconSet("zeecookbook/edit"), tr("Edit"), this, SLOT(editRecipe()));
    recipePopupMenu->insertItem(Resource::loadIconSet("zeecookbook/remove"), tr("Remove"), this, SLOT(removeRecipe()));
    recipePopupMenu->insertSeparator();
    recipePopupMenu->insertItem(Resource::loadIconSet("zeecookbook/beam"), tr("Beam"), this, SLOT(beamRecipe()));
    
    recipePopupMenu->popup(QCursor::pos());
}

void ZeeMainWindow::viewRecipe()
{
    int recipeId = m_pRecipeListView->selected();

    if (recipeId != -1) {
        ZeeRecipeViewWindow recipeViewWindow(recipeId, this);

        recipeViewWindow.showMaximized();
        recipeViewWindow.exec();

        m_pRecipeListView->updateView();
        updateView();
    }
}

void ZeeMainWindow::editRecipe()
{
    int recipeId = m_pRecipeListView->selected();

    if (recipeId != -1) {
        ZeeRecipeEditWindow recipeEditWindow(recipeId, this);

        recipeEditWindow.showMaximized();
        recipeEditWindow.exec();

        m_pRecipeListView->updateView();
        updateView();
    }
}

void ZeeMainWindow::removeRecipe()
{
    int recipeId = m_pRecipeListView->selected();

    if (recipeId != -1) {
        QMessageBox messageBox(tr("Remove Recipe"), tr("Do you want to remove\nthe selected recipe?"),
                            QMessageBox::Information, QMessageBox::Yes,
                            QMessageBox::No | QMessageBox::Escape | QMessageBox::Default,
                            QMessageBox::NoButton, this);

        messageBox.setIconPixmap(Resource::loadPixmap("zeecookbook/information"));
        messageBox.setButtonText(QMessageBox::Yes, tr("Yes"));
        messageBox.setButtonText(QMessageBox::No, tr("No"));

        if (messageBox.exec() == QMessageBox::Yes) {
            QString queryString = "DELETE FROM recipe WHERE id = %1";
            QSqlQuery query(queryString.arg(recipeId));

            queryString = "DELETE FROM recipe_category WHERE recipe_id = %1";
            query.exec(queryString.arg(recipeId));

            queryString = "DELETE FROM recipe_ingredient WHERE recipe_id = %1";
            query.exec(queryString.arg(recipeId));

            m_pRecipeListView->updateView();
            updateView();
        }
    }
}

static const char* beamFileName = "/tmp/obex/zeecookbook.dat";

void ZeeMainWindow::beamRecipe()
{
    int recipeId = m_pRecipeListView->selected();

    if (recipeId != -1) {
        ZeeBeamDialog beamDialog(this);

        if (beamDialog.exec() == QDialog::Accepted) {
            QString format = beamDialog.selectedFormat();

            ZeeExporter* exporter = 0;

            if (format == tr("HTML")) {
                exporter = new ZeeHtmlExporter();
            } else if (format == tr("Meal-Master")) {
                exporter = new ZeeMealMasterExporter();
            }

            QString name = "";

            QString queryString = "SELECT name FROM recipe WHERE id = %1";
            QSqlQuery query(queryString.arg(recipeId));

            if (query.first()) {
                name = query.value(0).toString();
            }

            QString mimeType = exporter->mimeType();

            unlink(beamFileName);
            mkdir("/tmp/obex/", 0755);

            QFile file(beamFileName);
            
            file.open(IO_WriteOnly);
            QTextStream textStream(&file);

            exporter->exportRecipe(recipeId, textStream);

            file.close();

            delete exporter;

            Ir* ir = new Ir(this);
            connect(ir, SIGNAL(done(Ir*)), this, SLOT(beamRecipeDone(Ir*)));
            ir->send(beamFileName, name, mimeType);
        }
    }
}

void ZeeMainWindow::beamRecipeDone(Ir* ir)
{
    delete ir;
    unlink(beamFileName);
}

void ZeeMainWindow::searchTool()
{
    ZeeSearchDialog searchDialog(this);
    searchDialog.showMaximized();

    if (searchDialog.exec() == QDialog::Accepted) {
        QString filter = searchDialog.selected();
        m_pRecipeListView->setFilter(filter);

        m_pRecipeListView->updateView();
        updateView();
    }
}

void ZeeMainWindow::showAllTool()
{
    m_pRecipeListView->setFilter(QString::null);

    m_pRecipeListView->updateView();
    updateView();
}

void ZeeMainWindow::importTool()
{
    ZeeImportDialog importDialog(this);
    importDialog.showMaximized();

    if (importDialog.exec() == QDialog::Accepted) {
        DocLnk* docLnk = importDialog.selected();

        if (docLnk) {
            ZeeImporter* pImporter = 0;

            ZeeMealMasterImporter mealMasterImporter;

            pImporter = &mealMasterImporter;

            QFile file(docLnk->file());
            file.open(IO_ReadOnly | IO_Translate);
            QTextStream textStream(&file);

            ZeeProgressDialog progressDialog(this);

            progressDialog.setLabelText(tr("Importing recipes..."));
            progressDialog.setProgress(0);
            progressDialog.setTotalSteps(pImporter->totalSteps(textStream));
            connect(pImporter, SIGNAL(progressed(int)), &progressDialog, SLOT(setProgress(int)));
            progressDialog.show();

            pImporter->importRecipeList(textStream);

            file.close();

            m_pRecipeListView->updateView();
            updateView();
        }
    }
}

void ZeeMainWindow::exportTool()
{
    bool isRecipeSelected = (m_pRecipeListView->selected() != -1);
    int count = m_pRecipeListView->childCount();
    bool hasRecipe = (count > 0);

    ZeeExportDialog exportDialog(hasRecipe, isRecipeSelected, this);
    exportDialog.showMaximized();

    if (exportDialog.exec() == QDialog::Accepted) {
        QString name = exportDialog.selectedName();
        QString format = exportDialog.selectedFormat();

        ZeeExporter* pExporter = 0;

        ZeeHtmlExporter htmlExporter;
        ZeeMealMasterExporter mealMasterExporter;

        if (format == htmlExporter.name()) {
            pExporter = &htmlExporter;
        } else if (format == mealMasterExporter.name()) {
            pExporter = &mealMasterExporter;
        }

        DocLnk docLnk;
        docLnk.setName(name);
        docLnk.setType(pExporter->mimeType());

        FileManager fileManager;
        QIODevice* ioDevice = fileManager.saveFile(docLnk);
        QTextStream textStream(ioDevice);

        switch (exportDialog.selected()) {
            case ZeeExportDialog::Current:
                {
                    int recipeId = m_pRecipeListView->selected();

                    if (recipeId != -1) {
                        pExporter->exportRecipe(recipeId, textStream);
                    }
                }
                break;
            case ZeeExportDialog::Display:
                {
                    QValueList<int> recipeList;

                    QString filter = m_pRecipeListView->filter();
                    QString queryString = "SELECT id FROM recipe WHERE %1 ORDER BY name";
                    QSqlQuery query(queryString.arg(filter));

                    while (query.next()) {
                        int recipeId = query.value(0).toInt();
                        recipeList.append(recipeId);
                    }

                    pExporter->exportRecipeList(recipeList, textStream);
                }
                break;
            default:
                {
                    QValueList<int> recipeList;

                    QString queryString = "SELECT id FROM recipe ORDER BY name";
                    QSqlQuery query(queryString);

                    while (query.next()) {
                        int recipeId = query.value(0).toInt();
                        recipeList.append(recipeId);
                    }

                    pExporter->exportRecipeList(recipeList, textStream);
                }
                break;
        }

        delete ioDevice;
    }
}
