/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klImageConverter.cpp,v 1.2 2004/01/13 11:20:09 drgoldie Exp $
 *
\**************************************************************************/


#include "../include/klMain.h"
#include "../include/gl/gl.h"
#include "klImageConverter.h"


void*
klImageConverter::convert(int nWidth, int nHeight, const void* nFromPixels,
						   int nFromNumComp, int nFromFormat, int nFromDataType,
						   int nToNumComp, int nToFormat, int nToDataType)
{
	int size = nWidth*nHeight;
	READFUNC readFunc = getReadFunc(nFromNumComp, nFromFormat, nFromDataType);
	WRITEFUNC writeFunc = getWriteFunc(nToNumComp, nToFormat, nToDataType);

	if(!readFunc || !writeFunc)
		return 0;

	PIXEL pix;
	const unsigned char* fromPix = reinterpret_cast<const unsigned char*>(nFromPixels);
	unsigned char* toPix0 = new unsigned char[size*getNumPixelBytes(nToNumComp, nToDataType)], *toPix=toPix0;

	while(size--)
	{
		fromPix += (*readFunc)(fromPix, pix);
		toPix += (*writeFunc)(toPix, pix);
	}

	return toPix0;
}


int klImageConverter::getNumPixelBytes(int nNumComp, int nDataType)
{
	int size;

	switch(nDataType)
	{
	case GL_UNSIGNED_BYTE:
	case GL_BYTE:
		size = 1;
		break;

	case GL_UNSIGNED_SHORT:
	case GL_SHORT:
		size = 2;
		break;

	case GL_UNSIGNED_INT:
	case GL_INT:
	case GL_FLOAT:
		size = 4;
		break;

	default:
		return 0;
	}

	return nNumComp*size;
}


klImageConverter::READFUNC klImageConverter::getReadFunc(int nNumComp, int nFormat, int nDataType)
{
	switch(nNumComp)
	{
	case 3:
		if(nFormat==GL_RGB && nDataType==GL_UNSIGNED_BYTE)
			return readRGB888;
		break;

	case 4:
		if(nFormat==GL_RGB && nDataType==GL_UNSIGNED_BYTE)
			return readRGBX8888;
		if(nFormat==GL_RGBA && nDataType==GL_UNSIGNED_BYTE)
			return readRGBA8888;
		break;
	}

	return 0;
}


klImageConverter::WRITEFUNC klImageConverter::getWriteFunc(int nNumComp, int nFormat, int nDataType)
{
	switch(nNumComp)
	{
	case 3:
		if(nFormat==GL_RGB && nDataType==GL_UNSIGNED_BYTE)
			return writeRGB888;
		break;

	case 4:
		if(nFormat==GL_RGB && nDataType==GL_UNSIGNED_BYTE)
			return writeRGBX8888;
		if(nFormat==GL_RGBA && nDataType==GL_UNSIGNED_BYTE)
			return writeRGBA8888;
		break;
	}

	return 0;
}


int klImageConverter::readRGB888(const void* nData, PIXEL& nPixel)
{
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(nData);

	nPixel.red = ptr[0];
	nPixel.green = ptr[1];
	nPixel.blue = ptr[2];
	nPixel.alpha = 255;

	return 3;
}


int klImageConverter::readRGBX8888(const void* nData, PIXEL& nPixel)
{
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(nData);

	nPixel.red = ptr[0];
	nPixel.green = ptr[1];
	nPixel.blue = ptr[2];
	nPixel.alpha = 255;

	return 4;
}


int klImageConverter::readRGBA8888(const void* nData, PIXEL& nPixel)
{
	const unsigned char* ptr = reinterpret_cast<const unsigned char*>(nData);

	nPixel.red = ptr[0];
	nPixel.green = ptr[1];
	nPixel.blue = ptr[2];
	nPixel.alpha = ptr[3];

	return 4;
}


int klImageConverter::writeRGB888(void* nData, const PIXEL& nPixel)
{
	unsigned char* ptr = reinterpret_cast<unsigned char*>(nData);

	ptr[0] = nPixel.red;
	ptr[1] = nPixel.green;
	ptr[2] = nPixel.blue;

	return 3;
}


int klImageConverter::writeRGBX8888(void* nData, const PIXEL& nPixel)
{
	unsigned char* ptr = reinterpret_cast<unsigned char*>(nData);

	ptr[0] = nPixel.red;
	ptr[1] = nPixel.green;
	ptr[2] = nPixel.blue;
	ptr[3] = 0;

	return 4;
}


int klImageConverter::writeRGBA8888(void* nData, const PIXEL& nPixel)
{
	unsigned char* ptr = reinterpret_cast<unsigned char*>(nData);

	ptr[0] = nPixel.red;
	ptr[1] = nPixel.green;
	ptr[2] = nPixel.blue;
	ptr[3] = nPixel.alpha;

	return 4;
}
