/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/klContext_Clipping.h,v 1.3 2004/02/10 18:19:30 drgoldie Exp $
 *
\**************************************************************************/


/**
class klContext
{
*/

enum CLIP_BITS {
	CLIP_LEFT = 0,
	CLIP_RIGHT = 10,
	CLIP_BOTTOM = 20,
	CLIP_TOP = 0,
	CLIP_NEAR = 10,
	CLIP_FAR = 20,
	CLIP_MSK = 1023
};

// Clips a primitive at the viewing frustrum
// (<-w,w>, <-w,w>, <-w,w>).
// clipPrimitve uses clipTmpVertices to store intermediate
// data.
// Notice: The number of vertices can increase due to
//         clipping. nSourVertices must be large enough to
//         hold the increased number of vertices.
//
void
clipPrimitve(klVertex* nSourVertices, int& nNumVertices)
{
	int				numTmp1 = nNumVertices, numTmp2;
	klVertex		*tmp1V = nSourVertices, *tmp2V = clipTmpVertices;
	unsigned int	clipLRB, clipTNF, numV=(unsigned int)nNumVertices;
	bool			reversedTmp=false;


	// clipping is a very expensive process (lots of value copying and maths)
	// so we do a very fast classification to see if clipping is necessary at all)
	classifyClipping(nSourVertices, nNumVertices, clipLRB, clipTNF);


	if(!(clipLRB|clipTNF))							// completely inside the frustum?
		return;

	if(((clipLRB>>CLIP_LEFT)&CLIP_MSK)==numV)		// completely left of the frustrum?
	{
		nNumVertices = 0;
		return;
	}
	if(((clipLRB>>CLIP_RIGHT)&CLIP_MSK)==numV)		// completely right of the frustrum?
	{
		nNumVertices = 0;
		return;
	}
	if(((clipLRB>>CLIP_BOTTOM)&CLIP_MSK)==numV)		// completely below the frustrum?
	{
		nNumVertices = 0;
		return;
	}
	if(((clipTNF>>CLIP_TOP)&CLIP_MSK)==numV)		// completely above the frustrum?
	{
		nNumVertices = 0;
		return;
	}
	if(((clipTNF>>CLIP_NEAR)&CLIP_MSK)==numV)		// completely in front of the frustrum?
	{
		nNumVertices = 0;
		return;
	}
	if(((clipTNF>>CLIP_FAR)&CLIP_MSK)==numV)		// completely behind the frustrum?
	{
		nNumVertices = 0;
		return;
	}

	// if the polygon is not completely outside or inside
	// the frustum we need to clip it...

	if((clipLRB>>CLIP_LEFT)&CLIP_MSK)				// need to clip at left plane?
	{
		clipAtLeftPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if((clipLRB>>CLIP_RIGHT)&CLIP_MSK)				// need to clip at right plane?
	{
		if(reversedTmp)
			clipAtRightPlane(tmp1V, numTmp1, tmp2V, numTmp2);
		else
			clipAtRightPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if((clipLRB>>CLIP_BOTTOM)&CLIP_MSK)				// need to clip at bottom plane?
	{
		if(reversedTmp)
			clipAtBottomPlane(tmp1V, numTmp1, tmp2V, numTmp2);
		else
			clipAtBottomPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if((clipTNF>>CLIP_TOP)&CLIP_MSK)				// need to clip at top plane?
	{
		if(reversedTmp)
			clipAtTopPlane(tmp1V, numTmp1, tmp2V, numTmp2);
		else
			clipAtTopPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if((clipTNF>>CLIP_NEAR)&CLIP_MSK)				// need to clip at near plane?
	{
		if(reversedTmp)
			clipAtNearPlane(tmp1V, numTmp1, tmp2V, numTmp2);
		else
			clipAtNearPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if((clipTNF>>CLIP_FAR)&CLIP_MSK)				// need to clip at far plane?
	{
		if(reversedTmp)
			clipAtFarPlane(tmp1V, numTmp1, tmp2V, numTmp2);
		else
			clipAtFarPlane(tmp2V, numTmp2, tmp1V, numTmp1);
		reversedTmp = !reversedTmp;
	}

	if(reversedTmp)
	{
		nNumVertices = numTmp2;
		for(int i=0; i<nNumVertices; i++)
			nSourVertices[i] = tmp2V[i];
	}
	else
		nNumVertices = numTmp1;


	/*clipAtNearPlane(  tmp2V, numTmp2, tmp1V, numTmp1);
	clipAtFarPlane(   tmp1V, numTmp1, tmp2V, numTmp2);
	clipAtLeftPlane(  tmp2V, numTmp2, tmp1V, numTmp1);
	clipAtRightPlane( tmp1V, numTmp1, tmp2V, numTmp2);
	clipAtTopPlane(   tmp2V, numTmp2, tmp1V, numTmp1);
	clipAtBottomPlane(tmp1V, numTmp1, tmp2V, numTmp2);

	nNumVertices = numTmp1;*/

	// Correct all clipped values since they might be
	// a little bit off, due to fixed point arithmentic inprecisions
	//
	/*for(int i=0; i<nNumVertices; i++)
	{
		klVec4& pos = nSourVertices[i].pos;
		klFloat w = pos[3];

		if(pos[0]<-w)
			pos[0] = -w;
		else if(pos[0]>w)
			pos[0] = w;

		if(pos[1]<-w)
			pos[1] = -w;
		else if(pos[1]>w)
			pos[1] = w;

		if(pos[2]<-w)
			pos[2] = -w;
		else if(pos[2]>w)
			pos[2] = w;
	}*/
}


// high speed test whether clipping is necessary at which planes.
// this test uses no conditional code and no complex maths (mul, div, etc.).
//
// the result (how many vertices are on the interior side of the clipping
// planes) is returned in nLRB && nTNR.
//
// note: this function uses 10bits per plane and can therefore clip-test
// polygons with up to 1024 vertices.
//
// how it works:
//   this functions makes use of the fact that a test (x<y) can be rewritten
//   as (x-y<0). testing <0 can be done by looking at the MSB which contains a 1
//   if the integer is smaller than zero.
//   the function sums up all MSBs to calculate how many tests failed.
//
void
classifyClipping(const klVertex* nVertices, int nNumVertices, unsigned int& nLRB, unsigned int& nTNF)
{
	enum {
		MSB = 31
	};

	unsigned int lrb=0, tnf=0;

	for(int i=0; i<nNumVertices; i++)
	{
		int x = nVertices[i].pos[0].getFixed(),
			y = nVertices[i].pos[1].getFixed(),
			z = nVertices[i].pos[2].getFixed(),
			wPos = nVertices[i].pos[3].getFixed(), wNeg = -wPos;

		lrb += (((unsigned int)(x-wNeg))>>MSB)<<CLIP_LEFT;
		lrb += (((unsigned int)(wPos-x))>>MSB)<<CLIP_RIGHT;

		lrb += ((unsigned int)((y-wNeg))>>MSB)<<CLIP_BOTTOM;
		tnf += ((unsigned int)((wPos-y))>>MSB)<<CLIP_TOP;

		tnf += ((unsigned int)((z-wNeg))>>MSB)<<CLIP_NEAR;
		tnf += ((unsigned int)((wPos-z))>>MSB)<<CLIP_FAR;
	}

	nLRB = lrb;
	nTNF = tnf;
}


//////////////////////////////////////////////////////////////////////////////////////
//
//                  Internal Clipping Methods called by clipPrimitve()
//


void
clipAtNearPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int		i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
				*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[2]>-v1->pos[3])					// next vertex is in the frustum
		{
			if(v0->pos[2]>-v0->pos[3])				// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else									// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[2]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[2]+v0->pos[2]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[2]>-v0->pos[3])				// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[2]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[2]+v0->pos[2]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


void
clipAtFarPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int			i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
				*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[2]<v1->pos[3])								// next vertex is in the frustum
		{
			if(v0->pos[2]<v0->pos[3])							// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else									// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[2]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[2]+v0->pos[2]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[2]<v0->pos[3])							// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[2]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[2]+v0->pos[2]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


void
clipAtLeftPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int		i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
			*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[0]>-v1->pos[3])							// next vertex is in the frustum
		{
			if(v0->pos[0]>-v0->pos[3])						// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else									// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[0]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[0]+v0->pos[0]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[0]>-v0->pos[3])						// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[0]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[0]+v0->pos[0]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


void
clipAtRightPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int		i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
			*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[0]<v1->pos[3])								// next vertex is in the frustum
		{
			if(v0->pos[0]<v0->pos[3])							// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else										// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[0]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[0]+v0->pos[0]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[0]<v0->pos[3])							// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[0]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[0]+v0->pos[0]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


void
clipAtTopPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int		i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
			*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[1]<v1->pos[3])								// next vertex is in the frustum
		{
			if(v0->pos[1]<v0->pos[3])							// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else										// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[1]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[1]+v0->pos[1]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[1]<v0->pos[3])							// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[1]-v0->pos[3]) / (v1->pos[3]-v0->pos[3] - v1->pos[1]+v0->pos[1]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


void
clipAtBottomPlane(klVertex* dstVertices, int& numDstVertices, klVertex* srcVertices, int numSrcVertices)
{
	int		i;
	klVertex	*v0 = srcVertices+numSrcVertices-1,
			*v1 = v0;

	numDstVertices = 0;

	for(i=0; i<numSrcVertices; i++)
	{
		v0 = v1;
		v1 = srcVertices+i;

		if(v1->pos[1]>-v1->pos[3])							// next vertex is in the frustum
		{
			if(v0->pos[1]>-v0->pos[3])						// prev vertex is in the frustum too
			{
				dstVertices[numDstVertices++] = *v1;
			}
			else										// prev vertex is NOT in the frustum
			{
				klFloat f = (v0->pos[1]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[1]+v0->pos[1]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
				dstVertices[numDstVertices++] = *v1;
			}
		}
		else
		{
			if(v0->pos[1]>-v0->pos[3])						// prev vertex is in the frustum
			{
				klFloat f = (v0->pos[1]+v0->pos[3]) / (-v1->pos[3]+v0->pos[3] - v1->pos[1]+v0->pos[1]);

				dstVertices[numDstVertices++].interpolateInclColor(*v0,*v1, f);
			}
		}
	}
}


/**
}  // class klContext
*/

