/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565_Scanline.h,v 1.6 2004/02/11 14:16:21 drgoldie Exp $
 *
\**************************************************************************/


//
// class klRSW565 {
//


// modulates two pixel buffers by combining RGB component-wise
// (destroys previous buffer content)
//
void drawScanLine_ModulateRGB(BUF_PIXELTYPE* nSrc0, const BUF_PIXELTYPE* nSrc1)
{
	// calculate number of visible pixels
	int len, rlIdx;
	for(len=0,rlIdx=0; rlIdx<numScanlineRunLengths; rlIdx+=2)
		len += scanlineRunLengths[rlIdx];

	register unsigned int c0,c1,tmp;

	while(len--)
	{
		c0 = *nSrc0;
		c1 = *nSrc1++;

		tmp = (c0&REDBLUE_MASK) * (c1&REDBLUE_MASK);
		*(nSrc0++) =	((tmp>>16)&RED_MASK) | 
						((((c0&GREEN_MASK) * (c1&GREEN_MASK))>>11)&GREEN_MASK) |
						((tmp>>5)&BLUE_MASK);
	}
}


// modulates two pixel buffers by combining RGBA component-wise
// (destroys previous buffer content)
//
void drawScanLine_ModulateRGBA(BUF_PIXELTYPE* nSrc0, const BUF_PIXELTYPE* nSrc1)
{
	// calculate number of visible pixels
	int len, rlIdx;
	for(len=0,rlIdx=0; rlIdx<numScanlineRunLengths; rlIdx+=2)
		len += scanlineRunLengths[rlIdx];

	register unsigned int c0,c1,tmpRB,tmpAG;

	while(len--)
	{
		c0 = *nSrc0;
		c1 = *nSrc1++;

		//c0 |= 0x00ff0000;		// unmark to change color have full alpha
		//c1 |= 0x00ff0000;		// unmark to change color have full alpha

/*
		// this version does full precision modulation using three multiplications
		//
		tmpRB = (c0&REDBLUE_MASK) * (c1&REDBLUE_MASK);
		*(nSrc0++) =	((tmpRB>>16)&RED_MASK) |										// red
						((((c0&GREEN_MASK) * (c1&GREEN_MASK))>>11)&GREEN_MASK) |		// green
						((tmpRB>>5)&BLUE_MASK) |										// blue
						((((c0>>16) * (c1>>16))&0xFF00)<<8);							// alpha
*/

		// this version needs only two multiplications for all four channel
		// (tradeoff is having only 5-bits precision for alpha; 6-bits would be optimal)
		tmpRB =  (c0&REDBLUE_MASK) * (c1&REDBLUE_MASK);
		tmpAG = ((c0>>19) | ((c0&GREEN_MASK)<<5)) * ((c1>>19) | ((c1&GREEN_MASK)<<5));

		*(nSrc0++) =	((tmpRB>>16)&RED_MASK) |										// red
						((tmpAG>>21)&GREEN_MASK) |										// green
						((tmpRB>> 5)&BLUE_MASK) |										// blue
						((tmpAG<<14)&ALPHA_MASK);										// alpha
	}
}


// fills a pixel buffer with gouraud shaded pixels
// (destroys previous buffer content)
//
void drawScanLine_Shaded(const Edge *nLeft, const Edge *nRight, const klFloat& nInvWidth, BUF_PIXELTYPE* nDst)
{
#define RASTER_PIX(offs)	*(nDst++) = iRGB.getRGB565() | 0x00FF0000;	iRGB.step();


	InterpolatorRGB_Linear	iRGB(nLeft, nRight, nInvWidth);

	for(int len,rlIdx=0;;)
	{
		// calculate visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len>16)
		{
			RASTER_SPAN16
			nDst += 16;
			len-=16;
		}

		while(len--)
			{
				*(nDst++) = iRGB.getRGB565() | 0x00FF0000;
				iRGB.step();
			}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels
		//
		len = scanlineRunLengths[rlIdx++];
		if(rlIdx>=numScanlineRunLengths)
			break;

		iRGB.step(len);
	}

#undef RASTER_PIX
}


// fills a pixel buffer with gouraud shaded pixels including alpha values
// (destroys previous buffer content)
//
void drawScanLine_Shaded_Alpha(const Edge *nLeft, const Edge *nRight, const klFloat& nInvWidth, BUF_PIXELTYPE* nDst)
{
#define RASTER_PIX(offs)	*(nDst++) = iRGBA.getARGB8565();  iRGBA.step();

	InterpolatorRGBA_Linear	iRGBA(nLeft, nRight, nInvWidth);

	for(int len,rlIdx=0;;)
	{
		// calculate visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len>16)
		{
			RASTER_SPAN16
			nDst += 16;
			len-=16;
		}

		while(len--)
			{
				*(nDst++) = iRGBA.getARGB8565();
				iRGBA.step();
			}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels
		//
		len = scanlineRunLengths[rlIdx++];
		if(rlIdx>=numScanlineRunLengths)
			break;

		iRGBA.step(len);
	}

#undef RASTER_PIX
}


// fills a pixel buffer with gouraud shaded pixels
// (destroys previous buffer content)
//
void drawScanLine_Shaded_VidMem(const Edge *nLeft, const Edge *nRight, const klFloat& nInvWidth, PIXELTYPE* nDst)
{
#define RASTER_PIX(offs)	nDst[offs] = (unsigned short)iRGB.getRGB565();	iRGB.step();

	InterpolatorRGB_Linear	iRGB(nLeft, nRight, nInvWidth);

	for(int len,rlIdx=0;;)
	{
		// calculate visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		while(len>16)
		{
			RASTER_SPAN16
			nDst += 16;
			len-=16;
		}

		while(len--)
			{
				*(nDst++) = (unsigned short)iRGB.getRGB565();
				iRGB.step();
			}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels
		//
		len = scanlineRunLengths[rlIdx++];
		if(rlIdx>=numScanlineRunLengths)
			break;

		nDst += len;
		iRGB.step(len);
	}

#undef RASTER_PIX
}


// applies fog to a pixel buffer
//
void drawScanLine_Fog(const Edge *nLeft, const Edge *nRight,
					  const klFloat& nInvWidth, BUF_PIXELTYPE* nPixels)
{
	InterpolatorInt_Linear	iFog(nLeft->Fog.getInt(), nRight->Fog.getInt(), nInvWidth);

	for(int len,rlIdx=0;;)
	{
		// calculate visible pixels
		//
		len = scanlineRunLengths[rlIdx++];

		register unsigned int tmp;

		while(len--)
			{
				tmp = *nPixels;
				*nPixels = blendPixel16((unsigned short)tmp, (unsigned short)fogCol, iFog.get()) | (tmp&0xffff0000);
				nPixels++;
				iFog.step();
			}

		if(rlIdx>=numScanlineRunLengths)
			break;

		// skip invisible pixels
		//
		len = scanlineRunLengths[rlIdx++];
		if(rlIdx>=numScanlineRunLengths)
			break;

		iFog.step(len);
	}
}


// does various per-fragment tests and builds an RLE
// list of visible and invisible pixels
//
int drawScanLine_fillRunLengthList(const Edge *nLeft, const Edge *nRight, const klFloat& nPreStep,
								   int nLen, const klFloat& nLenInv, DEPTHTYPE* nSrc)
{
	int numIdx = 0;


	//
	// TODO: implement missing tests such as stencil alpha
	//


	// finally do z-tests
	//
	numIdx = (this->*depthFunc)(nLeft->Z, nRight->Z, nPreStep, nLen, nLenInv, nSrc);


	return numIdx;
}


// main routine for drawing scanlines
//
void drawScanLine(const Edge *nLeft, const Edge *nRight,
				  const TexGradients &nGradients, const MipMapLevel* nTexture)
{
	klFloat			XStart = ceil(nLeft->X);							// first pixel x-coordinate
	klFloat			Width = ceil(nRight->X) - XStart, WidthInv;//=Width;

	if(Width<=0)
		return;

	klFloat			XPrestep = XStart - nLeft->X;						// pre-step for first pixel
	int				offset = curLineOffset + XStart.getInt();			// offset in the buffers
	BUF_PIXELTYPE*	curFinal = NULL;


	// TODO: Scissor Clipping (would be better done during frustum clipping)
	//

	int WidthI = Width.getInt();
	WidthInv = lookupInv[WidthI];
	//WidthInv.inverse();


	// build a run-length table out of z-test, stencil-test, etc...
	// note: this function (optionally) does z-writing too.
	//
	numScanlineRunLengths = drawScanLine_fillRunLengthList(nLeft, nRight, XPrestep, WidthI, WidthInv, zBuffer+offset);


	// check if we need to write any pixels at all...
	//
	if((settings.colorMask&COLOR_MASK_RGB)==0)			// check only RGB since we don't have alpha-dest yet...
		return;


	// for the frequent case of "no fog and no blending" we have special optimized
	// functions that do all in one step and directy write into video memory
	//
	if(scanLineDoSpecialFunc)
	{
		switch(shadeTexMode)
		{
		case MODE_PURE_SHADING:									// pure gouraud shading - no texturing
			drawScanLine_Shaded_VidMem(nLeft, nRight, WidthInv, cBuffer+offset);
			return;

		case MODE_PURE_TEXTURING:								// pure texturing - no gouraud shading
			if(scanLineTexPersp)
				drawScanLine_Texture_VidMem(nLeft,XPrestep, cBuffer+offset, nGradients, nTexture);
			else
				drawScanLine_AffineTexture_VidMem(nLeft, nRight, WidthInv, cBuffer+offset, nTexture);
			return;

		case MODE_MIXED_SHADING_TEXTURING:						// modulated texturing + gouraud shading
			if(scanLineTexPersp)
				drawScanLine_Texture_Shaded_VidMem(nLeft,nRight, XPrestep, WidthInv, cBuffer+offset, nGradients, nTexture);
			else
				drawScanLine_AffineTexture_Shaded_VidMem(nLeft, nRight, WidthInv, cBuffer+offset, nTexture);
			return;
		}
	}


	// 
	// From here on we go for general rendering (no specialized functions)
	// that write into temporary scanline buffers...
	//

	// there are only three cases for shading/texturing:
	//   shading, texturing or shading+texuring
	//
	switch(shadeTexMode)
	{
	case MODE_PURE_SHADING:									// pure gouraud shading - no texturing
		// TODO: optimize for flat shading
		if(settings.blendEnabled)
			drawScanLine_Shaded_Alpha(nLeft, nRight, WidthInv, scanlineGouraud);
		else
			drawScanLine_Shaded(nLeft, nRight, WidthInv, scanlineGouraud);
		curFinal = scanlineGouraud;
		break;

	case MODE_PURE_TEXTURING:								// pure texturing - no gouraud shading
		// both of these methods include alpha channel
		// (which is trivial since we don't have to modulate anything)
		//
		if(scanLineTexPersp)
			drawScanLine_Texture(nLeft, XPrestep, scanlineTexture0, nGradients, nTexture);
		else
			drawScanLine_AffineTexture(nLeft, nRight, WidthInv, scanlineTexture0, nTexture);
		curFinal = scanlineTexture0;
		break;

	case MODE_MIXED_SHADING_TEXTURING:						// modulated texturing + gouraud shading
		if(scanLineTexPersp)
		{
			if(settings.blendEnabled)
				drawScanLine_Texture_Shaded_Alpha(nLeft, nRight, XPrestep, WidthInv, scanlineTexture0, nGradients, nTexture);
			else
				drawScanLine_Texture_Shaded(nLeft, nRight, XPrestep, WidthInv, scanlineTexture0, nGradients, nTexture);
		}
		else
		{
			if(settings.blendEnabled)
				drawScanLine_AffineTexture_Shaded_Alpha(nLeft, nRight, WidthInv, scanlineTexture0, nTexture);
			else
				drawScanLine_AffineTexture_Shaded(nLeft, nRight, WidthInv, scanlineTexture0, nTexture);
		}
		curFinal = scanlineTexture0;
		break;
	}


	// apply fog to the current output
	//
	if(settings.fog==RASTER_FOG_ON)
		drawScanLine_Fog(nLeft, nRight, WidthInv, curFinal);


	// finally write our scanline into video memory
	//
	drawScanLine_PixelWrite(settings.blendSrc, settings.blendDst, curFinal, cBuffer+offset);
}


//
// } class klRSW565
//
