/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565_Public.h,v 1.6 2004/02/11 14:16:21 drgoldie Exp $
 *
\**************************************************************************/


//
// these methods implement the public interface
// implementation of klRasterizer
//

klRSW565::klRSW565()
{
	int i;

	cBuffer = NULL;
	zBuffer = NULL;
	width = height = 0;

	surface = NULL;
	display = NULL;

	clearColor = 0;
	clearDepth = 1.0f;

	cLeft = 0.0f;
	cTop = 0.0f;

	fogCol = 0;

	depthMax = depthMaxHalf = 30000.0f;
	depthMaxHalf *= 0.5f;

	//settings.blendMode = RASTER_BLEND_OFF;
	//settings.color = RASTER_COLOR_GOURAUD;
	settings.fog = RASTER_FOG_OFF;
	settings.polyMode = RASTER_FILL;
	settings.ztest = RASTER_ZTEST_ALWAYS;
	settings.zwrite = RASTER_ZWRITE_OFF;

	clipEnabled = false;

	currentTexture = NULL;
	for(i=0; i<MAX_TEXTURES; i++)
		textures[i] = NULL;


	// define some constants which we need
	// in fixed point format
	constants._100.setInt(100);
	constants._1over100.setFixed(655);		// = 0.01f
	constants._0.setInt(0);
	constants._1.setInt(1);
	constants._255.setInt(255);
	constants._254.setInt(254);
	constants._764.setInt(764);
	constants.white.setInt(255,255,255,255);

	lookupInv[0].setInt(0);
	for(i=1; i<MAX_LOOKUPINV; i++)
	{
		lookupInv[i].setInt(i);
		lookupInv[i].inverse();
	}
}


klRSW565::~klRSW565()
{
	if(!surface && zBuffer)
		delete zBuffer;
}


inline void
klRSW565::setSize(int nWidth, int nHeight)
{
	if(width!=nWidth && height!=nHeight)
	{
		width = nWidth;
		height = nHeight;

		screenCenterX = (width>>1);
		screenCenterY = (height>>1);

		deviceScaleX = screenCenterX + 0.01f;		// make the scale factor a little bit larger, so that we reach
		deviceScaleY = screenCenterY + 0.01f;		// the screen's border even due to fixed point errors

		cRight.setInt(width);		cRight -= cLeft;
		cBottom.setInt(height);		cBottom -= cTop;

		if(surface==NULL)
		{
			if(zBuffer)
				delete zBuffer;
			if(width!=0 && height!=0)
				zBuffer = new DEPTHTYPE[getNumPixels()];
		}
	}
}


inline void
klRSW565::setColorBuffer(void* nMem)
{
	cBuffer = reinterpret_cast<PIXELTYPE*>(nMem);
}


inline void*
klRSW565::getColorBuffer()
{
	return cBuffer;
}


inline void
klRSW565::clear(int nFlags)
{
	int i, numPix = getNumPixels(), numPixH = numPix>>1;

	if(nFlags&GL_COLOR_BUFFER_BIT)
	{
		unsigned int	*cb = reinterpret_cast<unsigned int*>(cBuffer),
						col = clearColor | (clearColor<<16);
		
		for(i=0; i<numPixH; i++)
			cb[i] = col;
	}

	if(nFlags&GL_DEPTH_BUFFER_BIT)
	{
		klFloat d = clearDepth*depthMax;
		for(i=0; i<numPix; i++)
			zBuffer[i] = d;
	}
}


inline void
klRSW565::setClearColor(const klColor4& nColor)
{
	clearColor = calcColor(nColor);
}


inline void
klRSW565::setClearDepth(klFloat nDepth)
{
	clearDepth = nDepth;
}


inline void
klRSW565::set2DClippingEnabled(bool nEnable)
{
	clipEnabled = nEnable;
}

inline void
klRSW565::set2DClipping(int nX0, int nY0, int nX1, int nY1)
{
	clipX0 = nX0;
	clipY0 = nY0;
	clipX1 = nX1;
	clipY1 = nY1;
}


inline void
klRSW565::setFogColor(const klColor4& nColor)
{
	fogCol = calcColor(nColor);
}


inline void
klRSW565::setPixel(int nX, int nY, const klColor3& nColor)
{
	klFloat x,y;

	x.setInt(nX);
	y.setInt(nY);

	if(clipEnabled && (x<clipX0 || x>clipX1 || y<clipY0 || y>clipY1))
		return;

	cBuffer[nX + nY*width] = calcColor(nColor);
}


inline void
klRSW565::setPixel(int nX, int nY, const klColor4& nColor)
{
	klFloat x,y;

	x.setInt(nX);
	y.setInt(nY);

	if(clipEnabled && (x<clipX0 || x>clipX1 || y<clipY0 || y>clipY1))
		return;

	cBuffer[nX + nY*width] = calcColor(nColor);
}


inline void
klRSW565::renderPrimitive(unsigned int nPrimType, const klVertex* nV, int nPrimLen)
{
	int j,j0,j1;

	// everything clipped away ?
	//
	if(nPrimLen==0)
		return;

	// calculate device coordiantes
	//
	for(j=0; j<nPrimLen; j++)
	{
		klFloat	px = screenCenterX + deviceScaleX*nV[j].pos[0],
				py = screenCenterY - deviceScaleY*nV[j].pos[1];

		if(px<cLeft)
			px = cLeft;
		if(px>cRight)
			px = cRight;
		if(py<cTop)
			py = cTop;
		if(py>cBottom)
			py = cBottom;

		vertices[j].pos[0] = px;
		vertices[j].pos[1] = py;
		vertices[j].pos[2] = depthMaxHalf + depthMaxHalf*nV[j].pos[2];		// projected z-value for z-buffer

		vertices[j].col = nV[j].col;
		vertices[j].fog = nV[j].fogDensity;
	}

	// calculate a non projected z-value for texturing
	// that lies in a good numeric range...
	if(settings.numtexstages>0)
	{
		klFloat maxZ;
		maxZ.setInt(0);

		for(j=0; j<nPrimLen; j++)
		{
			if(nV[j].unprojectedZ>maxZ)
				maxZ = nV[j].unprojectedZ;
			if(-nV[j].unprojectedZ>maxZ)
				maxZ = -nV[j].unprojectedZ;

			vertices[j].tex = nV[j].texCoord;
		}

		maxZ.inverse();
		for(j=0; j<nPrimLen; j++)
			vertices[j].texZ = maxZ + nV[j].unprojectedZ * maxZ;		// z-value for texture gradient (non-projected z)
	}

	RASTER_POLYGONMODE mode = settings.polyMode;


	// some drawing types require mode overriding...
	//
	switch(nPrimType)
	{
	case GL_POINTS:
		mode = RASTER_POINTS;
		break;

	case GL_LINES:
	case GL_LINE_STRIP:
	case GL_LINE_LOOP:
		mode = RASTER_LINES;
		break;
	}


	switch(mode)
	{
	case RASTER_POINTS:
		for(j=0; j<nPrimLen; j++)
			setPixel(vertices[j].pos[0].getInt(), vertices[j].pos[1].getInt(), vertices[j].col);
		break;

	case RASTER_LINES:
		// some draw types do not close the loop
		//
		switch(nPrimType)
		{
		case GL_LINE_STRIP:
		case GL_LINES:
			j=0;  j0=0;  j1=1;
			break;

		default:
			j=-1; j0=0;  j1=nPrimLen-1;
			break;
		}

		for(; j<nPrimLen-1; j++,j0=j,j1=j+1)
			drawLine(vertices[j0], vertices[j1]);

		if(nPrimType==GL_TRIANGLE_FAN)
			for(j=0; j<nPrimLen; j++)
				drawLine(vertices[0], vertices[j]);
		break;

	case RASTER_FILL:
		{
			MipMapLevel* mml = NULL;
			if(currentTexture && settings.doTex())
			{
				int mmlIdx = 0;
				if(currentTexture->mmLevels[1]!=NULL)
				{
					mmlIdx = getMipMapLevel(currentTexture, vertices, nPrimLen);
					if(mmlIdx==-1)		// area zero?
						return;
				}

				mml = currentTexture->mmLevels[mmlIdx];
			}

			for(j=1; j<nPrimLen-1; j++)
				drawTriangle(vertices[0], vertices[j], vertices[j+1], mml);
		}
		break;
	}
}


inline int
klRSW565::createTexture()
{
	int i;

	for(i=0; i<MAX_TEXTURES; i++)
		if(textures[i] == NULL)
		{
			textures[i] = new Texture;
			return i;
		}

	return -1;
}


inline void
klRSW565::destroyTexture(int nHandle)
{
	if(nHandle>=0 && nHandle<MAX_TEXTURES && textures[nHandle])
	{
		if(currentTexture==textures[nHandle])
			currentTexture = NULL;

		delete textures[nHandle];
		textures[nHandle] = NULL;
	}
}


inline void
klRSW565::setCurrentTexture(int nHandle)
{
	if(nHandle>=0 && nHandle<MAX_TEXTURES && textures[nHandle])
		currentTexture = textures[nHandle];
}


inline const klMipMapLevel_Info*
klRSW565::getMipMapLevelInfo(int nLevel)
{
	if(currentTexture && nLevel<Texture::MAX_MIPMAPLEVELS)
		return currentTexture->mmLevels[nLevel];

	return NULL;
}


inline void
klRSW565::updateCurrentTexture(int nLevel, int nInternalFormat, unsigned int nWidth, unsigned int nHeight,
							   int nBorder, GLenum nFormat, GLenum nPixelType, const void* nPixels)

{
	if(!currentTexture || nLevel>=Texture::MAX_MIPMAPLEVELS)
		return;

	MipMapLevel* mmLevel = currentTexture->mmLevels[nLevel];
	if(mmLevel)
		delete currentTexture->mmLevels[nLevel];
	mmLevel = currentTexture->mmLevels[nLevel] = new MipMapLevel();


	// update mipmap level information
	//
	mmLevel->internalFormat = nInternalFormat;
	mmLevel->width = nWidth;
	mmLevel->height = nHeight;
	mmLevel->border = nBorder;
	mmLevel->format = nFormat;
	mmLevel->pixelType = nPixelType;

	unsigned int i, size=nWidth*nHeight, cw4=nWidth*4;
	unsigned char	*tmpBuf=new unsigned char[size*4];

	// fill pixel buffer flipped, since our texturing routine wants it this way...
	//
	for(i=0; i<nHeight; i++)
		memcpy(tmpBuf+i*cw4, (const unsigned char*)nPixels+(nHeight-i-1)*cw4, cw4);

	//for(i=0; i<nHeight; i++)
	//	memcpy(tmpBuf+i*cw4, (const unsigned char*)nPixels+i*cw4, cw4);

	if(mmLevel->pixels)
		delete mmLevel->pixels;
	mmLevel->pixels = new TEX_PIXELTYPE[size];
	convertRGBX32toRGB16(mmLevel->pixels, tmpBuf, size);

	/*if(mmLevel->alphas)
		delete mmLevel->alphas;
	mmLevel->alphas = new unsigned char[size];*/
	for(i=0; i<size; i++)
		//mmLevel->alphas[i] = tmpBuf[(i<<2)+3];
		mmLevel->pixels[i] |= (tmpBuf[(i<<2)+3]<<16);


	delete tmpBuf;
}


// this function does a simple - yet very fast - RGBX8888 to RGB565 conversion
inline void
klRSW565::drawPixels(const void* nPixels)
{
	unsigned int num = width*height/2;

	const unsigned int	*srcI = reinterpret_cast<const unsigned int*>(nPixels);
	unsigned int		*dstI = reinterpret_cast<unsigned int*>(cBuffer);
	register int		b,c;

	while(num--)
	{
		b = srcI[0];
		c = srcI[1];

		// do two pixels at once
		*dstI=    ((b>>8)&0xF800) | ((b>>5)&0x07E0) | ((b>>3)&0x1F) |
				((((c>>8)&0xF800) | ((c>>5)&0x07E0) | ((c>>3)&0x1F))<<16);

		srcI += 2;
		dstI += 1;
	}
}


inline bool
klRSW565::setSurface(klEGLSurface* nSurface)
{
	surface = nSurface;

	if(surface==NULL)
	{
		cBuffer = NULL;
		zBuffer = NULL;
		setSize(0,0);
		return true;
	}

	if(surface->getColorType()!=klEGLSurface::COLOR_TYPE_RGB565 ||
	   surface->getDepthType()!=klEGLSurface::DEPTH_TYPE_INT32)
	   return false;

	cBuffer = reinterpret_cast<PIXELTYPE*>(surface->getColorBuffer());
	zBuffer = reinterpret_cast<DEPTHTYPE*>(surface->getDepthBuffer());

	setSize(surface->getWidth(), surface->getHeight());
	return true;
}


inline klEGLSurface*
klRSW565::getSurface()
{
	return surface;
}


inline bool
klRSW565::setDisplay(klEGLDisplay* nDisplay)
{
	display = nDisplay;
	return true;
}


inline klEGLDisplay*
klRSW565::getDisplay()
{
	return display;
}


inline bool
klRSW565::applySettings(klRasterizer_Settings& nSettings)
{
	if(&settings!=&nSettings)
		settings = nSettings;


	// check wether we have a special rasterization function for this case
	//
	scanLineDoSpecialFunc =	settings.numtexstages<=1 &&
							settings.fog==RASTER_FOG_OFF &&
							!settings.blendEnabled;


	// get the function that finally writes pixels into the frame buffer
	//
	pixelWriteFunc = getPixelWriteFunc(settings);


	// get an depth test and write function
	//
	depthFunc = drawScanLine_GetDepthFunc(settings.zwrite, settings.ztest);


	// calculate edge options that are independent of vertex data
	//
	edgeOptionsBase = 0;
	if(settings.ztest!=RASTER_ZTEST_NEVER && (settings.ztest!=RASTER_ZTEST_ALWAYS || !settings.zwrite==RASTER_ZWRITE_OFF))
		edgeOptionsBase |= OPT_DEPTH;
	if(settings.fog==RASTER_FOG_ON)
		edgeOptionsBase |= OPT_FOG;


	settings = nSettings;
	return true;
}


/**
}  // class klRSW565
*/
