/**************************************************************************\
 *
 *  This file is part of the Klimt library.
 *  Copyright (C) 2003 by IMS, Vienna University of Technology.
 *  All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  ("GPL") version 2 as published by the Free Software Foundation.
 *  See the file LICENSE.GPL at the root directory of this source
 *  distribution for additional information about the GNU GPL.
 *  For the full GPL license see
 *  <URL:http://www.gnu.org/copyleft/gpl.html>
 *
 *  For using Klimt with software that can not be combined with the
 *  GNU GPL, and for taking advantage of the additional benefits of
 *  our support services, please contact IMS about acquiring a
 *  Klimt Professional Edition License.
 *
 *  Contact: <mailto:klimt@studierstube.org>
 *  See <URL:http://www.studierstube.org/klimt>
 *  for more information.
 *
 *  Vienna University of Technology
 *  Institute for Software Technology and Interactive Systems
 *  Interactive Media Systems Group
 *  Favoritenstrasse 9-11/188/2
 *  A-1040 Vienna, Austria
 *  <URL:http://www.ims.tuwien.ac.at>.
 *
 **************************************************************************
 *
 * $Header: /cvsroot/klimt/klimt/klimt/src/RasterizerSW565/klRSW565.h,v 1.4 2004/02/11 14:16:21 drgoldie Exp $
 *
\**************************************************************************/


#ifndef __KLRSW565_HEADERFILE__
#define __KLRSW565_HEADERFILE__

#include "../klRasterizer_Settings.h"
#include "../klEGLDisplay.h"
#include "../klEGLSurface.h"

#include "../Base/klMath.h"

class klRSW565
{
public:
	enum
	{
		MAX_PRIMITIVE_LENGTH = 128,
		MAX_TEXTURES = 128,
		MAX_SCANLINE_LENGTH = 1024,
		MAX_SL_RL = 1024,

		RED_MASK    = 0x1F << 11,
		GREEN_MASK  = 0x3F << 5,
		BLUE_MASK   = 0x1F,
		ALPHA_MASK = 0x00FF0000,
		REDBLUE_MASK = RED_MASK | BLUE_MASK,

		MAX_LOOKUPINV = 1024,

		RASTER_ADD_SHIFTMASK = (0x0F<<11) | (0x1F<<5) | 0x0F,
	};

	enum GOURAUD_COLOR
	{
		GOURAUD_WHITE,				// this is the fastest case, we can do pure texturing...
		GOURAUD_FLAT,				// no shading needed, just one color from gouraud
		GOURAUD_SHADED				// general case, if texenv is modulate we'll have to do that...
	};

	enum SHADE_TEX_MODE
	{
		MODE_PURE_SHADING = 0,
		MODE_PURE_TEXTURING = 1,
		MODE_MIXED_SHADING_TEXTURING = 2
	};

	enum EDGE_OPT
	{
		OPT_SHADING_RGB = 1,
		OPT_SHADING_ALPHA = 2,
		OPT_DEPTH = 4,
		OPT_FOG = 8,
		OPT_AFFINE_TEX = 16
	};

	enum EDGE_HINT {
		HINT_FLAT = 1,
		HINT_GOURAUD = 2,
		HINT_TEXTURE = 3,
		HINT_TEXTURE_GOURAUD = 4
	};

	typedef unsigned short	PIXELTYPE;		// pixel value type
	typedef unsigned int	BUF_PIXELTYPE;	// pixel value type in temporary storage (scanline composition)
	typedef unsigned int	TEX_PIXELTYPE;	// pixel value type in textures: [X8-A8-RGB565]
	typedef klFloat			DEPTHTYPE;		// depth value type


	// public interface of klRasterizer
	//
	klRSW565();
	~klRSW565();

	/// sets buffer size
	/**
	 *  This will only work with a software rasterizer
	 *  where the color buffer does not depend on the
	 *  current display/window
	 */
	void setSize(int nWidth, int nHeight);

	/// sets a color buffer to render into
	/**
	 *  This will only work with a software rasterizer
	 *  where the color buffer is accessible
	 */
	void setColorBuffer(void* nMem);

	/// returns the color buffer
	/**
	 *  This will only work with a software rasterizer
	 *  where the color buffer is accessible
	 */
	void* getColorBuffer();

	/// return the width of the buffer
	int getWidth() const  {  return width;  }

	/// return the height of the buffer
	int getHeight() const  {  return height;  }

	/// clears buffer (pass same flags as to glClear)
	void clear(int nFlags);

	/// sets a clear color value
	void setClearColor(const klColor4& nColor);

	/// sets a clear depth value
	void setClearDepth(klFloat nDepth);

	/// enables/disables 2D clipping (normally used for scissor test)
	void set2DClippingEnabled(bool nEnable);

	/// sets the clipping rectangle (0,0 is left top)
	void set2DClipping(int nX0, int nY0, int nX1, int nY1);

	/// sets a fog color value
	void setFogColor(const klColor4& nColor);

	/// sets a single pixel
	void setPixel(int nX, int nY, const klColor3& nColor);

	/// sets a single pixel
	void setPixel(int nX, int nY, const klColor4& nColor);

	/// renders one primitive
	void renderPrimitive(unsigned int nPrimType, const klVertex* nV, int nPrimLen);

	/// returns the current raster settings
	klRasterizer_Settings& getSettings()  {  return settings;  }

	/// applys raster settings to the rasterizer engine
	bool applySettings(klRasterizer_Settings& nSettings);

	/// creates an empty texture, returns texture handle
	int createTexture();

	/// destroys a texture
	void destroyTexture(int nHandle);

	/// make a texture the current textures
	void setCurrentTexture(int nHandle);

	/// returns an info struct about a mipmap level of the current texture (if not exists, NULL is returned)
	const klMipMapLevel_Info* getMipMapLevelInfo(int nLevel);

	/// updates a mipmap level of a precreated texture, glTexImage2D for parameter description
	/**
	 *  notice: nPixels MUST be passed in RGBA 32-bits
	 */
	void updateCurrentTexture(int nLevel, int nInternalFormat, unsigned int nWidth, unsigned int nHeight,
							  int nBorder, GLenum nFormat, GLenum nPixelType, const void* nPixels);

	/// fill the complete color buffer with pixels (nPixels must be RGBX-32
	/**
	 *  notice: this method is only temporary to allow video background filling
	 *          in the future a more complete implementation of glDrawPixels will be available
	 */
	void drawPixels(const void* nPixels);

	/// sets a display which the rasterizer will render into
	bool setSurface(klEGLSurface* nSurface);

	klEGLSurface* getSurface();

	bool setDisplay(klEGLDisplay* nDisplay);

	klEGLDisplay* getDisplay();


protected:
	struct CONSTANTS {
		klFloat _100, _1over100, _0, _1, _255, _254, _764;
		klVec4	white;
	} constants;

	#include "klRSW565_HelperStructs.h"

	int						width,height;
	PIXELTYPE				*cBuffer;
	DEPTHTYPE				*zBuffer;

	klEGLSurface*			surface;
	klEGLDisplay*			display;

	PIXELTYPE				clearColor;
	DEPTHTYPE				clearDepth;

	PIXELTYPE				fogCol;

	bool					clipEnabled;
	klFloat					clipX0, clipY0, clipX1, clipY1;

	DeviceVertex			vertices[MAX_PRIMITIVE_LENGTH];

	klRasterizer_Settings	settings;

	Texture					*textures[MAX_TEXTURES],
							*currentTexture;

	typedef int (klRSW565::Edge::* STEPFUNC)();
	typedef int (klRSW565::* DEPTHFUNC)(const klFloat& nLeftZ, const klFloat& nRightZ, const klFloat& nPreStep, int nLen, const klFloat& nLenInv, DEPTHTYPE* nSrc);
	typedef void (klRSW565::* PIXELWRITEFUNC)(const BUF_PIXELTYPE* nSrc, PIXELTYPE* nDst);

	// some values needed for device conversion
	klFloat					depthMax, depthMaxHalf, screenCenterX, screenCenterY, deviceScaleX, deviceScaleY,
							cLeft,cTop,cRight,cBottom;

	// scanline stuff
	//
	BUF_PIXELTYPE			scanlineGouraud[MAX_SCANLINE_LENGTH],
							scanlineTexture0[MAX_SCANLINE_LENGTH];
	PIXELTYPE				scanLineSrcBlend[MAX_SCANLINE_LENGTH];
	DEPTHTYPE				scanlineDepth[MAX_SCANLINE_LENGTH];

	int						curLineOffset;					// stores the current raster line offset
	unsigned short			scanlineRunLengths[MAX_SL_RL];	// stores lengths of pixels that passed/failed the z-test
	int						numScanlineRunLengths;

	GOURAUD_COLOR			gouraudColor;
	bool					gouraudFullAlpha;

	bool					scanLineDoSpecialFunc;
	bool					scanLineTexPersp;
	int						edgeOptions, edgeOptionsBase;

	SHADE_TEX_MODE			shadeTexMode;

	DEPTHFUNC				depthFunc;
	PIXELWRITEFUNC			pixelWriteFunc;

	klFloat					lookupInv[MAX_LOOKUPINV];

	// include additional code
	//
	#include "klRSW565_Internal.h"
	#include "klRSW565_Line.h"
	#include "klRSW565_Triangle.h"

	#include "klRSW565_InterpolatorInt.h"
	#include "klRSW565_InterpolatorRGB.h"
	#include "klRSW565_InterpolatorRGBA.h"
	#include "klRSW565_InterpolatorZ.h"

	#include "klRSW565_Scanline_PixelWrite.h"
	#include "klRSW565_Scanline_Texture.h"
	#include "klRSW565_Scanline_ZBuffer.h"
	#include "klRSW565_Scanline.h"
};


#include "klRSW565_Public.h"

#endif //__KLRSW565_HEADERFILE__
