#ifndef ZSURFACE_H
#define ZSURFACE_H

   #include <qimage.h>
   #include <qpixmap.h>
   #include <qstring.h>
   #include <qdirectpainter_qws.h>
   #include <qgfx_qws.h>
   #include <stdlib.h>





class ZFont;
//! The ZSurface class is a memory area to which you can draw images and primitives.
/*!
 It  provides set of optimized methods for various types of memory transfers ( blitting) between memory buffers (ZSurfaces) and/or
 framebuffer (video) memory on ARM powered, qte based devices.<Br>
 Each ZSurface object has a memory buffer associated with it and , optionally, a alpha channel buffer which specifies
 the transparency of a pixel. 0 means completely transparent and 255 means opaque.<br>
 A surface has the parameters width(), height() and the actual pixels and alpha data.<br>
 It can be initialized using an existing QImage or QPixmap object.<br>
 Generally there are two kinds of methods available: <br>
  - relatively slow setup-time methods ( constructor, create methods)<br>
  - fast, ASM optimized transfer/primitives methods.<Br>

Every transfer (blit) method as well as primitives drawing methods (lines and text) with exception of setPixel() getPixel() perform
internal clipping/parameter validation before each operation.<Br>
<b>Usage:</b><br>
This class was designed with a single goal - fast, optimized memory transfers between memory surfaces and/or framebuffer memory.
<br>Nothing more, nothing less.<br> This is not a "game programming library" - SDL or Allegro style. There is absolutely no provision
for any sort of event handling or anything else beyond video operations.<br>
You can think of it as an extension to QPainter class.
It can be incorporated into qte based applications, SDL or whatever else suits you best.
The only limitation is that you need to be able to supply a valid QDirectPainter object ( which is used to obtain pointer to a framebuffer memory.)
<br>
If you were to a write a qte application using this class, generally you would write your code like this:<p>
1) You would create your main window ( just like any other qte based application.)<br>
2) At this point ZSurface only supports full-screen application/games and therefore your next step would be to switch your application
into full screen mode ( refer to the example code or <a href="http://www.eongames.com/diary/2002-02-28.html"> this page </a> for more information about
this subject)<Br>
3) Next you would create backbuffer ZSurface which MUST be the same size as your screen (240x320 at this point.)<br>
4) Load/create all other ZSurfaces for your sprites/objects etc ...<br>
5) Create some sort of heartbeat function ( for example using QTimer) and draw all your images/sprites in that method using backbuffer surface as a destination.<br>
6) At the end of that method call bitBlit(QDirectPainter *dst, const ZSurface *src,ZROTATION r) giving as a src your backbuffer surface.<Br>
This will blit content of your backbuffer to the framebuffer memory (display.)
<p>

Of course, as mentioned above you are free to do write any sort of code you want - as long as you can get your hands on QDirectPainter object to use
it in your final blit to the screen memory, you are fine.<P>

<B> TO DO:</B>

- Support for ZROTATION Z90 and Z270 ( landscape modes) on A300.
- Support for ipaq devices.
- Support for windowed mode ( using the content of a window as a opposed to entire screen during memory/framebuffer transfer.)
- Support for RLE encoded sprites.<br>
 
 
*/

class ZSurface
{


public:

   enum ZROTATION
   {
      Z0,
      Z90,
      Z270
   };

/*!
 Constructs an invalid (null) ZSurface object.
*/ 
   ZSurface();
/*! 
Constructs a ZSurface object with w width, h height and alpha=false. 
*/
   ZSurface(int w, int h, bool alpha=false);
/*! 
Constructs a ZSurface out of a valid QImage object.
If alpha is set to true and the img object contains alpha channel data, this 
information will retained by the object (and later used by bitBlitAlpha.)
*/   
   ZSurface(const QImage &img, bool alpha=false);
/*! 
Constructs a ZSurface out of a valid QPixmap object.
The Pixmap is used to create QImage object which then in turn is used to create ZSurface
*/    
   ZSurface(const QPixmap &pix, bool alpha=false);
/*!
Constructs a deep copy of ZSurface object.
*/
   ZSurface(const ZSurface &);        // copy constructor
/*! 
Destroys the object and cleans up
*/
   virtual ~ZSurface();

/*! 
Assigns a deep copy of z to this ZSurface object and returns a reference to it.
*/   
   ZSurface & operator=(const ZSurface &z);

/*!
Initializes an existing instance of ZSurface object from a valid QImage.
If the existing ZSurface object is a valid one and the size of the ZSurface doesn't change (w*h), the already
allocated memory will be reused. If the object is invalid or the new size is different,
a new memory block will be allocated and initialized with content of QImage.<br>
Returns false if memory allocation failed or QImage depth is not 32.
*/ 
   virtual bool create(const QImage &, bool alpha=false);
/*! Same as create(QImage ..)  but using QPixmap as a source. If set to true, the alpha channel
is initialized with 255 (opaque)
*/   
   virtual bool create(const QPixmap &, bool alpha=false);
/*!
Initializes an existing instance of ZSurface object with a given set of parameters.
If the existing ZSurface object is a valid one and the size of the ZSurface doesn't change (w*h), the already
allocated memory will be reused. If the object is invalid or the new size is different,
a new memory block will be allocated and initialized with 0.If set to true,the alpha
channel is initialized with 255(opaque)<br>
Returns false if memory allocation failed.
*/   
   virtual bool create(int width, int height, bool alpha=false);

/*! 
Fills the entire surface with the color value clr (fast method)
*/
   void fill(QColor clr);

/*! 
Draws a line form (x1,y1) to (x2,y2) using the color value clr
*/
   bool  drawLine(int x1, int y1, int x2, int y2, QColor clr);
/*!
Draws a line from (x1,y1) to (x2,y2) blending the color value clr with
the existing pixels usign static alpha value opacity.
*/   
   bool  drawLineBlend(int x1, int y1, int x2, int y2, QColor clr, int opacity);




   static ZROTATION rotation() 
   {
      return zrot;
   }

   static void setRotation(ZROTATION zr )
   {
      zrot=zr;
   }

/*!
Copies a block of pixels from src to a valid QDirectPainter object using rotation r.
The src ZSurface has to have the same size as the destination QDirectPainter object.
This method should be used if you want to transfer content of the backbuffer ZSurface
 ( or any surface that has the same dimensions as the screen ) to the framebuffer (video)
 memory. <br>
 Returns false if the size of the src surface does not match QDirectPainter size.<br>
 (fast ASM method)
*/
   static bool bitBlit(QDirectPainter *dst, const ZSurface *src,ZROTATION r);

/*! 
Copies a block of pixels from src to dst surface. The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/
   static bool bitBlit(ZSurface *dst, int dx, int dy, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);

/*! 
Copies a block of pixels from src to dst surface, blending each pixel using static
alpha value opacity (0-32).<br> The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)
<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/
   
   static bool bitBlitBlend(int opacity, ZSurface *dst, int dx, int dy, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);
/*!
Copies a block of pixels from src to dst surface blending each pixel
using alpha channel information from src. If the src surface does not contain alpha
channel this call will have the same effect as bitBlit.<Br>
The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/
   static bool bitBlitAlpha(ZSurface *dst, int dx, int dy, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);
/*!
Copies a block of pixels from src to dst. For any src pixel matching QColor value key,
the coresponding dst pixel will remain unchanged.<br>
The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/
   static bool bitBlitKeyed(ZSurface *dst, int dx, int dy,const QColor &key, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);

/*!
Copies a block of pixels from src to dst blending each pixel using static
alpha value opacity (0-32). For any src pixel matching QColor value key,
the coresponding dst pixel will remain unchanged.<br>
The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/   
   static bool bitBlitKeyedBlend(int opacity, ZSurface *dst, int dx, int dy,const QColor &key, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);
   
/*!
Copies a block of pixels from src to dst using value of QColor fillColor to replace destination pixel.
For any src pixel matching QColor value key,the coresponding dst pixel will remain unchanged.<br>
The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/

   static bool fillBlitKeyed(ZSurface *dst, int dx, int dy,const QColor &key,const QColor &fillColor, const ZSurface *src, int sx=0, int sy=0, int sw=-1, int sh=-1);

/*!
Copies a block of pixels from src to dst blending QColor value fillColor with a dst pixel using static
alpha value opacity (0-32).
For any src pixel matching QColor value key,the coresponding dst pixel will remain unchanged.<br>
The sx,sy is the top left pixel
in src (0,0) by default, dx,dy is the top left pixel in dst and sw,sh is the size of src
(all of src by default.)<br>
Returns false if any of the surfaces is invalid or the src rectangle does not
intersect with the dst rectangle.
<br>(fast ASM method)
*/


   static bool fillBlitKeyedBlend(int opacity,ZSurface *dst, int dx, int dy,const QColor &key,const QColor &fillColor, const ZSurface *src,
                              int sx=0, int sy=0, int sw=-1, int sh=-1);

/*!
Fills the rectangle (dx,dy,rw,rh) with the QColor value clr.<br>
Returns false if the src surface is invalid or the specified rectangle does not intersect witht the surface rectangle.
<br>(fast ASM method)
*/

   static bool fillRect(ZSurface *dst, int dx, int dy,int rw, int rh,const QColor &clr);

/*!
Fills the rectangle (dx,dy,rw,rh) blending QColor value fillColor with the src pixels using static alpha value opacity (0-32).<br>
Returns false if the src surface is invalid or the specified rectangle does not intersect witht the surface rectangle.
<br>(fast ASM method)
*/

   static bool fillRectBlend(int opacity,ZSurface *dst, int dx, int dy,int rw, int rh,const QColor &clr);

/*!
Draws text on the surface using ZFont f, optionally blending with the surface pixels using static alpha opacity (0-32 , default 32 means no blending.)
<br>Returns width of the drawn text in pixels.
<br>(fast ASM method)
*/
   int ZSurface::drawText(int dx, int ny,const QString &s, ZFont *f, int opacity=32);

/*!
Draws text on the surface using ZFont f, with a color value c, optionally blending with the surface pixels using static alpha opacity (0-32 , default 32 means no blending.)
<Br>Returns width of the drawn text in pixels.
<br>(fast ASM method)
*/
   int ZSurface::drawText(int dx, int ny,const QString &s, ZFont *f,QColor c, int opacity=32);
   
/*!
Returns true if this ZSurface object is valid ( has been properly initialized and has a valid memory block associated with it.)
*/
   inline bool isValid() const
   {
      return (bool) data;
   }

/*!
Returns true if this ZSurface object has an alpha buffer associated with it.
*/

   inline bool hasAlpha() const
   {
      return (bool) adata;
   }

/*!
Returns the width of the surface in pixels. 
*/
   inline int width() const
   {
      return iW;
   }

/*!
Returns the total size of the surface memory block in bytes ( scanlineWidth()*height()*(depth()/8) ). 
*/

   inline int size() const
   {
      return iTW*iH*iD;
   }

/*!
Returns the spacing in pixels from one line to the next.<br>
The scan lines are always dword aligned ( 2 pixels) so this value will return width()+1 if during the initialization of the surface the requested width for the surface
was not even.
*/

   inline int scanlineWidth() const
   {
      return iTW;
   }

/*!
Returns the height of the surface in pixels. 
*/

   inline int height() const
   {
      return iH;
   }

/*!
Returns depth of the surface (currently always 16). 
*/
   inline int depth() const
   {
      return iD*8;
   }


/*!
Draws/Plots a single point at (x,y) using QColor value c. 
*/

   inline void setPixel(int x, int y, const QColor &c)
   {
      if ( (x>=0 && x <iW) && (y>=0 && y<iH) )
      {
         *((ushort*)data+(y*iTW)+x)=qt_convRgbTo16(c.rgb());
      }
   }

/*!
Retrieves a color value of a single point at (x,y) into QColor reference c. 
*/

   inline void  getPixel(int x, int y, QColor &c)
   {
      if ( (x>=0 && x <iW) && (y>=0 && y<iH) )
      {
         c.setRgb(qt_conv16ToRgb((unsigned short)*((ushort*)data+(y*iTW)+x)));
      }
   }
/*!
 Returns a pointer to the pixel data.
*/
   inline unsigned short* bits() 
   {
      return (unsigned short*)data;
   }

/*!
 Returns a pointer to the alpha channel data.
*/

   inline unsigned short* alpha()
   {
      return (unsigned short*)adata;
   }


protected:

   void initFromImage(const QImage &i);


private:

   typedef struct BLIT_PARAMETERS
   {
      unsigned char   *dst;       // dst starting mem address
      unsigned char   *src;       // src starting mem address
      unsigned int    sw;         // source width in pixels
      unsigned int    sh;         // source height in pixels
      unsigned int    ss;         // source line step ( in bytes)
      unsigned int    ds;         // dest line step ( in bytes)
      unsigned int    key;        // transparent color
      unsigned int    color;      // color used for filling surfaces 
      unsigned int    alpha;
      unsigned char   *asrc;
      unsigned int    depth;    // if blit is aligned
   } BLIT_PARS;

   enum
   {
      TOP = 0x1, BOTTOM = 0x2, RIGHT = 0x4, LEFT = 0x8
   };

   bool clipLine (int &x0, int &y0, int &x1, int &y1, QRect clip);
   unsigned int  ZSurface::getCode (int x, int y, QRect r);

   static bool ZSurface::computeBlitPars( BLIT_PARS *bpar,ZSurface *dst, int dx, int dy, const ZSurface *src, int sx, int sy, int sw, int sh);
   static bool ZSurface::computeBlitPars( BLIT_PARS *bpar,ZSurface *dst, int dx, int dy,int sw, int sh);

   void initSurface();
   void copySurface(const ZSurface&);

   int     iW;
   int     iTW;                // true width with offset;
   int     iH;
   int     iD;
      
   unsigned char *data;        // actual data
   unsigned char *adata;

   static  ZROTATION zrot;
};                    

#endif // INPUT_DIALOG_H

